<?php

namespace App\Http\Controllers;

use App\Models\SalesPage;
use App\Models\SalesPageAnalytics;
use App\Models\SalesPageOrder;
use App\Models\Order;
use App\Models\SystemSetting;
use App\Helpers\PhoneHelper;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Jenssegers\Agent\Agent;

class SalesPagePublicController extends Controller
{
    /**
     * Display seller landing page (all sales pages)
     */
    public function showSellerLanding($seller)
    {
        // Normalize to lowercase for case-insensitive lookup
        $sellerSlug = strtolower($seller);

        // Try to find user by url_slug (new format)
        $user = \App\Models\User::where('url_slug', $sellerSlug)->first();

        // Fallback: try by name for legacy URLs
        if (!$user) {
            $user = \App\Models\User::where('name', $seller)->first();
        }

        if (!$user) {
            abort(404, 'Seller not found');
        }

        // Get all active sales pages for this seller
        $salesPages = SalesPage::where('user_id', $user->id)
            ->where('is_active', true)
            ->orderBy('created_at', 'desc')
            ->get();

        return view('sales-pages.public.seller-landing', compact('user', 'salesPages'));
    }

    /**
     * Display the sales page
     */
    public function show($seller, $slug)
    {
        $salesPage = SalesPage::where('slug', $slug)
            ->where('is_active', true)
            ->firstOrFail();

        // Get products for this sales page
        $products = $salesPage->getAvailableProductsModels();

        if ($products->isEmpty()) {
            abort(404, 'No products available on this sales page');
        }

        // Get fee settings for this seller
        $seller = $salesPage->user;
        $service = new \App\Services\BayarCashService($seller);

        // Get FPX fee settings
        $fpxFeeSettings = $service->getFeeSettings('fpx', $seller);

        // Get DuitNow fee settings
        $duitnowFeeSettings = $service->getFeeSettings('duitnow', $seller);

        return view('sales-pages.public.show', compact('salesPage', 'products', 'fpxFeeSettings', 'duitnowFeeSettings'));
    }

    /**
     * Track page view
     */
    public function trackView(Request $request, $slug)
    {
        $salesPage = SalesPage::where('slug', $slug)
            ->where('is_active', true)
            ->firstOrFail();

        // Generate or get session ID
        $sessionId = $request->session()->getId();

        // Check if this session already viewed this page recently (within 30 minutes)
        $recentView = SalesPageAnalytics::where('sales_page_id', $salesPage->id)
            ->where('session_id', $sessionId)
            ->where('event_type', 'view')
            ->where('created_at', '>=', now()->subMinutes(30))
            ->first();

        if (!$recentView) {
            // Get device info
            $agent = new Agent();
            $agent->setUserAgent($request->userAgent());

            $deviceType = 'desktop';
            if ($agent->isMobile()) {
                $deviceType = 'mobile';
            } elseif ($agent->isTablet()) {
                $deviceType = 'tablet';
            }

            // Create analytics record
            SalesPageAnalytics::create([
                'sales_page_id' => $salesPage->id,
                'event_type' => 'view',
                'session_id' => $sessionId,
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent(),
                'referrer' => $request->header('referer'),
                'utm_source' => $request->get('utm_source'),
                'utm_medium' => $request->get('utm_medium'),
                'utm_campaign' => $request->get('utm_campaign'),
                'utm_content' => $request->get('utm_content'),
                'device_type' => $deviceType,
                'browser' => $agent->browser(),
                'os' => $agent->platform(),
            ]);

            // Increment view count
            $salesPage->incrementViews();
        }

        return response()->json(['success' => true]);
    }

    /**
     * Submit order from sales page
     */
    public function submitOrder(Request $request, $slug)
    {
        $salesPage = SalesPage::where('slug', $slug)
            ->where('is_active', true)
            ->firstOrFail();

        $validated = $request->validate([
            'customer_name' => 'required|string|max:255',
            'customer_phone' => 'required|string|max:50',
            'customer_email' => 'nullable|email|max:255',
            'customer_address' => 'required|string',
            'customer_city' => 'required|string|max:100',
            'customer_state' => 'required|string|max:100',
            'customer_postcode' => 'required|string|max:20',
            'items' => 'required|array|min:1',
            'items.*.product_id' => 'required|integer',
            'items.*.quantity' => 'required|integer|min:1|max:100',
            'items.*.price' => 'required|numeric|min:0',
            'items.*.name' => 'required|string',
            'payment_method' => 'required|in:fpx,duitnow,cod',
        ]);

        // Calculate total and prepare items
        $items = [];
        $totalAmount = 0;

        foreach ($validated['items'] as $item) {
            $subtotal = $item['price'] * $item['quantity'];

            // Fetch product to get SKU and bundle information
            $product = \App\Models\Product::find($item['product_id']);

            $lineItem = [
                'product_id' => $item['product_id'],
                'name' => $item['name'],
                'sku' => $product ? $product->sku : '',
                'quantity' => $item['quantity'],
                'price' => $item['price'],
                'total' => $subtotal,
            ];

            // If product is a bundle, include bundle items with SKUs
            if ($product && $product->isBundle() && $product->bundle_items) {
                $bundleItems = [];
                foreach ($product->bundle_items as $bundleItem) {
                    if (isset($bundleItem['product_id'])) {
                        $bundleProduct = \App\Models\Product::find($bundleItem['product_id']);
                        $bundleItems[] = [
                            'product_id' => $bundleItem['product_id'],
                            'sku' => $bundleProduct ? $bundleProduct->sku : ($bundleItem['sku'] ?? ''),
                            'name' => $bundleProduct ? $bundleProduct->name : ($bundleItem['name'] ?? ''),
                            'quantity' => $bundleItem['quantity'] ?? 1,
                        ];
                    } else {
                        // Legacy format with just SKU and name
                        $bundleItems[] = [
                            'sku' => $bundleItem['sku'] ?? '',
                            'name' => $bundleItem['name'] ?? '',
                            'quantity' => $bundleItem['quantity'] ?? 1,
                        ];
                    }
                }
                $lineItem['bundle_items'] = $bundleItems;
            }

            $items[] = $lineItem;
            $totalAmount += $subtotal;
        }

        // Format phone number
        $phone = PhoneHelper::formatMalaysian($validated['customer_phone']);

        // Create or update customer record for auto-fill feature
        \App\Models\Customer::updateOrCreate(
            ['phone' => $phone],
            [
                'first_name' => $validated['customer_name'],
                'last_name' => '',
                'email' => $validated['customer_email'] ?? null,
                'address_1' => $validated['customer_address'],
                'city' => $validated['customer_city'],
                'state' => $validated['customer_state'],
                'postcode' => $validated['customer_postcode'],
                'country' => 'Malaysia',
                'username' => $phone, // Use phone as username
                'password' => bcrypt(\Illuminate\Support\Str::random(16)), // Random password
            ]
        );

        // Get session ID for analytics
        $sessionId = $request->session()->getId();

        // Create sales page order
        $salesPageOrder = SalesPageOrder::create([
            'sales_page_id' => $salesPage->id,
            'customer_name' => $validated['customer_name'],
            'customer_email' => $validated['customer_email'],
            'customer_phone' => $phone,
            'customer_address' => $validated['customer_address'],
            'customer_city' => $validated['customer_city'],
            'customer_state' => $validated['customer_state'],
            'customer_postcode' => $validated['customer_postcode'],
            'items' => $items,
            'total_amount' => $totalAmount,
            'session_id' => $sessionId,
            'utm_data' => [
                'utm_source' => $request->input('utm_source'),
                'utm_medium' => $request->input('utm_medium'),
                'utm_campaign' => $request->input('utm_campaign'),
                'utm_content' => $request->input('utm_content'),
            ],
            'status' => 'pending',
        ]);

        // Track analytics
        $agent = new Agent();
        $agent->setUserAgent($request->userAgent());

        $deviceType = 'desktop';
        if ($agent->isMobile()) {
            $deviceType = 'mobile';
        } elseif ($agent->isTablet()) {
            $deviceType = 'tablet';
        }

        SalesPageAnalytics::create([
            'sales_page_id' => $salesPage->id,
            'event_type' => 'order',
            'session_id' => $sessionId,
            'ip_address' => $request->ip(),
            'user_agent' => $request->userAgent(),
            'utm_source' => $request->input('utm_source'),
            'utm_medium' => $request->input('utm_medium'),
            'utm_campaign' => $request->input('utm_campaign'),
            'utm_content' => $request->input('utm_content'),
            'device_type' => $deviceType,
            'browser' => $agent->browser(),
            'os' => $agent->platform(),
        ]);

        // Increment order count
        $salesPage->incrementOrders();

        // Process payment based on selected method
        $paymentController = new \App\Http\Controllers\SalesPagePaymentController();

        // Update payment method
        $paymentMethod = $validated['payment_method'];
        $codFee = 0;
        $paymentGatewayFee = 0;

        // Calculate fees based on payment method
        if ($paymentMethod === 'cod') {
            $codFee = \App\Services\BayarCashService::calculateCodFee($validated['customer_state']);
        } elseif (in_array($paymentMethod, ['fpx', 'duitnow'])) {
            // Calculate payment gateway fee for online payments
            $seller = $salesPage->user;
            $service = new \App\Services\BayarCashService($seller);
            $feeCalculation = $service->calculatePaymentGatewayFee($totalAmount, $paymentMethod, $seller);

            // Only add fee if customer bears it
            if ($feeCalculation['details']['bearer'] === 'customer') {
                $paymentGatewayFee = $feeCalculation['fee'];
            }
        }

        $grandTotal = $totalAmount + $codFee + $paymentGatewayFee;

        $salesPageOrder->update([
            'payment_method' => $paymentMethod,
            'payment_status' => 'pending',
            'cod_fee' => $codFee,
            'payment_gateway_fee' => $paymentGatewayFee,
            'grand_total' => $grandTotal,
        ]);

        // Handle payment directly
        try {
            switch ($paymentMethod) {
                case 'fpx':
                    $result = $paymentController->handleFpxPayment($salesPageOrder);
                    if ($result instanceof \Illuminate\Http\RedirectResponse) {
                        return response()->json([
                            'success' => true,
                            'redirect_url' => $result->getTargetUrl(),
                        ]);
                    }
                    break;

                case 'duitnow':
                    $result = $paymentController->handleDuitNowPayment($salesPageOrder);
                    if ($result instanceof \Illuminate\Http\RedirectResponse) {
                        return response()->json([
                            'success' => true,
                            'redirect_url' => $result->getTargetUrl(),
                        ]);
                    }
                    break;

                case 'cod':
                    $result = $paymentController->handleCodPayment($salesPageOrder);
                    if ($result instanceof \Illuminate\Http\RedirectResponse) {
                        // Extract order ID from URL
                        $url = parse_url($result->getTargetUrl());
                        parse_str($url['query'] ?? '', $params);
                        return response()->json([
                            'success' => true,
                            'order_id' => $params['order'] ?? null,
                        ]);
                    }
                    break;
            }
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Payment processing error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Payment processing failed: ' . $e->getMessage(),
            ], 500);
        }

        return response()->json([
            'success' => false,
            'message' => 'Unknown error occurred',
        ], 500);
    }

    /**
     * Thank you page
     */
    public function thankYou(Request $request)
    {
        $orderId = $request->get('order');

        $order = null;
        $salesPage = null;

        if ($orderId) {
            $order = Order::find($orderId);

            // Get sales page from order
            if ($order) {
                $salesPageOrder = SalesPageOrder::where('order_id', $order->id)->first();
                if ($salesPageOrder) {
                    $salesPage = $salesPageOrder->salesPage;
                }
            }
        }

        return view('sales-pages.public.thank-you', compact('order', 'salesPage'));
    }

    /**
     * Generate global order ID
     */
    protected function generateGlobalOrderId()
    {
        // Get the prefix from system settings (default: OMS)
        $prefix = SystemSetting::getGlobalIdPrefix();
        $lastOrder = Order::orderBy('id', 'desc')->first();
        $nextNumber = $lastOrder ? $lastOrder->id + 1 : 1;

        if ($nextNumber < 10000000) {
            return $prefix . str_pad($nextNumber, 8, '0', STR_PAD_LEFT);
        } else {
            return $prefix . $nextNumber;
        }
    }

    /**
     * Display default sales page for seller domain (root domain)
     */
    public function showBySellerDomain($sellerDomain)
    {
        // Check if seller has set a default sales page
        if (!$sellerDomain->default_sales_page_id) {
            abort(404, 'No default sales page configured for this domain.');
        }

        $salesPage = SalesPage::where('id', $sellerDomain->default_sales_page_id)
            ->where('user_id', $sellerDomain->user_id)
            ->where('is_active', true)
            ->firstOrFail();

        $products = $salesPage->getAvailableProductsModels();

        if ($products->isEmpty()) {
            abort(404, 'No products available on this sales page');
        }

        // Get fee settings for this seller
        $seller = $salesPage->user;
        $service = new \App\Services\BayarCashService($seller);

        // Get FPX fee settings
        $fpxFeeSettings = $service->getFeeSettings('fpx', $seller);

        // Get DuitNow fee settings
        $duitnowFeeSettings = $service->getFeeSettings('duitnow', $seller);

        return view('sales-pages.public.show', compact('salesPage', 'products', 'fpxFeeSettings', 'duitnowFeeSettings'));
    }

    /**
     * Display sales page by slug for seller domain
     */
    public function showBySellerDomainSlug($sellerDomain, $slug)
    {
        // Find sales page by slug that belongs to this domain's owner
        $salesPage = SalesPage::where('slug', $slug)
            ->where('user_id', $sellerDomain->user_id)
            ->where('is_active', true)
            ->firstOrFail();

        $products = $salesPage->getAvailableProductsModels();

        if ($products->isEmpty()) {
            abort(404, 'No products available on this sales page');
        }

        // Get fee settings for this seller
        $seller = $salesPage->user;
        $service = new \App\Services\BayarCashService($seller);

        // Get FPX fee settings
        $fpxFeeSettings = $service->getFeeSettings('fpx', $seller);

        // Get DuitNow fee settings
        $duitnowFeeSettings = $service->getFeeSettings('duitnow', $seller);

        return view('sales-pages.public.show', compact('salesPage', 'products', 'fpxFeeSettings', 'duitnowFeeSettings'));
    }
}
