<?php

namespace App\Http\Controllers;

use App\Models\SalesPageOrder;
use App\Models\SalesPagePayment;
use App\Models\Order;
use App\Models\SystemSetting;
use App\Services\BayarCashService;
use App\Services\OrderNumberService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

class SalesPagePaymentController extends Controller
{
    /**
     * Show payment method selection page
     */
    public function showPaymentSelection(SalesPageOrder $salesPageOrder)
    {
        $salesPage = $salesPageOrder->salesPage;

        if (!$salesPage->is_active) {
            abort(404);
        }

        // Get seller for fee calculations
        $seller = $salesPage->user;
        $service = new BayarCashService($seller);

        // Calculate COD fee
        $codFee = BayarCashService::calculateCodFee($salesPageOrder->customer_state);

        // Calculate FPX (Online Banking) fee
        $fpxFeeCalc = $service->calculatePaymentGatewayFee($salesPageOrder->total_amount, 'fpx', $seller);
        $fpxFee = $fpxFeeCalc['details']['bearer'] === 'customer' ? $fpxFeeCalc['fee'] : 0;

        // Calculate DuitNow QR fee
        $duitnowFeeCalc = $service->calculatePaymentGatewayFee($salesPageOrder->total_amount, 'duitnow', $seller);
        $duitnowFee = $duitnowFeeCalc['details']['bearer'] === 'customer' ? $duitnowFeeCalc['fee'] : 0;

        return view('sales-pages.public.payment-select', compact(
            'salesPageOrder',
            'salesPage',
            'codFee',
            'fpxFee',
            'duitnowFee'
        ));
    }

    /**
     * Process payment method selection
     */
    public function processPayment(Request $request, SalesPageOrder $salesPageOrder)
    {
        $validated = $request->validate([
            'payment_method' => 'required|in:fpx,duitnow,cod',
        ]);

        $paymentMethod = $validated['payment_method'];

        // Get the seller for payment gateway settings
        $seller = $salesPageOrder->salesPage->user;

        // Calculate COD fee if applicable
        $codFee = 0;
        if ($paymentMethod === 'cod') {
            $codFee = BayarCashService::calculateCodFee($salesPageOrder->customer_state);
        }

        // Calculate payment gateway fee for online payments (FPX/DuitNow)
        $gatewayFee = 0;
        if (in_array($paymentMethod, ['fpx', 'duitnow'])) {
            $service = new BayarCashService($seller);
            $feeCalculation = $service->calculatePaymentGatewayFee($salesPageOrder->total_amount, $paymentMethod, $seller);

            // Only add fee if customer bears it
            if ($feeCalculation['details']['bearer'] === 'customer') {
                $gatewayFee = $feeCalculation['fee'];
            }
        }

        $grandTotal = $salesPageOrder->total_amount + $codFee + $gatewayFee;

        // Update sales page order with payment details
        $salesPageOrder->update([
            'payment_method' => $paymentMethod,
            'payment_status' => 'pending',
            'cod_fee' => $codFee,
            'payment_gateway_fee' => $gatewayFee,
            'grand_total' => $grandTotal,
        ]);

        // Handle based on payment method
        switch ($paymentMethod) {
            case 'fpx':
                return $this->handleFpxPayment($salesPageOrder);

            case 'duitnow':
                return $this->handleDuitNowPayment($salesPageOrder);

            case 'cod':
                return $this->handleCodPayment($salesPageOrder);
        }
    }

    /**
     * Handle FPX payment
     */
    public function handleFpxPayment(SalesPageOrder $salesPageOrder)
    {
        try {
            // Get the sales page owner (seller)
            $seller = $salesPageOrder->salesPage->user;
            $service = new BayarCashService($seller);

            // Generate unique order number
            $orderNumber = 'SP' . $salesPageOrder->id . '-' . time();

            $paymentIntent = $service->createFpxPayment(
                orderNumber: $orderNumber,
                amount: $salesPageOrder->grand_total,
                payerName: $salesPageOrder->customer_name,
                payerEmail: $salesPageOrder->customer_email ?? 'noemail@example.com',
                payerPhone: $salesPageOrder->customer_phone,
                returnUrl: route('sales-page.payment.return'),
                callbackUrl: route('sales-page.payment.callback')
            );

            // Create payment record
            SalesPagePayment::create([
                'sales_page_order_id' => $salesPageOrder->id,
                'payment_method' => 'fpx',
                'bayarcash_order_number' => $orderNumber,
                'bayarcash_payment_url' => $paymentIntent->url ?? null,
                'status' => 'pending',
                'amount' => $salesPageOrder->grand_total,
                'transaction_data' => [
                    'id' => $paymentIntent->id ?? null,
                    'order_number' => $paymentIntent->orderNumber ?? null,
                    'amount' => $paymentIntent->amount ?? null,
                    'status' => $paymentIntent->status ?? null,
                    'url' => $paymentIntent->url ?? null,
                ],
                'expires_at' => now()->addHours(24),
            ]);

            // Redirect to payment gateway
            return redirect($paymentIntent->url);

        } catch (\Exception $e) {
            Log::error('FPX Payment Error: ' . $e->getMessage());

            return redirect()->route('sales-page.payment.select', $salesPageOrder->id)
                ->with('error', 'Failed to process payment. Please try again.');
        }
    }

    /**
     * Handle DuitNow QR payment
     */
    public function handleDuitNowPayment(SalesPageOrder $salesPageOrder)
    {
        try {
            // Get the sales page owner (seller)
            $seller = $salesPageOrder->salesPage->user;
            $service = new BayarCashService($seller);

            // Generate unique order number
            $orderNumber = 'SP' . $salesPageOrder->id . '-' . time();

            $paymentIntent = $service->createDuitNowPayment(
                orderNumber: $orderNumber,
                amount: $salesPageOrder->grand_total,
                payerName: $salesPageOrder->customer_name,
                payerEmail: $salesPageOrder->customer_email ?? 'noemail@example.com',
                payerPhone: $salesPageOrder->customer_phone,
                returnUrl: route('sales-page.payment.return'),
                callbackUrl: route('sales-page.payment.callback')
            );

            // Create payment record
            SalesPagePayment::create([
                'sales_page_order_id' => $salesPageOrder->id,
                'payment_method' => 'duitnow',
                'bayarcash_order_number' => $orderNumber,
                'bayarcash_payment_url' => $paymentIntent->url ?? null,
                'status' => 'pending',
                'amount' => $salesPageOrder->grand_total,
                'transaction_data' => [
                    'id' => $paymentIntent->id ?? null,
                    'order_number' => $paymentIntent->orderNumber ?? null,
                    'amount' => $paymentIntent->amount ?? null,
                    'status' => $paymentIntent->status ?? null,
                    'url' => $paymentIntent->url ?? null,
                ],
                'expires_at' => now()->addHours(24),
            ]);

            // Redirect to payment gateway
            return redirect($paymentIntent->url);

        } catch (\Exception $e) {
            Log::error('DuitNow Payment Error: ' . $e->getMessage());

            return redirect()->route('sales-page.payment.select', $salesPageOrder->id)
                ->with('error', 'Failed to process payment. Please try again.');
        }
    }

    /**
     * Handle COD payment
     */
    public function handleCodPayment(SalesPageOrder $salesPageOrder)
    {
        DB::beginTransaction();

        try {
            // Create payment record
            $payment = SalesPagePayment::create([
                'sales_page_order_id' => $salesPageOrder->id,
                'payment_method' => 'cod',
                'status' => 'pending',
                'amount' => $salesPageOrder->grand_total,
            ]);

            // Create main Order record
            $order = $this->createMainOrder($salesPageOrder);

            // Update sales page order
            $salesPageOrder->update([
                'order_id' => $order->id,
                'payment_status' => 'pending',
                'status' => 'confirmed',
            ]);

            DB::commit();

            // Redirect to thank you page
            return redirect()->route('sales-page.thank-you', ['order' => $order->id]);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('COD Payment Error: ' . $e->getMessage());

            return redirect()->route('sales-page.payment.select', $salesPageOrder->id)
                ->with('error', 'Failed to process order. Please try again.');
        }
    }

    /**
     * Handle BayarCash callback
     */
    public function handleCallback(Request $request)
    {
        Log::info('BayarCash Callback Received', $request->all());

        try {
            $orderNumber = $request->input('order_number');

            // Find payment by order number
            $payment = SalesPagePayment::where('bayarcash_order_number', $orderNumber)->first();

            if (!$payment) {
                Log::error('Payment not found for order: ' . $orderNumber);
                return response('Order not found', 404);
            }

            $salesPageOrder = $payment->salesPageOrder;
            $seller = $salesPageOrder->salesPage->user;

            // Initialize service with seller credentials
            $service = new BayarCashService($seller);

            // Verify callback
            if (!$service->verifyCallback($request->all())) {
                Log::error('BayarCash Callback Verification Failed');
                return response('Invalid signature', 400);
            }

            $status = $request->input('status');
            $statusDescription = $request->input('status_description');

            // BayarCash sends status as numeric code: 1=Pending, 2=Failed, 3=Successful
            $isSuccessful = ($status === '3' || $status === 3 ||
                           $status === 'success' || $status === 'successful' ||
                           strtolower($statusDescription ?? '') === 'successful');

            // Update payment based on status
            if ($isSuccessful) {
                DB::beginTransaction();

                try {
                    // Update payment status if not already paid
                    if ($payment->status !== 'paid') {
                        $payment->update([
                            'status' => 'paid',
                            'paid_at' => now(),
                            'transaction_data' => array_merge(
                                $payment->transaction_data ?? [],
                                ['callback' => $request->all()]
                            ),
                        ]);
                    }

                    // Create main order if not exists
                    if (!$salesPageOrder->order_id) {
                        $order = $this->createMainOrder($salesPageOrder);

                        $salesPageOrder->update([
                            'order_id' => $order->id,
                            'payment_status' => 'paid',
                            'status' => 'confirmed',
                        ]);

                        Log::info('Order created from callback', ['order_id' => $order->id]);
                    } else {
                        // Order already exists (created by return URL), just update status
                        $salesPageOrder->update([
                            'payment_status' => 'paid',
                            'status' => 'confirmed',
                        ]);

                        // Update main order
                        $salesPageOrder->order->update([
                            'status' => 'processing',
                            'payment_method' => $payment->payment_method,
                        ]);

                        Log::info('Order already exists, updated from callback', ['order_id' => $salesPageOrder->order_id]);
                    }

                    DB::commit();

                } catch (\Exception $e) {
                    DB::rollBack();
                    Log::error('Error processing callback: ' . $e->getMessage());
                }

            } elseif ($status === '2' || $status === 2 || $status === 'failed') {
                $payment->update([
                    'status' => 'failed',
                    'transaction_data' => array_merge(
                        $payment->transaction_data ?? [],
                        ['callback' => $request->all()]
                    ),
                ]);

                $salesPageOrder->update(['payment_status' => 'failed']);
            }

            return response('OK', 200);

        } catch (\Exception $e) {
            Log::error('BayarCash Callback Error: ' . $e->getMessage());
            return response('Server error', 500);
        }
    }

    /**
     * Handle return from BayarCash
     */
    public function handleReturn(Request $request)
    {
        Log::info('BayarCash Return URL Received', [
            'all_params' => $request->all(),
            'query_params' => $request->query(),
            'post_params' => $request->post(),
            'method' => $request->method(),
            'url' => $request->fullUrl()
        ]);

        try {
            $orderNumber = $request->input('order_number');

            if (!$orderNumber) {
                Log::error('No order_number in return URL', ['params' => $request->all()]);
                return redirect()->route('home')->with('error', 'Invalid payment response - no order number');
            }

            Log::info('Looking for payment with order number', ['order_number' => $orderNumber]);

            $payment = SalesPagePayment::where('bayarcash_order_number', $orderNumber)->first();

            if (!$payment) {
                Log::error('Payment not found', ['order_number' => $orderNumber]);
                return redirect()->route('home')->with('error', 'Payment not found');
            }

            Log::info('Payment found', ['payment_id' => $payment->id, 'sales_page_order_id' => $payment->sales_page_order_id]);

            $salesPageOrder = $payment->salesPageOrder;
            $seller = $salesPageOrder->salesPage->user;

            // Initialize service with seller credentials
            $service = new BayarCashService($seller);

            // Verify return URL callback (non-blocking for debugging)
            $verificationPassed = false;
            try {
                $verificationPassed = $service->verifyReturnCallback($request->all());
                Log::info('BayarCash Return URL Verification Result', ['passed' => $verificationPassed]);
            } catch (\Exception $e) {
                Log::error('BayarCash Return URL Verification Exception', ['error' => $e->getMessage()]);
            }

            // Comment out this check temporarily for debugging
            // if (!$verificationPassed) {
            //     Log::error('BayarCash Return URL Verification Failed');
            //     return redirect()->route('home')->with('error', 'Invalid payment response');
            // }

            $status = $request->input('status');
            $statusDescription = $request->input('status_description');
            Log::info('Payment status from return URL', ['status' => $status, 'status_description' => $statusDescription]);

            // BayarCash sends status as numeric code: 1=Pending, 2=Failed, 3=Successful
            $isSuccessful = ($status === '3' || $status === 3 ||
                           $status === 'success' || $status === 'successful' ||
                           strtolower($statusDescription ?? '') === 'successful');

            Log::info('Payment success check', ['is_successful' => $isSuccessful]);

            // If payment is successful, create order if not exists
            if ($isSuccessful && !$salesPageOrder->order_id) {
                DB::beginTransaction();

                try {
                    // Update payment status
                    $payment->update([
                        'status' => 'paid',
                        'paid_at' => now(),
                        'transaction_data' => array_merge(
                            $payment->transaction_data ?? [],
                            ['return' => $request->all()]
                        ),
                    ]);

                    // Create main order
                    $order = $this->createMainOrder($salesPageOrder);

                    // Update sales page order
                    $salesPageOrder->update([
                        'order_id' => $order->id,
                        'payment_status' => 'paid',
                        'status' => 'confirmed',
                    ]);

                    DB::commit();

                    Log::info('Order created from return URL', ['order_id' => $order->id]);

                    // Redirect to thank you page
                    return redirect()->route('sales-page.thank-you', ['order' => $order->id]);

                } catch (\Exception $e) {
                    DB::rollBack();
                    Log::error('Error creating order from return URL: ' . $e->getMessage());
                    return redirect()->route('home')->with('error', 'An error occurred processing your payment');
                }
            }

            // If order already exists, redirect to thank you page
            if ($salesPageOrder->order_id) {
                Log::info('Order already exists, redirecting to thank you page', ['order_id' => $salesPageOrder->order_id]);
                return redirect()->route('sales-page.thank-you', ['order' => $salesPageOrder->order_id]);
            }

            // If payment failed (status code 2 or "failed")
            $isFailed = ($status === '2' || $status === 2 || $status === 'failed');
            if ($isFailed) {
                $payment->update([
                    'status' => 'failed',
                    'transaction_data' => array_merge(
                        $payment->transaction_data ?? [],
                        ['return' => $request->all()]
                    ),
                ]);

                $salesPageOrder->update(['payment_status' => 'failed']);

                return redirect()->route('sales-page.payment.select', $salesPageOrder->id)
                    ->with('error', 'Payment failed. Please try again.');
            }

            // If status is pending or unknown, show pending message
            return view('sales-pages.public.payment-pending', compact('salesPageOrder', 'payment'));

        } catch (\Exception $e) {
            Log::error('Payment Return Error: ' . $e->getMessage());
            return redirect()->route('home')->with('error', 'An error occurred');
        }
    }


    /**
     * Create main Order record from SalesPageOrder
     */
    protected function createMainOrder(SalesPageOrder $salesPageOrder): Order
    {
        $salesPage = $salesPageOrder->salesPage;
        $seller = $salesPage->user;

        // Map payment method to title
        $paymentMethodTitles = [
            'cod' => 'Cash on Delivery',
            'fpx' => 'Online Banking (FPX)',
            'duitnow' => 'DuitNow QR',
        ];

        // Generate order number with seller code
        $orderNumberService = app(OrderNumberService::class);
        $orderNumber = $orderNumberService->generateSalesPageOrderNumber($seller);

        // Split customer name into first and last name for Pos Laju compatibility
        $nameParts = explode(' ', trim($salesPageOrder->customer_name), 2);
        $firstName = $nameParts[0] ?? '';
        $lastName = $nameParts[1] ?? '';

        // Calculate total fees (COD + Payment Gateway)
        $totalFees = ($salesPageOrder->cod_fee ?? 0) + ($salesPageOrder->payment_gateway_fee ?? 0);

        // Build fee notes
        $feeNotes = [];
        if ($salesPageOrder->cod_fee > 0) {
            $feeNotes[] = 'COD Fee: RM ' . number_format($salesPageOrder->cod_fee, 2);
        }
        if ($salesPageOrder->payment_gateway_fee > 0) {
            $feeNotes[] = 'Payment Gateway Fee: RM ' . number_format($salesPageOrder->payment_gateway_fee, 2);
        }
        $feeNotesString = !empty($feeNotes) ? ' (' . implode(', ', $feeNotes) . ')' : '';

        $order = Order::create([
            'store_id' => null,
            'order_number' => $orderNumber,
            'global_order_id' => $this->generateGlobalOrderId(),
            'status' => $salesPageOrder->payment_method === 'cod' ? 'pending' : 'processing',
            'payment_method' => $salesPageOrder->payment_method,
            'payment_method_title' => $paymentMethodTitles[$salesPageOrder->payment_method] ?? $salesPageOrder->payment_method,
            'billing' => [
                'first_name' => $firstName,
                'last_name' => $lastName,
                'email' => $salesPageOrder->customer_email ?? '',
                'phone' => $salesPageOrder->customer_phone,
                'address_1' => $salesPageOrder->customer_address,
                'address_2' => '',
                'city' => $salesPageOrder->customer_city,
                'state' => $salesPageOrder->customer_state,
                'postcode' => $salesPageOrder->customer_postcode,
                'country' => 'MY',
            ],
            'shipping' => [
                'first_name' => $firstName,
                'last_name' => $lastName,
                'email' => $salesPageOrder->customer_email ?? '',
                'phone' => $salesPageOrder->customer_phone,
                'address_1' => $salesPageOrder->customer_address,
                'address_2' => '',
                'city' => $salesPageOrder->customer_city,
                'state' => $salesPageOrder->customer_state,
                'postcode' => $salesPageOrder->customer_postcode,
                'country' => 'MY',
            ],
            'line_items' => $salesPageOrder->items,
            'total' => $salesPageOrder->grand_total,
            'subtotal' => $salesPageOrder->total_amount,
            'shipping_total' => 0,
            'fee_total' => $totalFees,
            'currency' => 'MYR',
            'notes' => 'Order from Sales Page: ' . $salesPage->name . $feeNotesString,
            'is_manual' => true,
            'created_by' => $salesPage->user_id,
            'date_created' => now(),
        ]);

        // Clear dashboard cache to show new order immediately
        $this->clearDashboardCache();

        return $order;
    }

    /**
     * Generate global order ID
     */
    protected function generateGlobalOrderId()
    {
        // Get the prefix from system settings (default: OMS)
        $prefix = SystemSetting::getGlobalIdPrefix();
        $lastOrder = Order::orderBy('id', 'desc')->first();
        $nextNumber = $lastOrder ? $lastOrder->id + 1 : 1;

        if ($nextNumber < 10000000) {
            return $prefix . str_pad($nextNumber, 8, '0', STR_PAD_LEFT);
        } else {
            return $prefix . $nextNumber;
        }
    }

    /**
     * Clear dashboard cache for all users to show new orders immediately
     */
    private function clearDashboardCache()
    {
        // Clear all dashboard-related cache keys
        $patterns = [
            'dashboard_period_stats_*',
            'dashboard_today_stats_*'
        ];

        foreach ($patterns as $pattern) {
            // Get all cache keys matching pattern and delete them
            // Since Laravel's file cache doesn't support wildcard delete natively,
            // we'll use a simple approach to clear common user types
            $userTypes = ['all', 'admin'];

            // Add seller cache keys
            $sellers = \App\Models\User::where('role', 'seller')->pluck('id');
            foreach ($sellers as $sellerId) {
                $userTypes[] = 'seller_' . $sellerId;
            }

            // Add manager cache keys
            $managers = \App\Models\User::where('role', 'manager')->pluck('id');
            foreach ($managers as $managerId) {
                $userTypes[] = 'manager_' . $managerId;
            }

            // Clear cache for each user type and time combination
            foreach ($userTypes as $userType) {
                // Clear stats for different time ranges (today's hour, recent dates)
                for ($i = 0; $i < 24; $i++) {
                    $time = now()->subHours($i);

                    // Today stats
                    \Cache::forget("dashboard_today_stats_{$userType}_" . $time->format('Y-m-d_H'));

                    // Period stats - common periods
                    $periods = [
                        ['start' => $time->copy()->startOfDay(), 'end' => $time->copy()->endOfDay()],
                        ['start' => $time->copy()->startOfWeek(), 'end' => $time->copy()->endOfWeek()],
                        ['start' => $time->copy()->startOfMonth(), 'end' => $time->copy()->endOfMonth()],
                    ];

                    foreach ($periods as $period) {
                        \Cache::forget("dashboard_period_stats_{$userType}_{$period['start']->format('Y-m-d_H-i')}_{$period['end']->format('Y-m-d_H-i')}");
                    }
                }
            }
        }
    }
}
