<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class PushSubscriptionController extends Controller
{
    /**
     * Store a new push subscription for the authenticated user
     */
    public function store(Request $request)
    {
        try {
            $user = $request->user();

            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not authenticated'
                ], 401);
            }

            $subscription = $request->all();

            // Validate subscription data
            if (!isset($subscription['endpoint']) || !isset($subscription['keys'])) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid subscription data'
                ], 422);
            }

            // Use package helper to create or update subscription
            $user->updatePushSubscription(
                $subscription['endpoint'],
                $subscription['keys']['p256dh'] ?? null,
                $subscription['keys']['auth'] ?? null,
                $subscription['contentEncoding'] ?? null
            );

            Log::info('Push subscription saved', [
                'user_id' => $user->id,
                'endpoint' => substr($subscription['endpoint'], 0, 50) . '...'
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Subscription saved successfully'
            ]);

        } catch (\Exception $e) {
            Log::error('Push subscription error', [
                'error' => $e->getMessage(),
                'user_id' => $request->user()->id ?? null
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to save subscription: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Delete a push subscription
     */
    public function destroy(Request $request)
    {
        try {
            $user = $request->user();
            $endpoint = $request->input('endpoint');

            if (!$endpoint) {
                return response()->json([
                    'success' => false,
                    'message' => 'Endpoint required'
                ], 422);
            }

            $user->deletePushSubscription($endpoint);

            Log::info('Push subscription deleted', [
                'user_id' => $user->id,
                'endpoint' => substr($endpoint, 0, 50) . '...'
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Subscription deleted successfully'
            ]);

        } catch (\Exception $e) {
            Log::error('Push subscription delete error', [
                'error' => $e->getMessage(),
                'user_id' => $request->user()->id ?? null
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to delete subscription'
            ], 500);
        }
    }

    /**
     * Get VAPID public key for client-side subscription
     */
    public function vapidPublicKey()
    {
        return response()->json([
            'publicKey' => config('webpush.vapid.public_key')
        ]);
    }
}
