<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rules\Password;

class ProfileController extends Controller
{
    public function edit()
    {
        $user = auth()->user();
        return view('profile.edit', compact('user'));
    }

    public function update(Request $request)
    {
        $user = auth()->user();

        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'short_name' => ['nullable', 'string', 'max:20'],
            'url_slug' => [
                'nullable',
                'string',
                'max:50',
                'regex:/^[a-z0-9-]+$/',
                'unique:users,url_slug,' . $user->id
            ],
            'auto_fill_customer_info' => ['nullable', 'boolean'],
            'email' => ['required', 'email', 'max:255', 'unique:users,email,' . $user->id],
            'phone' => ['nullable', 'string', 'max:20'],
            'profile_photo' => ['nullable', 'image', 'max:2048'], // 2MB max
            'company_name' => ['nullable', 'string', 'max:255'],
            'company_registration' => ['nullable', 'string', 'max:255'],
            'company_address' => ['nullable', 'string'],
            'company_city' => ['nullable', 'string', 'max:255'],
            'company_state' => ['nullable', 'string', 'max:255'],
            'company_postcode' => ['nullable', 'string', 'max:10'],
            'company_country' => ['nullable', 'string', 'max:255'],
            'company_phone' => ['nullable', 'string', 'max:20'],
            'company_email' => ['nullable', 'email', 'max:255'],
            'company_website' => ['nullable', 'url', 'max:255'],
            'company_logo' => ['nullable', 'image', 'max:2048'], // 2MB max
            'use_own_payment_gateway' => ['nullable', 'boolean'],
            'bayarcash_portal_key' => ['nullable', 'string', 'max:255'],
            'bayarcash_pat' => ['nullable', 'string', 'max:1000'],
            'bayarcash_secret_key' => ['nullable', 'string', 'max:255'],
            'bayarcash_environment' => ['nullable', 'in:sandbox,production'],
            'fpx_fee_type' => ['nullable', 'in:none,fixed,percentage,both'],
            'fpx_fixed_fee' => ['nullable', 'numeric', 'min:0', 'max:9999999.99'],
            'fpx_percentage_fee' => ['nullable', 'numeric', 'min:0', 'max:100'],
            'fpx_fee_bearer' => ['nullable', 'in:customer,merchant'],
            'duitnow_fee_type' => ['nullable', 'in:none,fixed,percentage,both'],
            'duitnow_fixed_fee' => ['nullable', 'numeric', 'min:0', 'max:9999999.99'],
            'duitnow_percentage_fee' => ['nullable', 'numeric', 'min:0', 'max:100'],
            'duitnow_fee_bearer' => ['nullable', 'in:customer,merchant'],
        ], [
            'url_slug.regex' => 'The URL slug can only contain lowercase letters, numbers, and hyphens.',
            'url_slug.unique' => 'This URL slug is already taken. Please choose a different one.',
        ]);

        // Convert checkbox value to boolean (0 if not present)
        $validated['auto_fill_customer_info'] = $request->has('auto_fill_customer_info') ? 1 : 0;
        $validated['use_own_payment_gateway'] = $request->has('use_own_payment_gateway') ? 1 : 0;

        try {
            // Handle profile photo upload
            if ($request->hasFile('profile_photo')) {
                // Delete old photo if exists
                if ($user->profile_photo) {
                    $oldPhotoPath = public_path($user->profile_photo);
                    if (file_exists($oldPhotoPath)) {
                        @unlink($oldPhotoPath);
                    }
                }

                // Create directory if it doesn't exist
                $photoDir = public_path('images/profile-photos');
                if (!file_exists($photoDir)) {
                    @mkdir($photoDir, 0755, true);
                }

                // Store the uploaded file
                $file = $request->file('profile_photo');
                $fileName = 'profile_' . $user->id . '_' . time() . '.' . $file->getClientOriginalExtension();
                $file->move($photoDir, $fileName);

                // Save relative path
                $validated['profile_photo'] = '/images/profile-photos/' . $fileName;
            }

            // Handle logo upload
            if ($request->hasFile('company_logo')) {
                // Delete old logo if exists (check both old storage path and new public path)
                if ($user->company_logo) {
                    // Old storage path
                    if (Storage::disk('public')->exists($user->company_logo)) {
                        @Storage::disk('public')->delete($user->company_logo);
                    }
                    // New public path
                    if (str_starts_with($user->company_logo, '/images/')) {
                        $oldPath = public_path($user->company_logo);
                        if (file_exists($oldPath)) {
                            @unlink($oldPath);
                        }
                    }
                }

                // Create images directory if it doesn't exist
                $imagesPath = public_path('images');
                if (!file_exists($imagesPath)) {
                    @mkdir($imagesPath, 0755, true);
                }

                // Store the uploaded file
                $file = $request->file('company_logo');
                $fileName = 'company_' . $user->id . '_' . time() . '_' . preg_replace('/[^a-zA-Z0-9._-]/', '', $file->getClientOriginalName());
                $file->move($imagesPath, $fileName);

                // Save relative path
                $validated['company_logo'] = '/images/' . $fileName;
            }

            $user->update($validated);

            return redirect()->route('profile.edit')
                ->with('success', 'Profile updated successfully!');
        } catch (\Exception $e) {
            \Log::error('Profile update error: ' . $e->getMessage(), [
                'user_id' => $user->id,
                'trace' => $e->getTraceAsString()
            ]);

            return redirect()->route('profile.edit')
                ->with('error', 'Failed to update profile. Please try again or contact support.');
        }
    }

    public function updatePassword(Request $request)
    {
        $validated = $request->validate([
            'current_password' => ['required', 'current_password'],
            'password' => ['required', Password::defaults(), 'confirmed'],
        ]);

        auth()->user()->update([
            'password' => Hash::make($validated['password'])
        ]);

        return redirect()->route('profile.edit')
            ->with('success', 'Password updated successfully!');
    }
}
