<?php

namespace App\Http\Controllers;

use App\Models\Product;
use Illuminate\Http\Request;

class ProductController extends Controller
{
    /**
     * Display a listing of the products
     * Shows user's own products AND admin products (global)
     */
    public function index(Request $request)
    {
        $user = $request->user();

        // Eager load user's product relationships to prevent N+1 queries
        $user->load('products');

        $query = Product::with('user:id,name,role')
            ->accessibleByUser($user->id);

        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('sku', 'like', "%{$search}%");
            });
        }

        // Filter by status
        if ($request->filled('status')) {
            $query->where('is_active', $request->status === 'active');
        }

        // Filter by ownership type (own products vs admin products)
        if ($request->filled('type')) {
            if ($request->type === 'own') {
                $query->where('user_id', $user->id);
            } elseif ($request->type === 'global') {
                $query->adminProducts();
            } elseif (in_array($request->type, ['single', 'bundle'])) {
                // Filter by product_type (single or bundle)
                $query->ofType($request->type);
            }
        }

        $products = $query->orderBy('name')->paginate(50);

        // Eager load the authenticated user's product relationships for checkbox state
        $user->load('products');

        return view('products.index', compact('products'));
    }

    /**
     * Show the form for creating a new product
     */
    public function create()
    {
        return view('products.create');
    }

    /**
     * Store a newly created product
     */
    public function store(Request $request)
    {
        $user = $request->user();

        $validated = $request->validate([
            'sku' => [
                'nullable',
                'string',
                'max:255',
                // Check SKU uniqueness: must be unique within user's products
                // Admin products are global, so admin SKUs must be globally unique
                function ($attribute, $value, $fail) use ($user) {
                    if (empty($value)) return;

                    // Check if SKU already exists for this user
                    $existsForUser = Product::where('user_id', $user->id)
                        ->where('sku', $value)
                        ->exists();

                    if ($existsForUser) {
                        $fail('This SKU already exists in your products.');
                        return;
                    }

                    // If user is seller, also check if SKU exists in admin products (global)
                    if ($user->role !== 'admin') {
                        $existsInAdmin = Product::whereHas('user', function($query) {
                            $query->where('role', 'admin');
                        })->where('sku', $value)->exists();

                        if ($existsInAdmin) {
                            $fail('This SKU is already used in a global product. Please use a different SKU.');
                        }
                    }
                },
            ],
            'name' => 'required|string|max:255',
            'price' => 'required|numeric|min:0',
            'stock' => 'nullable|integer|min:0',
            'description' => 'nullable|string',
            'is_active' => 'boolean',
            'product_type' => 'required|in:single,bundle',
            'bundle_items' => 'nullable|array',
            'bundle_items.*.product_id' => 'required_if:product_type,bundle|exists:products,id',
            'bundle_items.*.sku' => 'nullable|string',
            'bundle_items.*.name' => 'nullable|string',
            'bundle_items.*.quantity' => 'required_if:product_type,bundle|integer|min:1',
        ]);

        $validated['user_id'] = $user->id;
        $validated['is_active'] = $request->has('is_active');

        // Clear bundle_items if product_type is 'single'
        if ($validated['product_type'] === 'single') {
            $validated['bundle_items'] = null;
        }

        Product::create($validated);

        return redirect()->route('products.index')
            ->with('success', 'Product created successfully!');
    }

    /**
     * Show the form for editing the specified product
     */
    public function edit(Product $product)
    {
        // Check if user owns this product
        if ($product->user_id !== auth()->id()) {
            abort(403, 'Unauthorized access to this product.');
        }

        return view('products.edit', compact('product'));
    }

    /**
     * Update the specified product
     */
    public function update(Request $request, Product $product)
    {
        $user = $request->user();

        // Check if user owns this product
        if ($product->user_id !== $user->id) {
            abort(403, 'Unauthorized access to this product.');
        }

        $validated = $request->validate([
            'sku' => [
                'nullable',
                'string',
                'max:255',
                // Check SKU uniqueness (excluding current product)
                function ($attribute, $value, $fail) use ($user, $product) {
                    if (empty($value)) return;

                    // Check if SKU already exists for this user (excluding current product)
                    $existsForUser = Product::where('user_id', $user->id)
                        ->where('sku', $value)
                        ->where('id', '!=', $product->id)
                        ->exists();

                    if ($existsForUser) {
                        $fail('This SKU already exists in your products.');
                        return;
                    }

                    // If user is seller, also check if SKU exists in admin products (global)
                    if ($user->role !== 'admin') {
                        $existsInAdmin = Product::whereHas('user', function($query) {
                            $query->where('role', 'admin');
                        })->where('sku', $value)->exists();

                        if ($existsInAdmin) {
                            $fail('This SKU is already used in a global product. Please use a different SKU.');
                        }
                    }
                },
            ],
            'name' => 'required|string|max:255',
            'price' => 'required|numeric|min:0',
            'stock' => 'nullable|integer|min:0',
            'description' => 'nullable|string',
            'is_active' => 'boolean',
            'product_type' => 'required|in:single,bundle',
            'bundle_items' => 'nullable|array',
            'bundle_items.*.product_id' => 'required_if:product_type,bundle|exists:products,id',
            'bundle_items.*.sku' => 'nullable|string',
            'bundle_items.*.name' => 'nullable|string',
            'bundle_items.*.quantity' => 'required_if:product_type,bundle|integer|min:1',
        ]);

        $validated['is_active'] = $request->has('is_active');

        // Clear bundle_items if product_type is 'single'
        if ($validated['product_type'] === 'single') {
            $validated['bundle_items'] = null;
        }

        $product->update($validated);

        return redirect()->route('products.index')
            ->with('success', 'Product updated successfully!');
    }

    /**
     * Remove the specified product
     */
    public function destroy(Product $product)
    {
        // Check if user owns this product
        if ($product->user_id !== auth()->id()) {
            abort(403, 'Unauthorized access to this product.');
        }

        $product->delete();

        return redirect()->route('products.index')
            ->with('success', 'Product deleted successfully!');
    }

    /**
     * Toggle product availability for this user's checkout
     */
    public function toggle(Request $request, Product $product)
    {
        $validated = $request->validate([
            'is_enabled' => 'required|boolean',
        ]);

        $user = $request->user();

        // Sync the product to user's enabled products list
        // This will either create or update the pivot record
        $user->products()->syncWithoutDetaching([
            $product->id => ['is_enabled' => $validated['is_enabled']]
        ]);

        return response()->json([
            'success' => true,
            'message' => $validated['is_enabled']
                ? 'Product enabled for checkout'
                : 'Product disabled for checkout'
        ]);
    }

    /**
     * Search products for checkout (AJAX)
     * Returns only products that are enabled for this user's checkout
     */
    public function search(Request $request)
    {
        $user = $request->user();
        $search = $request->get('q', '');

        // Get only enabled products for this user
        $products = $user->enabledProducts()
            ->with('user:id,name,role')
            ->where('is_active', true)
            ->where(function($query) use ($search) {
                $query->where('products.name', 'like', "%{$search}%")
                      ->orWhere('products.sku', 'like', "%{$search}%");
            })
            ->orderBy('products.name', 'asc')
            ->get(['products.id', 'products.sku', 'products.name', 'products.price', 'products.user_id', 'products.product_type']);

        // Add a flag to indicate if product is global (admin product)
        $products = $products->map(function($product) {
            return [
                'id' => $product->id,
                'sku' => $product->sku,
                'name' => $product->name,
                'price' => $product->price,
                'is_global' => $product->isAdminProduct(),
                'is_bundle' => $product->isBundle(),
                'product_type' => $product->product_type,
                'display_name' => $product->name . ($product->isAdminProduct() ? ' (Global)' : ''),
            ];
        });

        return response()->json($products);
    }
}
