<?php

namespace App\Http\Controllers;

use App\Models\Order;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class PoslajuWebhookController extends Controller
{
    /**
     * Handle incoming webhook from Pos Laju
     * This endpoint receives tracking status updates from Pos Laju API
     * Also handles GET requests for webhook validation
     */
    public function handle(Request $request)
    {
        // Handle GET request for webhook validation
        if ($request->isMethod('GET')) {
            Log::info('Pos Laju Webhook Validation Request', [
                'method' => 'GET',
                'query' => $request->query(),
                'ip' => $request->ip(),
            ]);

            // Return success response for validation
            return response()->json([
                'success' => true,
                'message' => 'Webhook endpoint is active',
                'timestamp' => now()->toIso8601String()
            ], 200);
        }

        // Log the incoming webhook for debugging
        Log::info('Pos Laju Webhook Received', [
            'method' => $request->method(),
            'headers' => $request->headers->all(),
            'query' => $request->query(),
            'body' => $request->all(),
            'raw_body' => $request->getContent(),
        ]);

        try {
            // Get webhook data (Pos Laju sends as query params or JSON body)
            $data = $request->all();

            // Extract tracking information
            $trackingNumber = $data['connote_no']
                ?? $data['Tracking_Number']
                ?? $data['tracking_number']
                ?? null;

            $eventCode = $data['event_code']
                ?? $data['Event_Code']
                ?? $data['eventCode']
                ?? null;

            $eventDescription = $data['event_description']
                ?? $data['Event_Description']
                ?? $data['reason']
                ?? null;

            $status = $data['status']
                ?? $data['Status']
                ?? null;

            if (!$trackingNumber) {
                Log::warning('Pos Laju Webhook: Missing tracking number', ['data' => $data]);
                return response()->json([
                    'success' => false,
                    'message' => 'Missing tracking number'
                ], 400);
            }

            // Find order by tracking number
            $order = Order::where('tracking_number', $trackingNumber)->first();

            if (!$order) {
                Log::warning('Pos Laju Webhook: Order not found', [
                    'tracking_number' => $trackingNumber
                ]);

                // Still return 200 to acknowledge receipt
                return response()->json([
                    'success' => true,
                    'message' => 'Order not found but webhook acknowledged'
                ], 200);
            }

            // Update order with tracking event
            $this->updateOrderStatus($order, $eventCode, $eventDescription, $status, $data);

            Log::info('Pos Laju Webhook: Order updated successfully', [
                'order_id' => $order->id,
                'tracking_number' => $trackingNumber,
                'event_code' => $eventCode,
                'status' => $status
            ]);

            // Return success response
            return response()->json([
                'success' => true,
                'message' => 'Webhook processed successfully',
                'order_id' => $order->id
            ], 200);

        } catch (\Exception $e) {
            Log::error('Pos Laju Webhook Error', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'data' => $request->all()
            ]);

            // Still return 200 to prevent Pos Laju from retrying
            return response()->json([
                'success' => false,
                'message' => 'Webhook processing failed: ' . $e->getMessage()
            ], 200);
        }
    }

    /**
     * Update order status based on Pos Laju tracking event
     */
    protected function updateOrderStatus($order, $eventCode, $eventDescription, $status, $webhookData)
    {
        // Store webhook data in order
        $trackingEvents = $order->poslaju_tracking_events ?? [];
        $trackingEvents[] = [
            'event_code' => $eventCode,
            'event_description' => $eventDescription,
            'status' => $status,
            'received_at' => now()->toIso8601String(),
            'data' => $webhookData
        ];

        $order->poslaju_tracking_events = $trackingEvents;

        // Pos Laju Event Codes for delivery:
        // DLV, DEL, POD = Delivered/Proof of Delivery
        // Event codes reference: https://api-doc.pos.com.my/
        $deliveredEventCodes = ['DLV', 'DEL', 'POD', 'DELIVERED'];

        // Check if this is a delivery event
        $isDelivered = false;

        if ($eventCode && in_array(strtoupper($eventCode), $deliveredEventCodes)) {
            $isDelivered = true;
        }

        // Also check event description for "delivered" keyword
        if ($eventDescription && stripos($eventDescription, 'delivered') !== false) {
            $isDelivered = true;
        }

        // Auto-complete order if delivered
        if ($isDelivered && $order->status !== 'completed') {
            $oldStatus = $order->status;
            $order->status = 'completed';
            $order->date_modified = now();
            $order->delivery_completed_at = now();
            $order->auto_completed_by_webhook = true;

            Log::info('Order auto-completed by Pos Laju webhook', [
                'order_id' => $order->id,
                'order_number' => $order->order_number,
                'old_status' => $oldStatus,
                'new_status' => 'completed',
                'tracking_number' => $order->tracking_number,
                'event_code' => $eventCode
            ]);

            // Sync status back to WooCommerce if applicable
            if ($order->store_id && $order->store) {
                try {
                    $this->syncCompletionStatusToWooCommerce($order);
                } catch (\Exception $e) {
                    Log::error('Failed to sync completion to WooCommerce', [
                        'order_id' => $order->id,
                        'error' => $e->getMessage()
                    ]);
                }
            }

            // Sync to webhook source if applicable
            if ($order->webhook_source_id && $order->webhookSource) {
                try {
                    $this->syncCompletionStatusToWebhook($order);
                } catch (\Exception $e) {
                    Log::error('Failed to sync completion to webhook source', [
                        'order_id' => $order->id,
                        'error' => $e->getMessage()
                    ]);
                }
            }
        }

        // Update last tracking status
        $order->last_tracking_status = $eventDescription ?? $status ?? 'Updated';
        $order->last_tracking_update = now();

        $order->save();
    }

    /**
     * Sync completion status to WooCommerce
     */
    protected function syncCompletionStatusToWooCommerce($order)
    {
        if (!$order->store || !$order->woo_order_id) {
            return;
        }

        try {
            $woocommerce = $order->store->getWooCommerceClient();

            $woocommerce->put('orders/' . $order->woo_order_id, [
                'status' => 'completed'
            ]);

            Log::info('Order completion synced to WooCommerce', [
                'order_id' => $order->id,
                'woo_order_id' => $order->woo_order_id,
                'store_id' => $order->store_id
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to sync order completion to WooCommerce', [
                'order_id' => $order->id,
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
     * Sync completion status to webhook source
     */
    protected function syncCompletionStatusToWebhook($order)
    {
        if (!$order->webhookSource || !$order->webhookSource->callback_url) {
            return;
        }

        try {
            $webhookSource = $order->webhookSource;

            $response = \Http::post($webhookSource->callback_url, [
                'event' => 'order.completed',
                'order_number' => $order->order_number,
                'tracking_number' => $order->tracking_number,
                'status' => 'completed',
                'completed_at' => now()->toIso8601String()
            ]);

            Log::info('Order completion synced to webhook source', [
                'order_id' => $order->id,
                'webhook_source_id' => $webhookSource->id,
                'callback_url' => $webhookSource->callback_url,
                'response_status' => $response->status()
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to sync order completion to webhook source', [
                'order_id' => $order->id,
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }
}
