<?php

namespace App\Http\Controllers;

use App\Models\Order;
use App\Services\OrderSyncService;
use App\Services\OrderNumberService;
use App\Services\PoslajuService;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf;
use setasign\Fpdi\Fpdi;

class OrderController extends Controller
{
    protected $syncService;
    protected $orderNumberService;

    public function __construct(OrderSyncService $syncService, OrderNumberService $orderNumberService)
    {
        $this->syncService = $syncService;
        $this->orderNumberService = $orderNumberService;
    }

    public function index(Request $request)
    {
        $user = $request->user();

        // Default to "processing" status if no status filter is set
        $currentStatus = $request->get('status', 'processing');

        $query = Order::with([
                'store:id,name,store_prefix',
                'webhookSource:id,name,type',
                'creator:id,name',
                'creator.stores:id,store_prefix'
            ])
            ->select(['id', 'store_id', 'webhook_source_id', 'is_manual', 'created_by', 'order_number', 'global_order_id', 'woo_order_id', 'status', 'total', 'currency', 'payment_method', 'payment_method_title', 'billing', 'shipping', 'line_items', 'tracking_number', 'date_created']);

        // Apply role-based filtering
        if ($user->isSeller()) {
            $storeIds = $user->getStoreIds();
            $query->where(function($q) use ($storeIds, $user) {
                $q->whereIn('store_id', $storeIds)
                  ->orWhere('created_by', $user->id);
            });
        } elseif ($user->isManager()) {
            $storeIds = $user->getStoreIds();
            $sellerIds = $user->getManagedSellerIds();
            $query->where(function($q) use ($storeIds, $sellerIds) {
                $q->whereIn('store_id', $storeIds)
                  ->orWhereIn('created_by', $sellerIds);
            });
        }

        // Source filtering (WooCommerce vs Webhook)
        if ($request->filled('source')) {
            if ($request->source === 'woocommerce') {
                $query->whereNotNull('store_id');
            } elseif ($request->source === 'webhook') {
                $query->whereNotNull('webhook_source_id');
            }
        }

        // Status filtering with tabs - "all" shows everything, specific status filters by that status
        if ($currentStatus && $currentStatus !== 'all') {
            $query->where('status', $currentStatus);
        }

        if ($request->filled('payment_method')) {
            $query->where('payment_method', $request->payment_method);
        }

        // Handle store/webhook/salespage/checkout filtering
        if ($request->filled('store_id')) {
            $storeId = $request->store_id;
            if (str_starts_with($storeId, 'store_')) {
                $query->where('store_id', str_replace('store_', '', $storeId));
            } elseif (str_starts_with($storeId, 'webhook_')) {
                $query->where('webhook_source_id', str_replace('webhook_', '', $storeId));
            } elseif (str_starts_with($storeId, 'salespage_')) {
                // Filter for sales page orders - Use sales_page_orders relationship for unique filtering
                $salesPageId = str_replace('salespage_', '', $storeId);
                $query->where('is_manual', true)
                      ->where('order_number', 'like', 'SP-%')
                      ->whereIn('id', function($subQuery) use ($salesPageId) {
                          $subQuery->select('order_id')
                                   ->from('sales_page_orders')
                                   ->where('sales_page_id', $salesPageId);
                      });
            } elseif (str_starts_with($storeId, 'checkout_user_')) {
                // Filter for checkout orders by specific user
                $userId = str_replace('checkout_user_', '', $storeId);
                $query->where('is_manual', true)->where('created_by', $userId);
            } elseif ($storeId === 'checkout') {
                // Filter for all manual checkout orders
                $query->where('is_manual', true);
            } else {
                // Legacy support - assume it's a store ID
                $query->where('store_id', $storeId);
            }
        }

        // Date range filtering (uses our new indexes)
        if ($request->filled('date_from')) {
            $query->where('date_created', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->where('date_created', '<=', $request->date_to . ' 23:59:59');
        }

        // Optimized search using indexed fields first
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                // Check indexed fields first (faster)
                $q->where('order_number', 'like', "%{$search}%")
                  ->orWhere('global_order_id', 'like', "%{$search}%");

                // JSON searches last (slower but indexed with functional indexes)
                if (filter_var($search, FILTER_VALIDATE_EMAIL)) {
                    $q->orWhereRaw("JSON_UNQUOTE(JSON_EXTRACT(billing, '$.email')) = ?", [$search]);
                } else {
                    $q->orWhereRaw("JSON_UNQUOTE(JSON_EXTRACT(billing, '$.email')) LIKE ?", ["%{$search}%"])
                      ->orWhereRaw("JSON_UNQUOTE(JSON_EXTRACT(billing, '$.phone')) LIKE ?", ["%{$search}%"]);
                }
            });
        }

        $perPage = $request->get('per_page', 25);
        $orders = $query->orderBy('date_created', 'desc')->paginate($perPage);

        // Get status counts for tabs (with role-based filtering)
        $statusCountQuery = Order::query();

        // Apply same role-based filtering as main query
        if ($user->isSeller()) {
            $storeIds = $user->getStoreIds();
            $statusCountQuery->where(function($q) use ($storeIds, $user) {
                $q->whereIn('store_id', $storeIds)
                  ->orWhere('created_by', $user->id);
            });
        } elseif ($user->isManager()) {
            $storeIds = $user->getStoreIds();
            $sellerIds = $user->getManagedSellerIds();
            $statusCountQuery->where(function($q) use ($storeIds, $sellerIds) {
                $q->whereIn('store_id', $storeIds)
                  ->orWhereIn('created_by', $sellerIds);
            });
        }

        // Apply same source filtering if present
        if ($request->filled('source')) {
            if ($request->source === 'woocommerce') {
                $statusCountQuery->whereNotNull('store_id');
            } elseif ($request->source === 'webhook') {
                $statusCountQuery->whereNotNull('webhook_source_id');
            }
        }

        // Apply same store/webhook/salespage/checkout filtering to status counts
        if ($request->filled('store_id')) {
            $storeId = $request->store_id;
            if (str_starts_with($storeId, 'store_')) {
                $statusCountQuery->where('store_id', str_replace('store_', '', $storeId));
            } elseif (str_starts_with($storeId, 'webhook_')) {
                $statusCountQuery->where('webhook_source_id', str_replace('webhook_', '', $storeId));
            } elseif (str_starts_with($storeId, 'salespage_')) {
                $salesPageId = str_replace('salespage_', '', $storeId);
                $statusCountQuery->where('is_manual', true)
                      ->where('created_by', function($subQuery) use ($salesPageId) {
                          $subQuery->select('user_id')
                                   ->from('sales_pages')
                                   ->where('id', $salesPageId)
                                   ->limit(1);
                      })
                      ->where('order_number', 'like', 'SP-%');
            } elseif (str_starts_with($storeId, 'checkout_user_')) {
                $userId = str_replace('checkout_user_', '', $storeId);
                $statusCountQuery->where('is_manual', true)->where('created_by', $userId);
            } elseif ($storeId === 'checkout') {
                $statusCountQuery->where('is_manual', true);
            } else {
                // Legacy support - assume it's a store ID
                $statusCountQuery->where('store_id', $storeId);
            }
        }

        $statusCounts = [
            'all' => $statusCountQuery->count(),
            'processing' => (clone $statusCountQuery)->where('status', 'processing')->count(),
            'approval' => (clone $statusCountQuery)->where('status', 'approval')->count(),
            'printed' => (clone $statusCountQuery)->where('status', 'printed')->count(),
            'completed' => (clone $statusCountQuery)->where('status', 'completed')->count(),
            'failed' => (clone $statusCountQuery)->where('status', 'failed')->count(),
        ];

        // Get filter options for dropdowns (cached with user context)
        $userKey = $user->isSeller() ? 'seller_' . $user->id : ($user->isManager() ? 'manager_' . $user->id : 'admin');
        $filterOptions = \Cache::remember("order_filter_options_{$userKey}", 3600, function() use ($user) {
            $storesQuery = \App\Models\Store::select('id', 'name')->orderBy('name');

            // Filter stores for sellers and managers
            if ($user->isSeller() || $user->isManager()) {
                $storeIds = $user->getStoreIds();
                $storesQuery->whereIn('id', $storeIds);
            }

            return [
                'statuses' => Order::distinct('status')->pluck('status')->sort()->values(),
                'payment_methods' => Order::distinct('payment_method')->pluck('payment_method')->sort()->values(),
                'stores' => $storesQuery->get(),
            ];
        });

        return view('orders.index', compact('orders', 'filterOptions', 'statusCounts', 'currentStatus'));
    }

    /**
     * Show the form for creating a new manual order
     */
    public function create()
    {
        $user = auth()->user();

        // Get seller's first store (for manual orders)
        $store = null;
        if ($user->isSeller()) {
            $storeIds = $user->getStoreIds();
            $store = \App\Models\Store::find($storeIds[0] ?? null);
        }

        return view('checkout.index', compact('store'));
    }

    /**
     * Store a new manual order
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'billing_full_name' => 'required|string|max:255',
            'billing_phone' => 'required|string|max:50',
            'billing_email' => 'nullable|email|max:255',
            'billing_address_1' => 'required|string|max:500',
            'billing_city' => 'required|string|max:100',
            'billing_postcode' => 'required|string|max:20',
            'billing_state' => 'required|string|max:100',
            'line_items' => 'required|array|min:1',
            'line_items.*.name' => 'required|string',
            'line_items.*.sku' => 'nullable|string',
            'line_items.*.quantity' => 'required|integer|min:1',
            'line_items.*.price' => 'required|numeric|min:0',
            'payment_method' => 'required|string',
            'shipping_total' => 'required|numeric|min:0',
            'discount_total' => 'nullable|numeric|min:0',
            'notes' => 'nullable|string',
        ]);

        // Split full name into first and last name for database compatibility
        $nameParts = explode(' ', $validated['billing_full_name'], 2);
        $firstName = $nameParts[0];
        $lastName = $nameParts[1] ?? '';

        // Build billing array
        $billing = [
            'first_name' => $firstName,
            'last_name' => $lastName,
            'phone' => $validated['billing_phone'],
            'email' => $validated['billing_email'] ?? '',
            'address_1' => $validated['billing_address_1'],
            'address_2' => '',
            'city' => $validated['billing_city'],
            'postcode' => $validated['billing_postcode'],
            'state' => $validated['billing_state'],
            'country' => 'MY',
        ];

        // Calculate line items totals
        $lineItems = collect($validated['line_items'])->map(function ($item) {
            $itemTotal = $item['quantity'] * $item['price'];
            return [
                'name' => $item['name'],
                'sku' => $item['sku'] ?? '',
                'quantity' => $item['quantity'],
                'price' => $item['price'],
                'total' => $itemTotal,
            ];
        })->toArray();

        // Calculate totals
        $subtotal = collect($lineItems)->sum('total');
        $shippingTotal = $validated['shipping_total'];
        $discountTotal = $validated['discount_total'] ?? 0;
        $total = $subtotal + $shippingTotal - $discountTotal;

        // Get payment method title
        $paymentMethodTitles = [
            'cod' => 'Cash on Delivery',
            'bayarcash-wc' => 'Online Banking',
            'duitnowqr-wc' => 'DuitNow QR',
            'bank_transfer' => 'Bank Transfer',
        ];
        $paymentMethodTitle = $paymentMethodTitles[$validated['payment_method']] ?? ucfirst(str_replace('_', ' ', $validated['payment_method']));

        // Get user's store
        $user = $request->user();
        $storeId = null;
        if ($user->isSeller()) {
            $storeIds = $user->getStoreIds();
            $storeId = $storeIds[0] ?? null;
        }

        // Generate checkout order number with seller code (CK-SELLER-001234)
        $orderNumber = $this->orderNumberService->generateCheckoutOrderNumber($user);

        // Format phone number for customer lookup/creation
        $formattedPhone = \App\Helpers\PhoneHelper::formatMalaysian($validated['billing_phone']);

        // Create or update customer record for auto-fill feature
        $customer = \App\Models\Customer::updateOrCreate(
            ['phone' => $formattedPhone],
            [
                'first_name' => $validated['billing_full_name'],
                'last_name' => '',
                'email' => $validated['billing_email'] ?? null,
                'address_1' => $validated['billing_address_1'],
                'city' => $validated['billing_city'],
                'state' => $validated['billing_state'],
                'postcode' => $validated['billing_postcode'],
                'country' => 'Malaysia',
                'username' => $formattedPhone, // Use phone as username
                'password' => bcrypt(\Illuminate\Support\Str::random(16)), // Random password (not used for checkout)
            ]
        );

        // Create manual order
        $order = Order::create([
            'store_id' => $storeId,
            'is_manual' => true,
            'created_by' => $user->id,
            'customer_id' => $customer->id, // Link to customer
            'order_number' => $orderNumber,
            // global_order_id will auto-generate as OMS00000001 from Order model
            'status' => 'processing',
            'total' => $total,
            'subtotal' => $subtotal,
            'shipping_total' => $shippingTotal,
            'discount_total' => $discountTotal,
            'currency' => 'MYR',
            'payment_method' => $validated['payment_method'],
            'payment_method_title' => $paymentMethodTitle,
            'billing' => $billing,
            'shipping' => $billing, // Use billing as shipping for manual orders
            'line_items' => $lineItems,
            'notes' => $validated['notes'],
            'date_created' => now(),
            'date_modified' => now(),
        ]);

        // Clear dashboard cache to show new order immediately
        $this->clearDashboardCache();

        return redirect()->route('orders.show', $order)
            ->with('success', 'Order #' . $order->order_number . ' created successfully!');
    }


    public function edit(Order $order)
    {
        // Check if seller/manager has access to this order
        $user = request()->user();
        if ($user->isSeller()) {
            $storeIds = $user->getStoreIds();
            $hasAccess = ($order->store_id && in_array($order->store_id, $storeIds)) || $order->created_by === $user->id;
            if (!$hasAccess) {
                abort(403, 'Unauthorized access to this order.');
            }
        } elseif ($user->isManager()) {
            $storeIds = $user->getStoreIds();
            $sellerIds = $user->getManagedSellerIds();
            $hasAccess = ($order->store_id && in_array($order->store_id, $storeIds)) || in_array($order->created_by, $sellerIds);
            if (!$hasAccess) {
                abort(403, 'Unauthorized access to this order.');
            }
        }

        // OPTIMIZATION: Load only 10 most recent logs to reduce memory usage and page load time
        $order->load([
            'store',
            'logs' => function($query) {
                $query->with('user')
                      ->orderBy('created_at', 'desc')
                      ->limit(10);
            }
        ]);

        return view('orders.edit', compact('order'));
    }

    public function update(Request $request, Order $order)
    {
        // Check if seller/manager has access to this order
        $user = $request->user();
        if ($user->isSeller()) {
            $storeIds = $user->getStoreIds();
            $hasAccess = ($order->store_id && in_array($order->store_id, $storeIds)) || $order->created_by === $user->id;
            if (!$hasAccess) {
                abort(403, 'Unauthorized access to this order.');
            }
        } elseif ($user->isManager()) {
            $storeIds = $user->getStoreIds();
            $sellerIds = $user->getManagedSellerIds();
            $hasAccess = ($order->store_id && in_array($order->store_id, $storeIds)) || in_array($order->created_by, $sellerIds);
            if (!$hasAccess) {
                abort(403, 'Unauthorized access to this order.');
            }
        }

        $validated = $request->validate([
            'status' => 'sometimes|string|in:processing,approval,pending,completed,cancelled,refunded',
            'billing.first_name' => 'sometimes|string|nullable',
            'billing.last_name' => 'sometimes|string|nullable',
            'billing.email' => 'sometimes|email|nullable',
            'billing.phone' => 'sometimes|string|nullable',
            'billing.address_1' => 'sometimes|string|nullable',
            'billing.city' => 'sometimes|string|nullable',
            'billing.postcode' => 'sometimes|string|nullable',
            'billing.state' => 'sometimes|string|nullable',
            'billing.country' => 'sometimes|string|nullable',
            'shipping.first_name' => 'sometimes|string|nullable',
            'shipping.last_name' => 'sometimes|string|nullable',
            'shipping.address_1' => 'sometimes|string|nullable',
            'shipping.city' => 'sometimes|string|nullable',
            'shipping.postcode' => 'sometimes|string|nullable',
            'shipping.state' => 'sometimes|string|nullable',
            'shipping.country' => 'sometimes|string|nullable',
        ]);

        try {
            // Log the update attempt
            \Log::info('Order update attempt', [
                'order_id' => $order->id,
                'user_id' => auth()->id(),
                'data' => $validated
            ]);

            $result = $this->syncService->updateOrder(
                $order,
                $validated,
                auth()->id()
            );

            if ($result) {
                // Order was updated successfully
                // The sync service handles WooCommerce sync internally and logs any issues
                $message = 'Order updated successfully!';

                // If this is a WooCommerce order, mention sync status
                if ($order->store && $order->woo_order_id) {
                    $message .= ' Changes will be synced to WooCommerce.';
                }

                \Log::info('Order update successful', [
                    'order_id' => $order->id,
                    'message' => $message
                ]);

                return redirect()->route('orders.edit', $order)
                    ->with('success', $message);
            }

            \Log::warning('Order update returned false', [
                'order_id' => $order->id
            ]);

            return back()->with('error', 'Failed to update order. Please try again.');
        } catch (\Exception $e) {
            \Log::error('Order update exception', [
                'order_id' => $order->id,
                'error' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);

            return back()->with('error', 'Error updating order: ' . $e->getMessage());
        }
    }

    public function sync(Request $request)
    {
        try {
            // Increase memory limit and set maximum execution time
            ini_set('memory_limit', '512M');
            set_time_limit(300); // 5 minutes max

            // Get all active stores
            $stores = \App\Models\Store::where('is_active', true)->get();
            $totalSynced = 0;
            $errors = [];
            $startTime = time();
            $maxExecutionTime = 240; // 4 minutes to allow for response time

            // Check if this is a full sync request
            $fullSync = $request->get('full_sync', false);
            $monthsBack = $request->get('months_back', 12);

            foreach ($stores as $store) {
                // Check if we're approaching time limit
                if ((time() - $startTime) > $maxExecutionTime) {
                    $errors[] = "Sync stopped due to time limit. Some stores may not have been synced.";
                    break;
                }

                try {
                    $service = new \App\Services\WooCommerceService($store);

                    if ($fullSync) {
                        $syncedCount = $service->syncHistoricalOrders($monthsBack);
                    } else {
                        $syncedCount = $service->syncOrders();
                    }

                    $totalSynced += $syncedCount;

                    // Log successful sync
                    \Log::info("Synced {$syncedCount} orders from {$store->name}");

                } catch (\Exception $e) {
                    $errors[] = "Failed to sync {$store->name}: " . $e->getMessage();
                    \Log::error("Sync failed for {$store->name}: " . $e->getMessage());
                }

                // Small delay to prevent API rate limiting
                usleep(500000); // 0.5 second delay between stores
            }

            $executionTime = time() - $startTime;
            $syncType = $fullSync ? "full sync ({$monthsBack} months, completed & processing)" : "recent sync (30 days, completed & processing)";

            if (count($errors) > 0) {
                $errorMessage = implode('; ', $errors);
                return back()->with('error', "Synced {$totalSynced} orders total ({$syncType}). Execution time: {$executionTime}s. Errors: {$errorMessage}");
            }

            return back()->with('success', "Successfully synced {$totalSynced} orders from " . count($stores) . " store(s) ({$syncType}). Execution time: {$executionTime}s.");

        } catch (\Exception $e) {
            \Log::error('Sync failed: ' . $e->getMessage());
            return back()->with('error', 'Sync failed: ' . $e->getMessage());
        }
    }

    public function syncHistorical(Request $request)
    {
        $request->merge(['full_sync' => true]);
        return $this->sync($request);
    }

    public function syncSingleStore(Request $request, $storeId)
    {
        try {
            // Increase memory limit and set maximum execution time
            ini_set('memory_limit', '256M');
            set_time_limit(120); // 2 minutes max for single store

            $store = \App\Models\Store::findOrFail($storeId);
            $user = $request->user();

            // Check access: Sellers and Managers can only access their assigned stores
            if (!$user->isAdmin() && !in_array($store->id, $user->getStoreIds())) {
                abort(403, 'You do not have access to this store.');
            }

            if (!$store->is_active) {
                return back()->with('error', "Store {$store->name} is not active.");
            }

            $startTime = time();
            $fullSync = $request->get('full_sync', false);
            $monthsBack = $request->get('months_back', 12);

            $service = new \App\Services\WooCommerceService($store);

            if ($fullSync) {
                $syncedCount = $service->syncHistoricalOrders($monthsBack);
            } else {
                $syncedCount = $service->syncOrders();
            }

            $executionTime = time() - $startTime;
            $syncType = $fullSync ? "full sync ({$monthsBack} months)" : "recent sync (30 days)";

            \Log::info("Single store sync: {$syncedCount} orders from {$store->name} in {$executionTime}s");

            return back()->with('success', "Successfully synced {$syncedCount} orders from {$store->name} ({$syncType}). Execution time: {$executionTime}s.");

        } catch (\Exception $e) {
            \Log::error("Single store sync failed for store {$storeId}: " . $e->getMessage());
            return back()->with('error', 'Single store sync failed: ' . $e->getMessage());
        }
    }

    public function syncSingleStoreHistorical(Request $request, $storeId)
    {
        try {
            // Much longer timeout for historical sync
            ini_set('memory_limit', '1G');
            set_time_limit(600); // 10 minutes for historical sync

            $store = \App\Models\Store::findOrFail($storeId);
            $user = $request->user();

            // Check access: Sellers and Managers can only access their assigned stores
            if (!$user->isAdmin() && !in_array($store->id, $user->getStoreIds())) {
                abort(403, 'You do not have access to this store.');
            }

            if (!$store->is_active) {
                return back()->with('error', "Store {$store->name} is not active.");
            }

            $startTime = time();
            $monthsBack = $request->get('months_back', 12);
            $maxExecutionTime = 540; // 9 minutes to allow for response

            $service = new \App\Services\WooCommerceService($store);

            // Check if we're approaching time limit during sync
            $syncedCount = $service->syncHistoricalOrdersWithTimeLimit($monthsBack, $maxExecutionTime, $startTime);

            $executionTime = time() - $startTime;

            \Log::info("Single store historical sync: {$syncedCount} orders from {$store->name} in {$executionTime}s");

            return back()->with('success', "Successfully synced {$syncedCount} orders from {$store->name} (historical {$monthsBack} months). Execution time: {$executionTime}s. Note: If more orders exist, run again to continue.");

        } catch (\Exception $e) {
            \Log::error("Single store historical sync failed for store {$storeId}: " . $e->getMessage());
            return back()->with('error', 'Single store historical sync failed: ' . $e->getMessage());
        }
    }

    /**
     * Major Sync: Sync 24 months of historical orders with ALL statuses
     * For migration/data recovery purposes
     */
    public function syncSingleStoreMajor(Request $request, $storeId)
    {
        try {
            // INCREASED: Extended timeout and memory for major sync
            ini_set('memory_limit', '3G'); // Increased from 2G
            ini_set('max_execution_time', '1800'); // 30 minutes
            set_time_limit(1800); // 30 minutes (was 15 minutes)

            $store = \App\Models\Store::findOrFail($storeId);
            $user = $request->user();

            // Check access: Sellers and Managers can only access their assigned stores
            if (!$user->isAdmin() && !in_array($store->id, $user->getStoreIds())) {
                abort(403, 'You do not have access to this store.');
            }

            if (!$store->is_active) {
                return back()->with('error', "Store {$store->name} is not active.");
            }

            $startTime = time();
            $monthsBack = 24; // Fixed to 24 months for major sync
            $maxExecutionTime = 1740; // 29 minutes to allow for response (was 14 minutes)

            $service = new \App\Services\WooCommerceService($store);

            \Log::info("Starting major sync for {$store->name}: 24 months, all statuses");

            // Sync with ALL statuses (including failed, pending, etc.)
            $syncedCount = $service->syncMajorHistoricalOrders($monthsBack, $maxExecutionTime, $startTime);

            $executionTime = time() - $startTime;

            \Log::info("Major sync completed: {$syncedCount} orders from {$store->name} in {$executionTime}s (24 months, all statuses)");

            if ($syncedCount == 0) {
                return back()->with('warning', "⚠️ No new orders were synced from {$store->name}. This could mean: 1) All orders already exist, 2) No orders in the selected period, or 3) API connection issue. Check logs for details.");
            }

            return back()->with('success', "✅ Major sync completed! Successfully synced {$syncedCount} orders from {$store->name} (24 months, all statuses). Execution time: " . gmdate('i:s', $executionTime) . ". WhatsApp notifications were automatically disabled for synced orders.");

        } catch (\Exception $e) {
            \Log::error("Major sync failed for store {$storeId}: " . $e->getMessage(), [
                'exception' => $e,
                'trace' => $e->getTraceAsString()
            ]);
            return back()->with('error', 'Major sync failed: ' . $e->getMessage() . ' - Check logs for details.');
        }
    }

    public function show(Order $order)
    {
        // SECURITY: Check if user has access to this order
        $this->authorizeOrderAccess($order);

        $order->load(['store', 'webhookSource']);
        return view('orders.show', compact('order'));
    }

    public function invoice(Order $order)
    {
        // SECURITY: Check if user has access to this order
        $this->authorizeOrderAccess($order);

        $order->load(['store', 'webhookSource', 'creator', 'creator.stores', 'salesPageOrder.payment']);
        return view('orders.invoice', compact('order'));
    }

    public function downloadInvoice(Order $order)
    {
        // SECURITY: Check if user has access to this order
        $this->authorizeOrderAccess($order);

        $order->load(['store', 'webhookSource', 'creator', 'creator.stores', 'salesPageOrder.payment']);

        $pdf = Pdf::loadView('orders.invoice-pdf', compact('order'));
        $pdf->setPaper('a4', 'portrait');

        $filename = 'Invoice-' . ($order->global_order_id ?? $order->order_number) . '.pdf';

        return $pdf->download($filename);
    }

    public function viewInvoicePdf(Order $order)
    {
        // SECURITY: Check if user has access to this order
        $this->authorizeOrderAccess($order);

        $order->load(['store', 'webhookSource', 'creator', 'creator.stores']);

        $pdf = Pdf::loadView('orders.invoice-pdf', compact('order'));
        $pdf->setPaper('a4', 'portrait');

        return $pdf->stream('Invoice-' . ($order->global_order_id ?? $order->order_number) . '.pdf');
    }

    /**
     * Print AWB (Air Waybill) in Pos Laju format (100mm x 150mm)
     */
    public function printAWB(Order $order)
    {
        // SECURITY: Check if user has access to this order
        $this->authorizeOrderAccess($order);

        if (!$order->tracking_number) {
            return back()->with('error', 'No tracking number available. Please generate AWB first.');
        }

        $order->load(['store', 'webhookSource', 'creator', 'creator.stores']);

        $pdf = Pdf::loadView('orders.awb-pdf', compact('order'));
        // Use custom paper size: 100mm x 150mm (Pos Laju standard label size)
        $pdf->setPaper([0, 0, 283.46, 425.20], 'portrait'); // 100mm x 150mm in points

        $filename = 'AWB-' . $order->tracking_number . '.pdf';

        // Update order status to 'printed' when AWB is printed
        if ($order->status === 'approval') {
            $order->update(['status' => 'printed']);
        }

        return $pdf->stream($filename);
    }

    /**
     * Download AWB (Air Waybill) in Pos Laju format (100mm x 150mm)
     */
    public function downloadAWB(Order $order)
    {
        // SECURITY: Check if user has access to this order
        $this->authorizeOrderAccess($order);

        if (!$order->tracking_number) {
            return back()->with('error', 'No tracking number available. Please generate AWB first.');
        }

        $order->load(['store', 'webhookSource']);

        $pdf = Pdf::loadView('orders.awb-pdf', compact('order'));
        // Use custom paper size: 100mm x 150mm (Pos Laju standard label size)
        $pdf->setPaper([0, 0, 283.46, 425.20], 'portrait'); // 100mm x 150mm in points

        $filename = 'AWB-' . $order->tracking_number . '.pdf';

        // Update order status to 'printed' when AWB is downloaded
        if ($order->status === 'approval') {
            $order->update(['status' => 'printed']);
        }

        return $pdf->download($filename);
    }

    /**
     * Generate Pos Laju AWB consignment number
     */
    public function generateAWB(Request $request, Order $order)
    {
        // SECURITY: Check if user has access to this order
        $this->authorizeOrderAccess($order);
        try {
            // Always use admin's Pos Laju credentials (centralized courier management)
            $adminUser = \App\Models\User::where('role', 'admin')->first();
            $poslajuService = new PoslajuService($adminUser->id ?? null);
            $result = $poslajuService->generateAWB($order);

            if ($result['success']) {
                // Update order with tracking information and move to Approval status
                $order->update([
                    'tracking_number' => $result['tracking_number'],
                    'courier' => 'Pos Laju',
                    'poslaju_response' => $result['data'],
                    'awb_generated_at' => now(),
                    'awb_status' => 'generated',
                    'status' => 'approval' // Move to Approval tab after AWB generated
                ]);

                // Sync tracking number back to WooCommerce
                $this->syncTrackingToWooCommerce($order);

                // Log the AWB generation
                \Log::info('AWB generated successfully', [
                    'order_id' => $order->id,
                    'tracking_number' => $result['tracking_number'],
                    'status_changed_to' => 'approval'
                ]);

                // Redirect to Approval tab with success message
                return redirect()->route('orders.index', ['status' => 'approval'])
                    ->with('success', 'AWB generated successfully! Order moved to Approval tab. Tracking: ' . $result['tracking_number']);
            }

            // Failed to generate AWB - redirect back with error
            return redirect()->route('orders.index')
                ->with('error', 'Failed to generate AWB: ' . $result['message']);

        } catch (\Exception $e) {
            \Log::error('AWB generation failed', [
                'order_id' => $order->id,
                'error' => $e->getMessage()
            ]);

            return redirect()->route('orders.index')
                ->with('error', 'Failed to generate AWB: ' . $e->getMessage());
        }
    }

    /**
     * Sync tracking number back to WooCommerce
     */
    protected function syncTrackingToWooCommerce(Order $order)
    {
        try {
            if (!$order->tracking_number || !$order->store) {
                return false;
            }

            $store = $order->store;
            $woocommerce = new \Automattic\WooCommerce\Client(
                $store->url,
                $store->consumer_key,
                $store->consumer_secret,
                [
                    'version' => 'wc/v3',
                    'timeout' => 30,
                ]
            );

            // Update order meta data with tracking information
            $woocommerce->put('orders/' . $order->woo_order_id, [
                'meta_data' => [
                    [
                        'key' => '_tracking_number',
                        'value' => $order->tracking_number
                    ],
                    [
                        'key' => '_courier',
                        'value' => $order->courier
                    ],
                    [
                        'key' => '_awb_generated_at',
                        'value' => $order->awb_generated_at->toDateTimeString()
                    ]
                ]
            ]);

            \Log::info('Tracking synced to WooCommerce', [
                'order_id' => $order->id,
                'woo_order_id' => $order->woo_order_id,
                'tracking_number' => $order->tracking_number
            ]);

            return true;

        } catch (\Exception $e) {
            \Log::error('Failed to sync tracking to WooCommerce', [
                'order_id' => $order->id,
                'error' => $e->getMessage()
            ]);

            return false;
        }
    }

    /**
     * Track Pos Laju shipment
     */
    public function trackShipment(Order $order)
    {
        // SECURITY: Check if user has access to this order
        $this->authorizeOrderAccess($order);

        try {
            if (!$order->tracking_number) {
                return response()->json([
                    'success' => false,
                    'message' => 'No tracking number available for this order'
                ], 404);
            }

            $poslajuService = new PoslajuService();
            $result = $poslajuService->trackShipment($order->tracking_number);

            return response()->json($result);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to track shipment: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Send WhatsApp notification manually
     */
    public function sendWhatsApp(Order $order)
    {
        // SECURITY: Check if user has access to this order
        $this->authorizeOrderAccess($order);

        try {
            // Check if order has tracking number
            if (!$order->tracking_number) {
                return redirect()->back()->with('error', 'Cannot send WhatsApp: Order does not have a tracking number yet.');
            }

            // Check if already sent
            if ($order->whatsapp_sent_at) {
                return redirect()->back()->with('warning', 'WhatsApp notification was already sent on ' . $order->whatsapp_sent_at->format('M d, Y H:i'));
            }

            // Determine which user's WhatsApp settings to use based on order source
            $notificationUserId = null;

            if ($order->store) {
                // WooCommerce store order: Use store owner's settings
                $notificationUserId = $order->store->user_id;
            } elseif ($order->webhookSource) {
                // Webhook/Channel order: Use webhook source owner's settings (if applicable)
                $notificationUserId = $order->webhookSource->user_id ?? null;
            } elseif ($order->is_manual && $order->created_by) {
                // Manual orders (Sales Pages / Checkout): Use creator's settings
                $notificationUserId = $order->created_by;
            }

            // Send WhatsApp notification using the appropriate user's credentials
            $whatsappService = new \App\Services\WhatsAppNotificationService($notificationUserId);
            $result = $whatsappService->sendOrderNotification($order, 'awb_generated');

            if ($result) {
                // Update whatsapp_sent_at timestamp
                $order->whatsapp_sent_at = now();
                $order->save();

                return redirect()->back()->with('success', 'WhatsApp notification sent successfully!');
            } else {
                return redirect()->back()->with('error', 'Failed to send WhatsApp notification. Please check your WhatsApp settings and logs.');
            }

        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Error sending WhatsApp: ' . $e->getMessage());
        }
    }

    /**
     * Mark order as completed
     */
    public function markAsCompleted(Order $order)
    {
        // SECURITY: Check if user has access to this order
        $this->authorizeOrderAccess($order);

        try {

            // Check if order is already completed
            if ($order->status === 'completed') {
                return redirect()->back()->with('warning', 'Order is already marked as completed.');
            }

            // Update order status to completed
            $order->status = 'completed';
            $order->date_modified = now();
            $order->save();

            // Sync status back to WooCommerce if this order came from a store
            if ($order->store_id && $order->store) {
                $this->syncCompletionStatusToWooCommerce($order);
            }

            // If order came from webhook source, send webhook notification back
            if ($order->webhook_source_id && $order->webhookSource) {
                $this->syncCompletionStatusToWebhook($order);
            }

            \Log::info('Order marked as completed', [
                'order_id' => $order->id,
                'order_number' => $order->order_number,
                'marked_by' => $user->id
            ]);

            return redirect()->back()->with('success', 'Order #' . $order->order_number . ' has been marked as completed.');

        } catch (\Exception $e) {
            \Log::error('Failed to mark order as completed', [
                'order_id' => $order->id,
                'error' => $e->getMessage()
            ]);

            return redirect()->back()->with('error', 'Failed to mark order as completed: ' . $e->getMessage());
        }
    }

    /**
     * Sync completion status back to WooCommerce
     */
    protected function syncCompletionStatusToWooCommerce(Order $order)
    {
        try {
            if (!$order->store || !$order->woo_order_id) {
                return false;
            }

            $store = $order->store;
            $woocommerce = new \Automattic\WooCommerce\Client(
                $store->url,
                $store->consumer_key,
                $store->consumer_secret,
                [
                    'version' => 'wc/v3',
                    'timeout' => 30,
                ]
            );

            // Update order status to completed in WooCommerce
            $woocommerce->put('orders/' . $order->woo_order_id, [
                'status' => 'completed'
            ]);

            \Log::info('Order completion status synced to WooCommerce', [
                'order_id' => $order->id,
                'woo_order_id' => $order->woo_order_id,
                'store' => $store->name
            ]);

            return true;

        } catch (\Exception $e) {
            \Log::error('Failed to sync completion status to WooCommerce', [
                'order_id' => $order->id,
                'error' => $e->getMessage()
            ]);

            return false;
        }
    }

    /**
     * Sync completion status back to webhook source
     */
    protected function syncCompletionStatusToWebhook(Order $order)
    {
        try {
            if (!$order->webhookSource || !$order->webhookSource->webhook_url) {
                return false;
            }

            $webhookSource = $order->webhookSource;

            // Prepare webhook payload
            $payload = [
                'event' => 'order.completed',
                'order_id' => $order->id,
                'order_number' => $order->order_number,
                'global_order_id' => $order->global_order_id,
                'status' => 'completed',
                'completed_at' => now()->toIso8601String(),
            ];

            // Send webhook
            $client = new \GuzzleHttp\Client();
            $response = $client->post($webhookSource->webhook_url, [
                'json' => $payload,
                'timeout' => 10,
                'headers' => [
                    'Content-Type' => 'application/json',
                    'X-Webhook-Source' => 'omnia',
                ]
            ]);

            \Log::info('Order completion status sent to webhook source', [
                'order_id' => $order->id,
                'webhook_source' => $webhookSource->name,
                'response_code' => $response->getStatusCode()
            ]);

            return true;

        } catch (\Exception $e) {
            \Log::error('Failed to send completion status to webhook source', [
                'order_id' => $order->id,
                'error' => $e->getMessage()
            ]);

            return false;
        }
    }

    /**
     * Delete an order
     */
    public function destroy(Order $order)
    {
        try {
            // Check if seller/manager has access to this order
            $user = request()->user();
            if ($user->isSeller()) {
                $storeIds = $user->getStoreIds();
                $hasAccess = ($order->store_id && in_array($order->store_id, $storeIds)) || $order->created_by === $user->id;
                if (!$hasAccess) {
                    abort(403, 'Unauthorized access to this order.');
                }
            } elseif ($user->isManager()) {
                $storeIds = $user->getStoreIds();
                $sellerIds = $user->getManagedSellerIds();
                $hasAccess = ($order->store_id && in_array($order->store_id, $storeIds)) || in_array($order->created_by, $sellerIds);
                if (!$hasAccess) {
                    abort(403, 'Unauthorized access to this order.');
                }
            }

            $orderNumber = $order->order_number;

            // Delete order logs first (foreign key constraint)
            $order->logs()->delete();

            // Delete the order
            $order->delete();

            \Log::info('Order deleted', [
                'order_id' => $order->id,
                'order_number' => $orderNumber,
                'deleted_by' => $user->id
            ]);

            return redirect()->route('orders.index')
                ->with('success', "Order #{$orderNumber} has been deleted successfully.");

        } catch (\Exception $e) {
            \Log::error('Order deletion failed', [
                'order_id' => $order->id,
                'error' => $e->getMessage()
            ]);

            return redirect()->back()->with('error', 'Failed to delete order: ' . $e->getMessage());
        }
    }

    /**
     * Bulk Push AWB
     */
    public function bulkPushAwb(Request $request)
    {
        $request->validate([
            'order_ids' => 'required|array|max:50',
            'order_ids.*' => 'required|exists:orders,id'
        ]);

        $orderIds = $request->order_ids;
        $orders = Order::whereIn('id', $orderIds)->get();

        $successCount = 0;
        $skippedCount = 0;
        $failedOrders = [];
        $skippedOrders = [];

        // Always use admin's Pos Laju credentials (centralized courier management)
        $adminUser = \App\Models\User::where('role', 'admin')->first();
        $poslajuService = new PoslajuService($adminUser->id ?? null);

        foreach ($orders as $order) {
            // Skip if order already has tracking number
            if ($order->tracking_number) {
                // But still migrate status if stuck in processing
                if ($order->status === 'processing') {
                    $order->update(['status' => 'approval']);
                    $successCount++;
                    \Log::info('Order migrated from processing to approval (already had tracking)', [
                        'order_id' => $order->id,
                        'order_number' => $order->order_number,
                        'tracking_number' => $order->tracking_number
                    ]);
                } else {
                    $skippedCount++;
                    $skippedOrders[] = $order->order_number . ' (already has tracking)';
                }
                continue;
            }

            try {
                $result = $poslajuService->generateAWB($order);

                if ($result['success']) {
                    // Update order with tracking information
                    $order->update([
                        'tracking_number' => $result['tracking_number'],
                        'courier' => 'Pos Laju',
                        'poslaju_response' => $result['data'],
                        'awb_generated_at' => now(),
                        'awb_status' => 'generated',
                        'status' => 'approval'
                    ]);

                    // Sync tracking number back to WooCommerce
                    $this->syncTrackingToWooCommerce($order);

                    $successCount++;
                } else {
                    $failedOrders[] = $order->order_number . ' (' . $result['message'] . ')';
                }
            } catch (\Exception $e) {
                $failedOrders[] = $order->order_number . ' (' . $e->getMessage() . ')';
            }
        }

        // Build success message
        $message = "Bulk AWB Push completed: {$successCount} successful";

        if ($skippedCount > 0) {
            $message .= ", {$skippedCount} skipped";
        }

        if (count($failedOrders) > 0) {
            $message .= ". Failed orders: " . implode(', ', $failedOrders);
        }

        if (count($skippedOrders) > 0) {
            $message .= ". Skipped orders: " . implode(', ', $skippedOrders);
        }

        if ($successCount > 0) {
            return redirect()->route('orders.index')->with('success', $message);
        } else {
            return redirect()->route('orders.index')->with('error', $message);
        }
    }

    /**
     * Bulk Download AWB (Merged PDF)
     */
    public function bulkDownloadAwb(Request $request)
    {
        $request->validate([
            'order_ids' => 'required|array|max:50',
            'order_ids.*' => 'required|exists:orders,id'
        ]);

        $orderIds = $request->order_ids;
        $orders = Order::whereIn('id', $orderIds)
            ->with(['store', 'webhookSource'])
            ->whereNotNull('tracking_number')
            ->get();

        if ($orders->isEmpty()) {
            return redirect()->route('orders.index')->with('error', 'No orders with tracking numbers found.');
        }

        try {
            $mergedPdf = $this->mergePdfs($orders, 'awb');

            $filename = 'Bulk-AWB-' . now()->format('Y-m-d-His') . '.pdf';

            // Update all downloaded orders to 'printed' status
            foreach ($orders as $order) {
                if ($order->status === 'approval') {
                    $order->update(['status' => 'printed']);
                }
            }

            return response($mergedPdf, 200)
                ->header('Content-Type', 'application/pdf')
                ->header('Content-Disposition', 'attachment; filename="' . $filename . '"');

        } catch (\Exception $e) {
            \Log::error('Bulk AWB download failed: ' . $e->getMessage());
            return redirect()->route('orders.index')->with('error', 'Failed to generate merged AWB PDF: ' . $e->getMessage());
        }
    }

    /**
     * Bulk Download Invoice (Merged PDF)
     */
    public function bulkDownloadInvoice(Request $request)
    {
        $request->validate([
            'order_ids' => 'required|array|max:50',
            'order_ids.*' => 'required|exists:orders,id'
        ]);

        $orderIds = $request->order_ids;
        $orders = Order::whereIn('id', $orderIds)
            ->with(['store', 'webhookSource'])
            ->get();

        if ($orders->isEmpty()) {
            return redirect()->route('orders.index')->with('error', 'No orders found.');
        }

        try {
            $mergedPdf = $this->mergePdfs($orders, 'invoice');

            $filename = 'Bulk-Invoices-' . now()->format('Y-m-d-His') . '.pdf';

            return response($mergedPdf, 200)
                ->header('Content-Type', 'application/pdf')
                ->header('Content-Disposition', 'attachment; filename="' . $filename . '"');

        } catch (\Exception $e) {
            \Log::error('Bulk invoice download failed: ' . $e->getMessage());
            return redirect()->route('orders.index')->with('error', 'Failed to generate merged invoice PDF: ' . $e->getMessage());
        }
    }

    /**
     * Bulk Mark as Completed
     */
    public function bulkMarkCompleted(Request $request)
    {
        $request->validate([
            'order_ids' => 'required|array|max:50',
            'order_ids.*' => 'required|exists:orders,id'
        ]);

        $orderIds = $request->order_ids;
        $orders = Order::whereIn('id', $orderIds)->get();

        $successCount = 0;
        $skippedCount = 0;
        $failedOrders = [];
        $skippedOrders = [];

        foreach ($orders as $order) {
            // Skip if already completed
            if ($order->status === 'completed') {
                $skippedCount++;
                $skippedOrders[] = $order->order_number . ' (already completed)';
                continue;
            }

            try {
                // Update order status to completed
                $order->status = 'completed';
                $order->date_modified = now();
                $order->save();

                // Sync status back to WooCommerce if this order came from a store
                if ($order->store_id && $order->store) {
                    $this->syncCompletionStatusToWooCommerce($order);
                }

                // If order came from webhook source, send webhook notification back
                if ($order->webhook_source_id && $order->webhookSource) {
                    $this->syncCompletionStatusToWebhook($order);
                }

                $successCount++;

            } catch (\Exception $e) {
                $failedOrders[] = $order->order_number . ' (' . $e->getMessage() . ')';
            }
        }

        // Build success message
        $message = "Bulk Mark Completed: {$successCount} orders marked as completed";

        if ($skippedCount > 0) {
            $message .= ", {$skippedCount} skipped";
        }

        if (count($failedOrders) > 0) {
            $message .= ". Failed orders: " . implode(', ', $failedOrders);
        }

        if (count($skippedOrders) > 0) {
            $message .= ". Skipped orders: " . implode(', ', $skippedOrders);
        }

        if ($successCount > 0) {
            return redirect()->route('orders.index')->with('success', $message);
        } else {
            return redirect()->route('orders.index')->with('error', $message);
        }
    }

    /**
     * Merge multiple PDFs into one
     *
     * @param \Illuminate\Support\Collection $orders
     * @param string $type 'awb' or 'invoice'
     * @return string PDF content
     */
    protected function mergePdfs($orders, $type = 'invoice')
    {
        $fpdi = new Fpdi();

        foreach ($orders as $order) {
            // Generate individual PDF
            if ($type === 'awb') {
                $pdf = Pdf::loadView('orders.awb-pdf', compact('order'));
                $pdf->setPaper([0, 0, 283.46, 425.20], 'portrait'); // 100mm x 150mm
            } else {
                $pdf = Pdf::loadView('orders.invoice-pdf', compact('order'));
                $pdf->setPaper('a4', 'portrait');
            }

            // Get PDF content as string
            $pdfContent = $pdf->output();

            // Save to temporary file
            $tempFile = tempnam(sys_get_temp_dir(), 'pdf_');
            file_put_contents($tempFile, $pdfContent);

            // Get page count
            $pageCount = $fpdi->setSourceFile($tempFile);

            // Import each page
            for ($pageNo = 1; $pageNo <= $pageCount; $pageNo++) {
                // Add a page
                if ($type === 'awb') {
                    $fpdi->AddPage('P', [100, 150]); // 100mm x 150mm
                } else {
                    $fpdi->AddPage('P', 'A4');
                }

                // Import page
                $templateId = $fpdi->importPage($pageNo);

                // Use the imported page
                $fpdi->useTemplate($templateId, 0, 0, null, null, true);
            }

            // Clean up temporary file
            unlink($tempFile);
        }

        // Return merged PDF
        return $fpdi->Output('S'); // 'S' returns the document as a string
    }

    /**
     * Clear dashboard cache for all users to show new orders immediately
     */
    private function clearDashboardCache()
    {
        // Clear all dashboard-related cache keys
        $patterns = [
            'dashboard_period_stats_*',
            'dashboard_today_stats_*'
        ];

        foreach ($patterns as $pattern) {
            // Get all cache keys matching pattern and delete them
            // Since Laravel's file cache doesn't support wildcard delete natively,
            // we'll use a simple approach to clear common user types
            $userTypes = ['all', 'admin'];

            // Add seller cache keys
            $sellers = \App\Models\User::where('role', 'seller')->pluck('id');
            foreach ($sellers as $sellerId) {
                $userTypes[] = 'seller_' . $sellerId;
            }

            // Add manager cache keys
            $managers = \App\Models\User::where('role', 'manager')->pluck('id');
            foreach ($managers as $managerId) {
                $userTypes[] = 'manager_' . $managerId;
            }

            // Clear cache for each user type and time combination
            foreach ($userTypes as $userType) {
                // Clear stats for different time ranges (today's hour, recent dates)
                for ($i = 0; $i < 24; $i++) {
                    $time = now()->subHours($i);

                    // Today stats
                    \Cache::forget("dashboard_today_stats_{$userType}_" . $time->format('Y-m-d_H'));

                    // Period stats - common periods
                    $periods = [
                        ['start' => $time->copy()->startOfDay(), 'end' => $time->copy()->endOfDay()],
                        ['start' => $time->copy()->startOfWeek(), 'end' => $time->copy()->endOfWeek()],
                        ['start' => $time->copy()->startOfMonth(), 'end' => $time->copy()->endOfMonth()],
                    ];

                    foreach ($periods as $period) {
                        \Cache::forget("dashboard_period_stats_{$userType}_{$period['start']->format('Y-m-d_H-i')}_{$period['end']->format('Y-m-d_H-i')}");
                    }
                }
            }
        }
    }

    /**
     * SECURITY: Check if the current user has permission to access this order
     * Admins can access all orders
     * Sellers can only access orders from their stores or orders they created
     * Managers can access orders from their managed stores or created by their managed sellers
     *
     * @param Order $order
     * @return void
     * @throws \Symfony\Component\HttpKernel\Exception\HttpException
     */
    protected function authorizeOrderAccess(Order $order)
    {
        $user = request()->user();

        // Admin has access to all orders
        if ($user->isAdmin()) {
            return;
        }

        // Check seller access
        if ($user->isSeller()) {
            $storeIds = $user->getStoreIds();
            $hasAccess = ($order->store_id && in_array($order->store_id, $storeIds)) || $order->created_by === $user->id;

            if (!$hasAccess) {
                abort(403, 'Unauthorized access to this order.');
            }
            return;
        }

        // Check manager access
        if ($user->isManager()) {
            $storeIds = $user->getStoreIds();
            $sellerIds = $user->getManagedSellerIds();
            $hasAccess = ($order->store_id && in_array($order->store_id, $storeIds)) || in_array($order->created_by, $sellerIds);

            if (!$hasAccess) {
                abort(403, 'Unauthorized access to this order.');
            }
            return;
        }

        // Default deny for any other role
        abort(403, 'Unauthorized access to this order.');
    }
}