<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\MarketingStrategyInput;
use App\Models\HistoricalSale;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Http;
use Carbon\Carbon;

class MarketingStrategyController extends Controller
{
    /**
     * Display marketing strategy dashboard
     */
    public function index(Request $request)
    {
        $year = $request->input('year', date('Y'));

        // Get all marketing data for the year
        $marketingData = MarketingStrategyInput::forYear($year)
            ->orderBy('month')
            ->get();

        // Get HQ revenue data for each month
        $historicalSales = HistoricalSale::where('year', $year)->get()->keyBy('month');

        // Enrich marketing data with HQ-specific metrics
        $marketingDataWithHq = $marketingData->map(function($item) use ($historicalSales) {
            $historicalSale = $historicalSales->get($item->month);

            $item->hq_revenue = $historicalSale ? $historicalSale->hq_sales : 0;
            $item->stokis_sales = $historicalSale ? $historicalSale->stokis_sales : 0;
            $item->agent_sales = $historicalSale ? $historicalSale->agent_sales : 0;

            // Calculate HQ-only ROI
            if ($item->total_marketing_spend > 0) {
                $hqProfit = $item->hq_revenue - $item->total_marketing_spend;
                $item->hq_roi_percentage = ($hqProfit / $item->total_marketing_spend) * 100;
            } else {
                $item->hq_roi_percentage = 0;
            }

            return $item;
        });

        // Calculate totals
        $totalHqRevenue = $marketingDataWithHq->sum('hq_revenue');
        $totalStokisSales = $marketingDataWithHq->sum('stokis_sales');
        $totalAgentSales = $marketingDataWithHq->sum('agent_sales');
        $totalSpend = $marketingDataWithHq->sum('total_marketing_spend');
        $totalRevenue = $marketingDataWithHq->sum('actual_revenue');

        // Calculate average ROIs
        $avgRoi = $marketingDataWithHq->avg('roi_percentage');
        $avgHqRoi = $marketingDataWithHq->avg('hq_roi_percentage');

        // Find best/worst months for both All Sales and HQ Only
        $bestMonthAll = $marketingDataWithHq->sortByDesc('roi_percentage')->first();
        $worstMonthAll = $marketingDataWithHq->where('roi_percentage', '>', 0)->sortBy('roi_percentage')->first();

        $bestMonthHq = $marketingDataWithHq->sortByDesc('hq_roi_percentage')->first();
        $worstMonthHq = $marketingDataWithHq->where('hq_roi_percentage', '>', 0)->sortBy('hq_roi_percentage')->first();

        $yearSummary = [
            'total_spend' => $totalSpend,
            'total_revenue' => $totalRevenue,
            'hq_revenue' => $totalHqRevenue,
            'stokis_sales' => $totalStokisSales,
            'agent_sales' => $totalAgentSales,
            'avg_roi' => $avgRoi,
            'hq_roi' => $avgHqRoi,
            'best_month' => $bestMonthAll,
            'worst_month' => $worstMonthAll,
            'best_month_hq' => $bestMonthHq,
            'worst_month_hq' => $worstMonthHq,
            'channel_performance' => $this->getChannelPerformance($marketingDataWithHq),
        ];

        return view('forecasting.marketing-strategy.index', compact('marketingDataWithHq', 'yearSummary', 'year'));
    }

    /**
     * Show marketing input form
     */
    public function showInput(Request $request)
    {
        $year = $request->input('year', date('Y'));
        $month = $request->input('month', date('n'));
        $periodType = $request->input('period_type', 'monthly');

        if ($periodType === 'yearly') {
            // Yearly analysis - aggregate all months
            $existingData = MarketingStrategyInput::forYear($year)->get();

            // Get historical sales for HQ revenue calculation
            $historicalSales = HistoricalSale::where('year', $year)->get();
            $totalHqRevenue = $historicalSales->sum('hq_sales');
            $totalStokisSales = $historicalSales->sum('stokis_sales');
            $totalAgentSales = $historicalSales->sum('agent_sales');

            // Create aggregated object for yearly view
            $aggregated = (object)[
                'year' => $year,
                'month' => null,
                'total_marketing_spend' => $existingData->sum('total_marketing_spend'),
                'brand_ambassador_spend' => $existingData->sum('brand_ambassador_spend'),
                'ads_spend' => $existingData->sum('ads_spend'),
                'social_media_spend' => $existingData->sum('social_media_spend'),
                'events_spend' => $existingData->sum('events_spend'),
                'other_spend' => $existingData->sum('other_spend'),
                'actual_revenue' => $existingData->sum('actual_revenue'),
                'roi_percentage' => $existingData->avg('roi_percentage'),
                'campaign_details' => $existingData->pluck('campaign_details')->filter()->implode("\n\n"),
                'target_audience' => $existingData->pluck('target_audience')->filter()->unique()->implode(', '),
                'channels_used' => $existingData->pluck('channels_used')->filter()->unique()->implode(', '),
                'ai_analysis' => null,
                'ai_recommendations' => null,
                'is_yearly' => true,
                'months_count' => $existingData->count(),
            ];

            $actualRevenue = $aggregated->actual_revenue;
            $hqRevenue = $totalHqRevenue;
            $stokisSales = $totalStokisSales;
            $agentSales = $totalAgentSales;

            return view('forecasting.marketing-strategy.input', compact('aggregated', 'actualRevenue', 'hqRevenue', 'stokisSales', 'agentSales', 'year', 'month', 'periodType'))->with('existingData', $aggregated);
        } else {
            // Monthly analysis
            $existingData = MarketingStrategyInput::forPeriod($year, $month)->first();

            // Get actual revenue from historical_sales
            $historicalSale = HistoricalSale::where('year', $year)
                ->where('month', $month)
                ->first();

            $actualRevenue = $historicalSale ? $historicalSale->revenue : 0;
            $hqRevenue = $historicalSale ? $historicalSale->hq_sales : 0;
            $stokisSales = $historicalSale ? $historicalSale->stokis_sales : 0;
            $agentSales = $historicalSale ? $historicalSale->agent_sales : 0;

            return view('forecasting.marketing-strategy.input', compact('existingData', 'actualRevenue', 'hqRevenue', 'stokisSales', 'agentSales', 'year', 'month', 'periodType'));
        }
    }

    /**
     * Save marketing strategy input
     */
    public function saveInput(Request $request)
    {
        $request->validate([
            'year' => 'required|integer',
            'month' => 'required|integer|between:1,12',
            'brand_ambassador_spend' => 'required|numeric|min:0',
            'ads_spend' => 'required|numeric|min:0',
            'social_media_spend' => 'required|numeric|min:0',
            'events_spend' => 'required|numeric|min:0',
            'other_spend' => 'required|numeric|min:0',
            'campaign_details' => 'nullable|string',
            'target_audience' => 'nullable|string',
            'channels_used' => 'nullable|string',
            // Ads breakdown
            'ads_facebook' => 'nullable|numeric|min:0',
            'ads_instagram' => 'nullable|numeric|min:0',
            'ads_tiktok' => 'nullable|numeric|min:0',
            'ads_google' => 'nullable|numeric|min:0',
            'ads_youtube' => 'nullable|numeric|min:0',
            'ads_twitter' => 'nullable|numeric|min:0',
            'ads_linkedin' => 'nullable|numeric|min:0',
            'ads_other' => 'nullable|numeric|min:0',
            // Social media breakdown
            'social_facebook' => 'nullable|numeric|min:0',
            'social_instagram' => 'nullable|numeric|min:0',
            'social_tiktok' => 'nullable|numeric|min:0',
            'social_youtube' => 'nullable|numeric|min:0',
            'social_twitter' => 'nullable|numeric|min:0',
            'social_linkedin' => 'nullable|numeric|min:0',
            'social_other' => 'nullable|numeric|min:0',
            // Revenue breakdown
            'revenue_checkout_form' => 'nullable|numeric|min:0',
            'revenue_sales_pages' => 'nullable|numeric|min:0',
            'revenue_webstore' => 'nullable|numeric|min:0',
            'revenue_marketplace' => 'nullable|numeric|min:0',
            'revenue_direct' => 'nullable|numeric|min:0',
            'revenue_other' => 'nullable|numeric|min:0',
            // Conversions
            'conversions_checkout_form' => 'nullable|integer|min:0',
            'conversions_sales_pages' => 'nullable|integer|min:0',
            'conversions_webstore' => 'nullable|integer|min:0',
            // Channel notes
            'channel_notes' => 'nullable|string',
        ]);

        // Calculate total marketing spend
        $totalSpend = $request->brand_ambassador_spend
                    + $request->ads_spend
                    + $request->social_media_spend
                    + $request->events_spend
                    + $request->other_spend;

        // Get actual revenue from historical_sales
        $historicalSale = HistoricalSale::where('year', $request->year)
            ->where('month', $request->month)
            ->first();

        $actualRevenue = $historicalSale ? $historicalSale->revenue : 0;

        // Calculate ROI
        $roi = 0;
        if ($totalSpend > 0) {
            $profit = $actualRevenue - $totalSpend;
            $roi = ($profit / $totalSpend) * 100;
        }

        // Create or update record
        $marketingInput = MarketingStrategyInput::updateOrCreate(
            [
                'year' => $request->year,
                'month' => $request->month,
            ],
            [
                'total_marketing_spend' => $totalSpend,
                'brand_ambassador_spend' => $request->brand_ambassador_spend,
                'ads_spend' => $request->ads_spend,
                'social_media_spend' => $request->social_media_spend,
                'events_spend' => $request->events_spend,
                'other_spend' => $request->other_spend,
                'campaign_details' => $request->campaign_details,
                'target_audience' => $request->target_audience,
                'channels_used' => $request->channels_used,
                // Ads breakdown
                'ads_facebook' => $request->ads_facebook ?? 0,
                'ads_instagram' => $request->ads_instagram ?? 0,
                'ads_tiktok' => $request->ads_tiktok ?? 0,
                'ads_google' => $request->ads_google ?? 0,
                'ads_youtube' => $request->ads_youtube ?? 0,
                'ads_twitter' => $request->ads_twitter ?? 0,
                'ads_linkedin' => $request->ads_linkedin ?? 0,
                'ads_other' => $request->ads_other ?? 0,
                // Social media breakdown
                'social_facebook' => $request->social_facebook ?? 0,
                'social_instagram' => $request->social_instagram ?? 0,
                'social_tiktok' => $request->social_tiktok ?? 0,
                'social_youtube' => $request->social_youtube ?? 0,
                'social_twitter' => $request->social_twitter ?? 0,
                'social_linkedin' => $request->social_linkedin ?? 0,
                'social_other' => $request->social_other ?? 0,
                // Revenue breakdown
                'revenue_checkout_form' => $request->revenue_checkout_form ?? 0,
                'revenue_sales_pages' => $request->revenue_sales_pages ?? 0,
                'revenue_webstore' => $request->revenue_webstore ?? 0,
                'revenue_marketplace' => $request->revenue_marketplace ?? 0,
                'revenue_direct' => $request->revenue_direct ?? 0,
                'revenue_other' => $request->revenue_other ?? 0,
                // Conversions
                'conversions_checkout_form' => $request->conversions_checkout_form ?? 0,
                'conversions_sales_pages' => $request->conversions_sales_pages ?? 0,
                'conversions_webstore' => $request->conversions_webstore ?? 0,
                // Channel notes
                'channel_notes' => $request->channel_notes,
                'actual_revenue' => $actualRevenue,
                'roi_percentage' => $roi,
                'created_by' => $marketingInput->created_by ?? Auth::id(),
                'updated_by' => Auth::id(),
            ]
        );

        return redirect()
            ->route('forecasting.marketing-strategy.input', ['year' => $request->year, 'month' => $request->month])
            ->with('success', "Marketing data saved! Total spend: RM " . number_format($totalSpend, 2) . " | ROI: " . number_format($roi, 2) . "%");
    }

    /**
     * Get AI suggestions for marketing strategy
     */
    public function getAiSuggestions(Request $request)
    {
        $year = $request->input('year', date('Y'));
        $month = $request->input('month', date('n'));
        $periodType = $request->input('period_type', 'monthly');
        $revenueMode = $request->input('revenue_mode', 'all_sales'); // all_sales or hq_only

        if ($periodType === 'yearly') {
            // Yearly strategic analysis
            return $this->getYearlyAiSuggestions($year, $revenueMode);
        }

        // Get current month's data
        $currentData = MarketingStrategyInput::forPeriod($year, $month)->first();

        if (!$currentData) {
            return response()->json([
                'success' => false,
                'message' => 'No marketing data found for this period. Please save marketing data first.'
            ], 404);
        }

        // Get previous year same month for YoY comparison
        $previousYearData = MarketingStrategyInput::forPeriod($year - 1, $month)->first();

        // Get all historical data for pattern analysis
        $historicalData = MarketingStrategyInput::where('year', '<=', $year)
            ->orderBy('year')
            ->orderBy('month')
            ->get();

        // Build AI prompt
        $prompt = $this->buildAiPrompt($currentData, $previousYearData, $historicalData);

        // Call AI service
        $aiProvider = env('AI_PROVIDER', 'ollama'); // chatgpt or ollama

        if ($aiProvider === 'chatgpt') {
            $suggestions = $this->getChatGptSuggestions($prompt);
        } else {
            $suggestions = $this->getOllamaSuggestions($prompt);
        }

        // Save AI suggestions to database
        $currentData->ai_analysis = $suggestions['analysis'] ?? '';
        $currentData->ai_recommendations = $suggestions['recommendations'] ?? '';
        $currentData->save();

        return response()->json([
            'success' => true,
            'suggestions' => $suggestions
        ]);
    }

    /**
     * Get yearly strategic AI suggestions
     */
    private function getYearlyAiSuggestions($year, $revenueMode = 'all_sales')
    {
        // Get all data for the year
        $yearData = MarketingStrategyInput::forYear($year)->orderBy('month')->get();

        if ($yearData->isEmpty()) {
            return response()->json([
                'success' => false,
                'message' => 'No marketing data found for this year. Please add monthly data first.'
            ], 404);
        }

        // Get HQ revenue data if in HQ mode
        $historicalSales = HistoricalSale::where('year', $year)->get();
        $totalHqRevenue = $historicalSales->sum('hq_sales');
        $totalStokisSales = $historicalSales->sum('stokis_sales');
        $totalAgentSales = $historicalSales->sum('agent_sales');

        // Get previous year data for comparison
        $previousYearData = MarketingStrategyInput::forYear($year - 1)->orderBy('month')->get();
        $previousYearHistoricalSales = HistoricalSale::where('year', $year - 1)->get();

        // Build yearly strategic prompt based on revenue mode
        $prompt = $this->buildYearlyAiPrompt($yearData, $previousYearData, $year, $revenueMode, [
            'hq_revenue' => $totalHqRevenue,
            'stokis_sales' => $totalStokisSales,
            'agent_sales' => $totalAgentSales,
            'previous_hq_revenue' => $previousYearHistoricalSales->sum('hq_sales'),
        ]);

        // Call AI service
        $aiProvider = env('AI_PROVIDER', 'ollama');

        if ($aiProvider === 'chatgpt') {
            $suggestions = $this->getChatGptSuggestions($prompt, true); // true = yearly/strategic
        } else {
            $suggestions = $this->getOllamaSuggestions($prompt);
        }

        return response()->json([
            'success' => true,
            'suggestions' => $suggestions,
            'yearly' => true,
            'revenue_mode' => $revenueMode
        ]);
    }

    /**
     * Build AI prompt for suggestions
     */
    private function buildAiPrompt($currentData, $previousYearData, $historicalData)
    {
        $prompt = "You are a marketing strategy expert analyzing sales and marketing data.\n\n";

        // Current month data
        $prompt .= "CURRENT MONTH ({$currentData->month_name} {$currentData->year}):\n";
        $prompt .= "- Total Marketing Spend: RM " . number_format($currentData->total_marketing_spend, 2) . "\n";
        $prompt .= "  - Brand Ambassadors: RM " . number_format($currentData->brand_ambassador_spend, 2) . "\n";
        $prompt .= "  - Ads: RM " . number_format($currentData->ads_spend, 2) . "\n";
        $prompt .= "  - Social Media: RM " . number_format($currentData->social_media_spend, 2) . "\n";
        $prompt .= "  - Events: RM " . number_format($currentData->events_spend, 2) . "\n";

        // Ads breakdown by platform
        if ($currentData->ads_spend > 0) {
            $prompt .= "\nPAID ADS BREAKDOWN:\n";
            if ($currentData->ads_facebook > 0) $prompt .= "  - Facebook Ads: RM " . number_format($currentData->ads_facebook, 2) . "\n";
            if ($currentData->ads_instagram > 0) $prompt .= "  - Instagram Ads: RM " . number_format($currentData->ads_instagram, 2) . "\n";
            if ($currentData->ads_tiktok > 0) $prompt .= "  - TikTok Ads: RM " . number_format($currentData->ads_tiktok, 2) . "\n";
            if ($currentData->ads_google > 0) $prompt .= "  - Google Ads: RM " . number_format($currentData->ads_google, 2) . "\n";
            if ($currentData->ads_youtube > 0) $prompt .= "  - YouTube Ads: RM " . number_format($currentData->ads_youtube, 2) . "\n";
            if ($currentData->ads_twitter > 0) $prompt .= "  - Twitter Ads: RM " . number_format($currentData->ads_twitter, 2) . "\n";
            if ($currentData->ads_linkedin > 0) $prompt .= "  - LinkedIn Ads: RM " . number_format($currentData->ads_linkedin, 2) . "\n";
            if ($currentData->ads_other > 0) $prompt .= "  - Other Ads: RM " . number_format($currentData->ads_other, 2) . "\n";
        }

        // Social media breakdown by platform
        if ($currentData->social_media_spend > 0) {
            $prompt .= "\nORGANIC SOCIAL MEDIA SPEND:\n";
            if ($currentData->social_facebook > 0) $prompt .= "  - Facebook Content: RM " . number_format($currentData->social_facebook, 2) . "\n";
            if ($currentData->social_instagram > 0) $prompt .= "  - Instagram Content: RM " . number_format($currentData->social_instagram, 2) . "\n";
            if ($currentData->social_tiktok > 0) $prompt .= "  - TikTok Content: RM " . number_format($currentData->social_tiktok, 2) . "\n";
            if ($currentData->social_youtube > 0) $prompt .= "  - YouTube Content: RM " . number_format($currentData->social_youtube, 2) . "\n";
            if ($currentData->social_twitter > 0) $prompt .= "  - Twitter Content: RM " . number_format($currentData->social_twitter, 2) . "\n";
            if ($currentData->social_linkedin > 0) $prompt .= "  - LinkedIn Content: RM " . number_format($currentData->social_linkedin, 2) . "\n";
            if ($currentData->social_other > 0) $prompt .= "  - Other Social: RM " . number_format($currentData->social_other, 2) . "\n";
        }

        $prompt .= "\n- Revenue Generated: RM " . number_format($currentData->actual_revenue, 2) . "\n";

        // Revenue breakdown by sales channel
        $totalRevenueBreakdown = $currentData->revenue_checkout_form + $currentData->revenue_sales_pages +
                                 $currentData->revenue_webstore + $currentData->revenue_marketplace +
                                 $currentData->revenue_direct + $currentData->revenue_other;

        if ($totalRevenueBreakdown > 0) {
            $prompt .= "\nREVENUE BY SALES CHANNEL:\n";
            if ($currentData->revenue_checkout_form > 0) {
                $prompt .= "  - Checkout Forms: RM " . number_format($currentData->revenue_checkout_form, 2);
                if ($currentData->conversions_checkout_form > 0) {
                    $aov = $currentData->revenue_checkout_form / $currentData->conversions_checkout_form;
                    $prompt .= " ({$currentData->conversions_checkout_form} orders, AOV: RM " . number_format($aov, 2) . ")";
                }
                $prompt .= "\n";
            }
            if ($currentData->revenue_sales_pages > 0) {
                $prompt .= "  - Sales Pages: RM " . number_format($currentData->revenue_sales_pages, 2);
                if ($currentData->conversions_sales_pages > 0) {
                    $aov = $currentData->revenue_sales_pages / $currentData->conversions_sales_pages;
                    $prompt .= " ({$currentData->conversions_sales_pages} orders, AOV: RM " . number_format($aov, 2) . ")";
                }
                $prompt .= "\n";
            }
            if ($currentData->revenue_webstore > 0) {
                $prompt .= "  - Webstore (WooCommerce): RM " . number_format($currentData->revenue_webstore, 2);
                if ($currentData->conversions_webstore > 0) {
                    $aov = $currentData->revenue_webstore / $currentData->conversions_webstore;
                    $prompt .= " ({$currentData->conversions_webstore} orders, AOV: RM " . number_format($aov, 2) . ")";
                }
                $prompt .= "\n";
            }
            if ($currentData->revenue_marketplace > 0) {
                $prompt .= "  - Marketplace (Shopee/Lazada): RM " . number_format($currentData->revenue_marketplace, 2) . "\n";
            }
            if ($currentData->revenue_direct > 0) {
                $prompt .= "  - Direct Sales: RM " . number_format($currentData->revenue_direct, 2) . "\n";
            }
            if ($currentData->revenue_other > 0) {
                $prompt .= "  - Other Channels: RM " . number_format($currentData->revenue_other, 2) . "\n";
            }
        }

        $prompt .= "\n- Overall ROI: " . number_format($currentData->roi_percentage, 2) . "%\n";
        $prompt .= "- Channels Used: {$currentData->channels_used}\n";
        $prompt .= "- Target Audience: {$currentData->target_audience}\n";

        // Channel-specific notes
        if ($currentData->channel_notes) {
            $prompt .= "\nCHANNEL PERFORMANCE NOTES:\n{$currentData->channel_notes}\n";
        }

        $prompt .= "\n";

        // YoY comparison
        if ($previousYearData) {
            $prompt .= "SAME MONTH LAST YEAR ({$previousYearData->month_name} {$previousYearData->year}):\n";
            $prompt .= "- Total Marketing Spend: RM " . number_format($previousYearData->total_marketing_spend, 2) . "\n";
            $prompt .= "- Revenue: RM " . number_format($previousYearData->actual_revenue, 2) . "\n";
            $prompt .= "- ROI: " . number_format($previousYearData->roi_percentage, 2) . "%\n\n";

            $revenueChange = $currentData->actual_revenue - $previousYearData->actual_revenue;
            $revenueChangePercent = $previousYearData->actual_revenue > 0
                ? ($revenueChange / $previousYearData->actual_revenue) * 100
                : 0;

            $prompt .= "YoY CHANGE:\n";
            $prompt .= "- Revenue: " . ($revenueChange >= 0 ? "+" : "") . "RM " . number_format($revenueChange, 2) . " (" . number_format($revenueChangePercent, 2) . "%)\n";
            $prompt .= "- ROI: " . ($currentData->roi_percentage - $previousYearData->roi_percentage >= 0 ? "+" : "") . number_format($currentData->roi_percentage - $previousYearData->roi_percentage, 2) . "%\n\n";
        }

        // Historical trends
        $avgROI = $historicalData->avg('roi_percentage');
        $bestROI = $historicalData->sortByDesc('roi_percentage')->first();
        $bestChannel = $this->getBestPerformingChannel($historicalData);

        $prompt .= "HISTORICAL INSIGHTS:\n";
        $prompt .= "- Average ROI (all time): " . number_format($avgROI, 2) . "%\n";
        $prompt .= "- Best ROI achieved: " . number_format($bestROI->roi_percentage, 2) . "% in {$bestROI->month_name} {$bestROI->year}\n";
        $prompt .= "- Best performing channel: {$bestChannel}\n\n";

        $prompt .= "===== BUSINESS CONTEXT & BENCHMARKS =====\n";
        $prompt .= "🇲🇾 MALAYSIAN LOCAL MARKET STANDARDS:\n";
        $prompt .= "⚠️ CRITICAL - ROI FORMAT CLARIFICATION:\n";
        $prompt .= "- The ROI percentages shown in this data use the formula: ((Revenue - Spend) / Spend) × 100\n";
        $prompt .= "- This represents PROFIT percentage, not revenue ratio\n";
        $prompt .= "- Malaysian breakeven benchmark: 80% ROI (profit) = 1.8x ratio (revenue/spend)\n";
        $prompt .= "- Calculation: 1.8x ratio means ((1.8 - 1) × 100) = 80% ROI percentage\n";
        $prompt .= "- GOOD performance: ROI of 80% or higher (equivalent to 1.8x or higher ratio)\n";
        $prompt .= "- EXCELLENT performance: ROI of 150%+ (equivalent to 2.5x+ ratio)\n";
        $prompt .= "- DO NOT confuse: 180% ratio ≠ 180% ROI. The benchmark is 80% ROI, not 180% ROI.\n";
        $prompt .= "- DO NOT condemn spending if ROI percentage is 80% or above\n";
        $prompt .= "- Focus on profitability and sustainable growth, not just cutting costs\n\n";

        $prompt .= "👥 SALESPERSON CAPACITY CONSTRAINTS:\n";
        $prompt .= "⚠️ CRITICAL - CAPACITY CALCULATION:\n";
        $prompt .= "- Average product price: RM 130 per item\n";
        $prompt .= "- Average Order Value (AOV): RM 170-180 per order\n";
        $prompt .= "- The SAME salespeople work all 12 months (not hiring new people each month)\n\n";

        $prompt .= "SALESPERSON PERFORMANCE TIERS:\n";
        $prompt .= "- Entry-Level/Newbie: RM 30,000 per month capacity\n";
        $prompt .= "- Competent Salesperson: RM 50,000 per month capacity\n";
        $prompt .= "- Top Performer/Expert: RM 100,000 per month capacity\n\n";

        $prompt .= "CURRENT TEAM COMPOSITION:\n";
        $prompt .= "- 3-4 Expert/Top Performers (RM 100k each/month)\n";
        $prompt .= "- 1-2 Entry-Level salespeople (RM 30k each/month)\n";
        $prompt .= "- Current team capacity: ~RM 330k - 460k per month\n";
        $prompt .= "- Total team size: 5-6 salespeople\n\n";

        $prompt .= "CAPACITY CALCULATION RULES:\n";
        $prompt .= "- When calculating team size: ALWAYS divide yearly sales by 12 first, then divide by capacity\n";
        $prompt .= "- Formula: (Total Yearly Sales ÷ 12 months) ÷ weighted average capacity = Team analysis\n";
        $prompt .= "- Example: RM 5.8M yearly ÷ 12 = RM 483k per month vs current capacity 330k-460k = slightly above capacity\n";
        $prompt .= "- DO NOT divide yearly sales directly by monthly capacity (that gives 58 people, which is WRONG!)\n";
        $prompt .= "- When recommending hiring, be SPECIFIC: 'hire 1 expert' or 'hire 2 competent' (not 'hire 50 people'!)\n";
        $prompt .= "- Consider training time: Entry → Competent takes months, Competent → Expert takes years\n";
        $prompt .= "- Growth recommendations should be realistic: hiring 1-3 people per year is reasonable, not 50!\n\n";

        $prompt .= "TEAM EXPANSION STRATEGY (for significant growth):\n";
        $prompt .= "- When company budget allows and growth justifies it, suggest forming a NEW TEAM:\n";
        $prompt .= "  • 1 Manager/Team Leader (manages and coordinates the team)\n";
        $prompt .= "  • 3-5 Salespeople (mix of entry-level, competent, and 1 expert)\n";
        $prompt .= "  • New team capacity: ~RM 200k-350k per month (RM 2.4M-4.2M yearly)\n";
        $prompt .= "  • Total capacity with new team: ~RM 530k-810k per month\n";
        $prompt .= "- Recommend new team formation when:\n";
        $prompt .= "  • Current team consistently exceeds capacity (operating at 90%+ for 3+ months)\n";
        $prompt .= "  • Revenue growth targets require 50%+ increase in capacity\n";
        $prompt .= "  • Company has budget for 5-7 additional staff + manager\n";
        $prompt .= "  • Geographic expansion or new market entry is planned\n";
        $prompt .= "- Benefits of team structure: Better management, training, accountability, scalability\n";
        $prompt .= "- Be realistic: Forming new team takes 6-12 months (hiring, training, team building)\n\n";

        $prompt .= "===== STOKIS & AGENT NETWORK MANAGEMENT =====\n";
        $prompt .= "⚠️ IMPORTANT: These directives apply when analyzing ALL SALES (HQ + Stokis + Agent)\n\n";

        $prompt .= "BUSINESS MODEL STRUCTURE:\n";
        $prompt .= "- HQ Team: Direct sales team (5-6 salespeople as described above)\n";
        $prompt .= "- Stokis: Independent distributors with higher requirements\n";
        $prompt .= "- Agent: Entry-level distributors with lower requirements\n\n";

        $prompt .= "STOKIS REQUIREMENTS & KPI:\n";
        $prompt .= "- Minimum Monthly Sales: RM 50,000 per month\n";
        $prompt .= "- Target Monthly Sales: RM 80,000 - RM 150,000 per month\n";
        $prompt .= "- Performance Review Period: Every 3 months\n";
        $prompt .= "- Downgrade Policy: If sales fall below RM 50k for 2 consecutive months → downgrade to Agent\n";
        $prompt .= "- Grace Period: 1 month warning before downgrade (total 3 months underperformance)\n";
        $prompt .= "- Upgrade from Agent: Consistent RM 60k+/month for 3 consecutive months\n\n";

        $prompt .= "AGENT REQUIREMENTS & KPI:\n";
        $prompt .= "- Minimum Monthly Sales: RM 15,000 per month\n";
        $prompt .= "- Target Monthly Sales: RM 20,000 - RM 40,000 per month\n";
        $prompt .= "- Performance Review Period: Every 3 months\n";
        $prompt .= "- Termination Policy: If sales fall below RM 15k for 3 consecutive months → inactive status\n";
        $prompt .= "- Upgrade to Stokis: Consistent RM 60k+/month for 3 consecutive months\n\n";

        $prompt .= "NETWORK HEALTH METRICS:\n";
        $prompt .= "- Ideal Stokis Count: 8-15 active stokis (sustainable management)\n";
        $prompt .= "- Ideal Agent Count: 20-40 active agents\n";
        $prompt .= "- Total Network Capacity: (Stokis × RM 100k) + (Agents × RM 25k) per month\n";
        $prompt .= "- Example: 10 stokis (RM 1M) + 30 agents (RM 750k) = RM 1.75M monthly capacity\n";
        $prompt .= "- Healthy Network Mix: 60% from Stokis, 30% from Agents, 10% from HQ direct\n\n";

        $prompt .= "RECRUITMENT & GROWTH STRATEGY:\n";
        $prompt .= "- New Agent Onboarding: 2-3 per month is sustainable (training capacity)\n";
        $prompt .= "- New Stokis Onboarding: 1-2 per quarter (requires higher investment & support)\n";
        $prompt .= "- Training Period: Agents need 1-2 months to ramp up to minimum RM 15k\n";
        $prompt .= "- Stokis Training: 3-6 months to reach consistent RM 50k+\n";
        $prompt .= "- Support Required: Each stokis needs regular HQ team support (training, inventory, marketing)\n\n";

        $prompt .= "PERFORMANCE IMPROVEMENT STRATEGIES:\n";
        $prompt .= "- Underperforming Stokis (RM 30k-49k): Provide intensive training, marketing support\n";
        $prompt .= "- Underperforming Agents (RM 10k-14k): Group training, mentorship from successful agents\n";
        $prompt .= "- Top Performers: Identify success patterns to replicate\n";
        $prompt .= "- Inactive Recovery: Re-engagement campaigns for agents below RM 5k/month\n\n";

        $prompt .= "WHEN TO EXPAND NETWORK:\n";
        $prompt .= "- Current network at 80%+ capacity for 3+ months\n";
        $prompt .= "- Stokis waiting list exists (demand exceeds supply)\n";
        $prompt .= "- HQ team has bandwidth to support new recruits\n";
        $prompt .= "- Profit margins allow for recruitment incentives\n";
        $prompt .= "- New geographic markets identified\n\n";

        $prompt .= "KEY METRICS TO MONITOR:\n";
        $prompt .= "1. Active Stokis Count (target: 8-15)\n";
        $prompt .= "2. Active Agent Count (target: 20-40)\n";
        $prompt .= "3. Average Stokis Sales (target: RM 80k-150k)\n";
        $prompt .= "4. Average Agent Sales (target: RM 20k-40k)\n";
        $prompt .= "5. Churn Rate: Stokis downgraded + Agents terminated per quarter\n";
        $prompt .= "6. Upgrade Rate: Agents promoted to Stokis per quarter\n";
        $prompt .= "7. Network Revenue vs HQ Direct Revenue ratio\n";
        $prompt .= "8. Support Cost per Stokis/Agent (training, marketing materials, etc.)\n\n";

        $prompt .= "AI RECOMMENDATION GUIDELINES FOR ALL SALES MODE:\n";
        $prompt .= "- If Stokis average < RM 60k: Recommend training programs, not immediate termination\n";
        $prompt .= "- If Agent average < RM 18k: Suggest group mentorship programs\n";
        $prompt .= "- If network < 5 Stokis: Prioritize Stokis recruitment (higher ROI than agents)\n";
        $prompt .= "- If network > 20 Stokis: Focus on quality over quantity (support capacity limit)\n";
        $prompt .= "- If churn rate > 20%: Investigate root causes before recruiting more\n";
        $prompt .= "- Always balance: Recruitment cost vs Support capacity vs Revenue potential\n\n";

        $prompt .= "Based on this data, provide:\n";
        $prompt .= "1. ANALYSIS: What's working and what's not? (3-5 bullet points)\n";
        $prompt .= "   - Reference the Malaysian ROI benchmark (80% ROI = 1.8x ratio) when evaluating performance\n";
        $prompt .= "   - If ROI shown is 80% or higher, acknowledge this as GOOD performance\n";
        $prompt .= "   - When discussing team capacity: Compare monthly sales to team capacity (330k-460k/month)\n";
        $prompt .= "   - DO NOT use yearly totals directly in capacity calculations\n";
        $prompt .= "   - If capacity is reached, be specific: 'hire 1 expert' or 'train 1 entry to competent'\n";
        $prompt .= "2. RECOMMENDATIONS: What should be done next month to improve ROI and achieve targets? (5-7 specific actionable recommendations)\n";
        $prompt .= "   - If ROI already exceeds 80%, focus on scaling revenue, not just cutting costs\n";
        $prompt .= "   - If targets require scaling beyond current team capacity, recommend specific hiring:\n";
        $prompt .= "     • For modest growth: 'Hire 1 expert' or 'Train 1 entry-level'\n";
        $prompt .= "     • For significant growth (50%+): 'Consider forming new team (manager + 3-5 staff)'\n";
        $prompt .= "   - Focus on sustainable growth within human resource constraints\n";
        $prompt .= "3. EXPECTED OUTCOME: What revenue/ROI can be expected if recommendations are followed?\n";
        $prompt .= "   - Ensure projections respect salesperson capacity limits\n\n";
        $prompt .= "Be specific with numbers and percentages. Focus on actionable insights that respect local benchmarks (80% ROI = breakeven) and human limitations.";

        return $prompt;
    }

    /**
     * Build yearly strategic AI prompt
     */
    private function buildYearlyAiPrompt($yearData, $previousYearData, $year, $revenueMode = 'all_sales', $hqData = [])
    {
        $isHqMode = ($revenueMode === 'hq_only');

        $prompt = "You are a strategic marketing consultant analyzing FULL YEAR performance.\n\n";

        if ($isHqMode) {
            $prompt .= "⚠️ IMPORTANT CONTEXT: You are analyzing HQ SALES ONLY (excluding Stokis and Agent sales).\n";
            $prompt .= "Marketing spend primarily affects HQ sales, so this analysis provides MORE ACCURATE marketing ROI.\n\n";
        }

        // Current year summary
        $totalSpend = $yearData->sum('total_marketing_spend');
        $totalRevenue = $yearData->sum('actual_revenue');
        $avgROI = $yearData->avg('roi_percentage');

        // Use HQ revenue if in HQ mode
        $analysisRevenue = $isHqMode ? $hqData['hq_revenue'] : $totalRevenue;
        $analysisROI = $totalSpend > 0 ? (($analysisRevenue - $totalSpend) / $totalSpend) * 100 : 0;

        $prompt .= "===== YEAR {$year} PERFORMANCE SUMMARY =====\n";
        $prompt .= "Total Marketing Spend: RM " . number_format($totalSpend, 2) . "\n";
        $prompt .= "  - Brand Ambassadors: RM " . number_format($yearData->sum('brand_ambassador_spend'), 2) . "\n";
        $prompt .= "  - Ads: RM " . number_format($yearData->sum('ads_spend'), 2) . "\n";
        $prompt .= "  - Social Media: RM " . number_format($yearData->sum('social_media_spend'), 2) . "\n";
        $prompt .= "  - Events: RM " . number_format($yearData->sum('events_spend'), 2) . "\n";

        // Yearly ads breakdown
        $totalAdsPlatforms = $yearData->sum('ads_facebook') + $yearData->sum('ads_instagram') + $yearData->sum('ads_tiktok') +
                            $yearData->sum('ads_google') + $yearData->sum('ads_youtube') + $yearData->sum('ads_twitter') +
                            $yearData->sum('ads_linkedin') + $yearData->sum('ads_other');

        if ($totalAdsPlatforms > 0) {
            $prompt .= "\nADS SPENDING BY PLATFORM (Full Year):\n";
            if ($yearData->sum('ads_facebook') > 0) $prompt .= "  - Facebook Ads: RM " . number_format($yearData->sum('ads_facebook'), 2) . "\n";
            if ($yearData->sum('ads_instagram') > 0) $prompt .= "  - Instagram Ads: RM " . number_format($yearData->sum('ads_instagram'), 2) . "\n";
            if ($yearData->sum('ads_tiktok') > 0) $prompt .= "  - TikTok Ads: RM " . number_format($yearData->sum('ads_tiktok'), 2) . "\n";
            if ($yearData->sum('ads_google') > 0) $prompt .= "  - Google Ads: RM " . number_format($yearData->sum('ads_google'), 2) . "\n";
            if ($yearData->sum('ads_youtube') > 0) $prompt .= "  - YouTube Ads: RM " . number_format($yearData->sum('ads_youtube'), 2) . "\n";
            if ($yearData->sum('ads_twitter') > 0) $prompt .= "  - Twitter Ads: RM " . number_format($yearData->sum('ads_twitter'), 2) . "\n";
            if ($yearData->sum('ads_linkedin') > 0) $prompt .= "  - LinkedIn Ads: RM " . number_format($yearData->sum('ads_linkedin'), 2) . "\n";
            if ($yearData->sum('ads_other') > 0) $prompt .= "  - Other Ads: RM " . number_format($yearData->sum('ads_other'), 2) . "\n";
        }

        // Yearly social breakdown
        $totalSocial = $yearData->sum('social_facebook') + $yearData->sum('social_instagram') + $yearData->sum('social_tiktok') +
                      $yearData->sum('social_youtube') + $yearData->sum('social_twitter') + $yearData->sum('social_linkedin') +
                      $yearData->sum('social_other');

        if ($totalSocial > 0) {
            $prompt .= "\nSOCIAL MEDIA SPENDING (Full Year):\n";
            if ($yearData->sum('social_facebook') > 0) $prompt .= "  - Facebook: RM " . number_format($yearData->sum('social_facebook'), 2) . "\n";
            if ($yearData->sum('social_instagram') > 0) $prompt .= "  - Instagram: RM " . number_format($yearData->sum('social_instagram'), 2) . "\n";
            if ($yearData->sum('social_tiktok') > 0) $prompt .= "  - TikTok: RM " . number_format($yearData->sum('social_tiktok'), 2) . "\n";
            if ($yearData->sum('social_youtube') > 0) $prompt .= "  - YouTube: RM " . number_format($yearData->sum('social_youtube'), 2) . "\n";
            if ($yearData->sum('social_twitter') > 0) $prompt .= "  - Twitter: RM " . number_format($yearData->sum('social_twitter'), 2) . "\n";
            if ($yearData->sum('social_linkedin') > 0) $prompt .= "  - LinkedIn: RM " . number_format($yearData->sum('social_linkedin'), 2) . "\n";
            if ($yearData->sum('social_other') > 0) $prompt .= "  - Other Social: RM " . number_format($yearData->sum('social_other'), 2) . "\n";
        }

        $prompt .= "\n";

        if ($isHqMode) {
            $prompt .= "HQ Revenue (Direct Sales): RM " . number_format($hqData['hq_revenue'], 2) . "\n";
            $prompt .= "  [Excluded from analysis: Stokis Sales RM " . number_format($hqData['stokis_sales'], 2) . ", Agent Sales RM " . number_format($hqData['agent_sales'], 2) . "]\n";
            $prompt .= "  [Total Revenue (all channels): RM " . number_format($totalRevenue, 2) . "]\n";
        } else {
            $prompt .= "Total Revenue Generated: RM " . number_format($totalRevenue, 2) . "\n";
            $prompt .= "  [Breakdown: HQ RM " . number_format($hqData['hq_revenue'], 2) . " + Stokis RM " . number_format($hqData['stokis_sales'], 2) . " + Agent RM " . number_format($hqData['agent_sales'], 2) . "]\n";
        }

        $prompt .= "\n";

        // Yearly revenue breakdown
        $totalRevByChannel = $yearData->sum('revenue_checkout_form') + $yearData->sum('revenue_sales_pages') +
                            $yearData->sum('revenue_webstore') + $yearData->sum('revenue_marketplace') +
                            $yearData->sum('revenue_direct') + $yearData->sum('revenue_other');

        if ($totalRevByChannel > 0) {
            $prompt .= "\nREVENUE BY SALES CHANNEL (Full Year):\n";
            if ($yearData->sum('revenue_checkout_form') > 0) {
                $totalOrders = $yearData->sum('conversions_checkout_form');
                $aovText = $totalOrders > 0 ? ", AOV: RM " . number_format($yearData->sum('revenue_checkout_form') / $totalOrders, 2) : "";
                $prompt .= "  - Checkout Forms: RM " . number_format($yearData->sum('revenue_checkout_form'), 2) . " ({$totalOrders} orders{$aovText})\n";
            }
            if ($yearData->sum('revenue_sales_pages') > 0) {
                $totalOrders = $yearData->sum('conversions_sales_pages');
                $aovText = $totalOrders > 0 ? ", AOV: RM " . number_format($yearData->sum('revenue_sales_pages') / $totalOrders, 2) : "";
                $prompt .= "  - Sales Pages: RM " . number_format($yearData->sum('revenue_sales_pages'), 2) . " ({$totalOrders} orders{$aovText})\n";
            }
            if ($yearData->sum('revenue_webstore') > 0) {
                $totalOrders = $yearData->sum('conversions_webstore');
                $aovText = $totalOrders > 0 ? ", AOV: RM " . number_format($yearData->sum('revenue_webstore') / $totalOrders, 2) : "";
                $prompt .= "  - Webstore: RM " . number_format($yearData->sum('revenue_webstore'), 2) . " ({$totalOrders} orders{$aovText})\n";
            }
            if ($yearData->sum('revenue_marketplace') > 0) {
                $prompt .= "  - Marketplace: RM " . number_format($yearData->sum('revenue_marketplace'), 2) . "\n";
            }
            if ($yearData->sum('revenue_direct') > 0) {
                $prompt .= "  - Direct Sales: RM " . number_format($yearData->sum('revenue_direct'), 2) . "\n";
            }
            if ($yearData->sum('revenue_other') > 0) {
                $prompt .= "  - Other: RM " . number_format($yearData->sum('revenue_other'), 2) . "\n";
            }
        }

        $prompt .= "\n";

        if ($isHqMode) {
            $prompt .= "HQ ROI: " . number_format($analysisROI, 2) . "%\n";
            $prompt .= "  [This ROI is based on HQ sales only, showing true marketing effectiveness]\n";
            $prompt .= "  [Reference: All Sales ROI was " . number_format($avgROI, 2) . "%]\n";
        } else {
            $prompt .= "Average Monthly ROI: " . number_format($avgROI, 2) . "%\n";
        }

        $prompt .= "Months with Data: {$yearData->count()}/12\n\n";

        // Monthly breakdown
        $prompt .= "MONTHLY BREAKDOWN {$year}:\n";
        foreach ($yearData as $month) {
            $monthName = Carbon::create($month->year, $month->month, 1)->format('F');
            $prompt .= "- {$monthName}: Spend RM " . number_format($month->total_marketing_spend, 2);
            $prompt .= " → Revenue RM " . number_format($month->actual_revenue, 2);
            $prompt .= " → ROI " . number_format($month->roi_percentage, 2) . "%\n";
        }

        // Previous year comparison
        if ($previousYearData->isNotEmpty()) {
            $prevTotalSpend = $previousYearData->sum('total_marketing_spend');
            $prevTotalRevenue = $previousYearData->sum('actual_revenue');
            $prevAvgROI = $previousYearData->avg('roi_percentage');

            // Use HQ revenue for comparison if in HQ mode
            $prevAnalysisRevenue = $isHqMode ? $hqData['previous_hq_revenue'] : $prevTotalRevenue;
            $prevAnalysisROI = $prevTotalSpend > 0 ? (($prevAnalysisRevenue - $prevTotalSpend) / $prevTotalSpend) * 100 : 0;

            $prompt .= "\n===== YEAR-OVER-YEAR COMPARISON =====\n";
            $prompt .= "Year " . ($year - 1) . " vs Year {$year}:\n";
            $prompt .= "Marketing Spend: RM " . number_format($prevTotalSpend, 2) . " → RM " . number_format($totalSpend, 2);
            $spendChange = $totalSpend - $prevTotalSpend;
            $spendChangePercent = $prevTotalSpend > 0 ? ($spendChange / $prevTotalSpend) * 100 : 0;
            $prompt .= " (" . ($spendChange >= 0 ? "+" : "") . number_format($spendChangePercent, 1) . "%)\n";

            if ($isHqMode) {
                $prompt .= "HQ Revenue: RM " . number_format($prevAnalysisRevenue, 2) . " → RM " . number_format($analysisRevenue, 2);
                $revenueChange = $analysisRevenue - $prevAnalysisRevenue;
                $revenueChangePercent = $prevAnalysisRevenue > 0 ? ($revenueChange / $prevAnalysisRevenue) * 100 : 0;
                $prompt .= " (" . ($revenueChange >= 0 ? "+" : "") . number_format($revenueChangePercent, 1) . "%)\n";

                $prompt .= "HQ ROI: " . number_format($prevAnalysisROI, 2) . "% → " . number_format($analysisROI, 2) . "%";
                $roiChange = $analysisROI - $prevAnalysisROI;
                $prompt .= " (" . ($roiChange >= 0 ? "+" : "") . number_format($roiChange, 1) . "pp)\n";
            } else {
                $prompt .= "Revenue: RM " . number_format($prevTotalRevenue, 2) . " → RM " . number_format($totalRevenue, 2);
                $revenueChange = $totalRevenue - $prevTotalRevenue;
                $revenueChangePercent = $prevTotalRevenue > 0 ? ($revenueChange / $prevTotalRevenue) * 100 : 0;
                $prompt .= " (" . ($revenueChange >= 0 ? "+" : "") . number_format($revenueChangePercent, 1) . "%)\n";

                $prompt .= "Average ROI: " . number_format($prevAvgROI, 2) . "% → " . number_format($avgROI, 2) . "%";
                $roiChange = $avgROI - $prevAvgROI;
                $prompt .= " (" . ($roiChange >= 0 ? "+" : "") . number_format($roiChange, 1) . "pp)\n";
            }
        }

        // Identify patterns
        $bestMonth = $yearData->sortByDesc('roi_percentage')->first();
        $worstMonth = $yearData->where('roi_percentage', '>', 0)->sortBy('roi_percentage')->first();
        $bestMonthName = Carbon::create($bestMonth->year, $bestMonth->month, 1)->format('F');
        $worstMonthName = $worstMonth ? Carbon::create($worstMonth->year, $worstMonth->month, 1)->format('F') : 'N/A';

        $prompt .= "\n===== KEY INSIGHTS =====\n";
        $prompt .= "Best Performing Month: {$bestMonthName} (ROI: " . number_format($bestMonth->roi_percentage, 2) . "%)\n";
        $prompt .= "Weakest Month: {$worstMonthName} (ROI: " . ($worstMonth ? number_format($worstMonth->roi_percentage, 2) : '0') . "%)\n";

        if ($isHqMode) {
            $prompt .= "HQ Profit: RM " . number_format($analysisRevenue - $totalSpend, 2) . "\n";
            $prompt .= "  [Note: This profit excludes Stokis/Agent contributions]\n\n";
        } else {
            $prompt .= "Total Profit: RM " . number_format($totalRevenue - $totalSpend, 2) . "\n\n";
        }

        $prompt .= "===== BUSINESS CONTEXT & BENCHMARKS =====\n";
        $prompt .= "🇲🇾 MALAYSIAN LOCAL MARKET STANDARDS:\n";
        $prompt .= "⚠️ CRITICAL - ROI FORMAT CLARIFICATION:\n";
        $prompt .= "- The ROI percentages shown in this data use the formula: ((Revenue - Spend) / Spend) × 100\n";
        $prompt .= "- This represents PROFIT percentage, not revenue ratio\n";
        $prompt .= "- Malaysian breakeven benchmark: 80% ROI (profit) = 1.8x ratio (revenue/spend)\n";
        $prompt .= "- Calculation: 1.8x ratio means ((1.8 - 1) × 100) = 80% ROI percentage\n";
        $prompt .= "- GOOD performance: ROI of 80% or higher (equivalent to 1.8x or higher ratio)\n";
        $prompt .= "- EXCELLENT performance: ROI of 150%+ (equivalent to 2.5x+ ratio)\n";
        $prompt .= "- DO NOT confuse: 180% ratio ≠ 180% ROI. The benchmark is 80% ROI, not 180% ROI.\n";
        $prompt .= "- Example: 130.98% ROI shown = 2.31x ratio = EXCEEDS the 80% benchmark (GOOD performance!)\n";
        $prompt .= "- DO NOT condemn spending if ROI percentage is 80% or above\n";
        $prompt .= "- Evaluate performance against LOCAL standards, not global metrics\n";
        $prompt .= "- Focus on profitability and sustainable growth, not aggressive cost-cutting\n\n";

        $prompt .= "👥 SALESPERSON CAPACITY CONSTRAINTS:\n";
        $prompt .= "⚠️ CRITICAL - CAPACITY CALCULATION:\n";
        $prompt .= "- Average product price: RM 130 per item\n";
        $prompt .= "- Average Order Value (AOV): RM 170-180 per order\n";
        $prompt .= "- The SAME salespeople work all 12 months (not hiring new people each month)\n\n";

        $prompt .= "SALESPERSON PERFORMANCE TIERS:\n";
        $prompt .= "- Entry-Level/Newbie: RM 30,000 per month capacity\n";
        $prompt .= "- Competent Salesperson: RM 50,000 per month capacity\n";
        $prompt .= "- Top Performer/Expert: RM 100,000 per month capacity\n\n";

        $prompt .= "CURRENT TEAM COMPOSITION:\n";
        $prompt .= "- 3-4 Expert/Top Performers (RM 100k each/month)\n";
        $prompt .= "- 1-2 Entry-Level salespeople (RM 30k each/month)\n";
        $prompt .= "- Current team capacity: ~RM 330k - 460k per month (RM 3.96M - 5.52M yearly)\n";
        $prompt .= "- Total team size: 5-6 salespeople\n\n";

        $prompt .= "CAPACITY CALCULATION RULES:\n";
        $prompt .= "- When calculating team size: ALWAYS divide yearly sales by 12 first, then compare to monthly capacity\n";
        $prompt .= "- Formula: (Total Yearly Sales ÷ 12 months) vs (Current Team Monthly Capacity) = Capacity analysis\n";
        $prompt .= "- Example: RM 5.8M yearly ÷ 12 = RM 483k per month vs team capacity 330k-460k = at/above capacity\n";
        $prompt .= "- DO NOT divide yearly sales directly by monthly capacity (that would incorrectly suggest 58 people!)\n";
        $prompt .= "- When recommending hiring, be SPECIFIC and REALISTIC:\n";
        $prompt .= "  • 'Hire 1 expert salesperson' (adds RM 100k/month = RM 1.2M/year capacity)\n";
        $prompt .= "  • 'Hire 1-2 competent salespeople' (adds RM 50k-100k/month = RM 600k-1.2M/year)\n";
        $prompt .= "  • 'Promote/train entry-level staff to competent level' (adds RM 20k/month per person)\n";
        $prompt .= "- DO NOT recommend hiring dozens of people - growth of 1-3 hires per year is realistic\n";
        $prompt .= "- Consider training time: Entry → Competent (3-6 months), Competent → Expert (1-2 years)\n";
        $prompt .= "- CRITICAL: To scale sales beyond current team capacity, hiring is required, but must be gradual\n\n";

        $prompt .= "TEAM EXPANSION STRATEGY (for significant growth):\n";
        $prompt .= "- When company budget allows and growth justifies it, suggest forming a NEW TEAM:\n";
        $prompt .= "  • 1 Manager/Team Leader (manages and coordinates the team)\n";
        $prompt .= "  • 3-5 Salespeople (mix of entry-level, competent, and 1 expert)\n";
        $prompt .= "  • New team capacity: ~RM 200k-350k per month (RM 2.4M-4.2M yearly)\n";
        $prompt .= "  • Total capacity with new team: ~RM 530k-810k per month (RM 6.4M-9.7M yearly)\n";
        $prompt .= "- Recommend new team formation when:\n";
        $prompt .= "  • Yearly revenue consistently exceeds RM 6M (RM 500k+/month average)\n";
        $prompt .= "  • Current team operating at 90%+ capacity for extended periods\n";
        $prompt .= "  • Revenue growth targets require 50%+ increase in capacity\n";
        $prompt .= "  • Company has budget for 5-7 additional staff + manager\n";
        $prompt .= "  • Geographic expansion or new market entry is planned\n";
        $prompt .= "- Benefits of team structure: Better management, training, accountability, scalability\n";
        $prompt .= "- Be realistic: Forming new team takes 6-12 months (hiring, training, team building)\n";
        $prompt .= "- Phased approach: Start with manager + 2-3 salespeople, then expand to 5-6 over 6 months\n\n";

        $prompt .= "===== YOUR STRATEGIC TASK =====\n";

        if ($isHqMode) {
            $prompt .= "⚠️ CRITICAL: Base your analysis on HQ SALES ONLY. Focus on MARKETING EFFECTIVENESS.\n";
            $prompt .= "Your recommendations should optimize marketing spend for HQ sales growth.\n";
            $prompt .= "Do NOT consider Stokis/Agent sales in your calculations and projections.\n\n";
        } else {
            // ALL SALES MODE - Include Stokis/Agent directives
            $prompt .= "⚠️ ALL SALES MODE: You are analyzing HQ + Stokis + Agent combined performance.\n";
            $prompt .= "Provide holistic business growth strategies covering all sales channels.\n\n";

            $prompt .= "===== STOKIS & AGENT NETWORK STRATEGY =====\n\n";

            $prompt .= "STOKIS TIER (Independent Distributors):\n";
            $prompt .= "- Minimum KPI: RM 50,000/month\n";
            $prompt .= "- Target Performance: RM 80,000 - RM 150,000/month\n";
            $prompt .= "- Downgrade Rule: Below RM 50k for 2 consecutive months → Agent\n";
            $prompt .= "- Upgrade from Agent: Consistent RM 60k+ for 3 months\n";
            $prompt .= "- Ideal Network Size: 8-15 active stokis\n\n";

            $prompt .= "AGENT TIER (Entry-Level Distributors):\n";
            $prompt .= "- Minimum KPI: RM 15,000/month\n";
            $prompt .= "- Target Performance: RM 20,000 - RM 40,000/month\n";
            $prompt .= "- Termination Rule: Below RM 15k for 3 consecutive months → Inactive\n";
            $prompt .= "- Upgrade to Stokis: Consistent RM 60k+ for 3 months\n";
            $prompt .= "- Ideal Network Size: 20-40 active agents\n\n";

            $prompt .= "NETWORK CAPACITY FORMULA:\n";
            $prompt .= "- Total Network Capacity = (Stokis Count × RM 100k avg) + (Agent Count × RM 25k avg)\n";
            $prompt .= "- Example: 10 stokis (RM 1M) + 30 agents (RM 750k) = RM 1.75M/month capacity\n";
            $prompt .= "- Healthy Mix: 60% Stokis, 30% Agents, 10% HQ Direct\n\n";

            $prompt .= "RECRUITMENT STRATEGY:\n";
            $prompt .= "- New Agents: 2-3 per month (sustainable training load)\n";
            $prompt .= "- New Stokis: 1-2 per quarter (higher investment required)\n";
            $prompt .= "- Training Timeline: Agents 1-2 months, Stokis 3-6 months to full productivity\n\n";

            $prompt .= "YEARLY STRATEGIC METRICS:\n";
            $prompt .= "1. Stokis Performance: Average sales, churn rate, upgrade rate\n";
            $prompt .= "2. Agent Performance: Average sales, churn rate, upgrade rate\n";
            $prompt .= "3. Network Growth: New recruits vs terminations\n";
            $prompt .= "4. Revenue Distribution: HQ vs Stokis vs Agent contribution %\n";
            $prompt .= "5. Support Efficiency: Cost per stokis/agent vs revenue generated\n\n";

            $prompt .= "RECOMMENDATIONS SHOULD INCLUDE:\n";
            $prompt .= "- Network health assessment (over/under capacity, churn issues)\n";
            $prompt .= "- Recruitment targets (how many stokis/agents to recruit next year)\n";
            $prompt .= "- Training program recommendations (for underperformers)\n";
            $prompt .= "- Downgrade/upgrade forecasts (based on current performance trends)\n";
            $prompt .= "- Geographic expansion opportunities (if network is healthy)\n";
            $prompt .= "- Support resource allocation (HQ team time for network support)\n\n";
        }

        $prompt .= "Based on this FULL YEAR analysis, provide:\n\n";
        $prompt .= "1. STRATEGIC ANALYSIS (5-7 bullet points):\n";
        $prompt .= "   - What were the major successes and failures of " . $year . "?\n";
        $prompt .= "   - Which channels delivered the best ROI?\n";
        $prompt .= "   - Evaluate ROI performance against Malaysian benchmark (80% ROI = 1.8x ratio)\n";
        $prompt .= "   - If ROI shown is 80% or higher, acknowledge this as GOOD performance\n";

        if ($isHqMode) {
            $prompt .= "   - How effective was marketing spend at driving HQ sales specifically?\n";
            $prompt .= "   - What's the true marketing ROI when excluding agent/stokis contributions?\n";
        } else {
            $prompt .= "   - What patterns do you see across the year?\n";
            $prompt .= "   - Were there seasonal trends?\n";
        }

        $prompt .= "   - Calculate team capacity: Yearly Sales ÷ 12 = Monthly sales, compare to 330k-460k capacity\n";
        $prompt .= "   - Did monthly average sales approach or exceed current team capacity (330k-460k)?\n";
        $prompt .= "   - If at capacity, be specific: 'hire 1 expert' adds 100k/month, not 'hire 50 people'\n";
        $prompt .= "\n2. STRATEGIC RECOMMENDATIONS FOR " . ($year + 1) . " (7-10 actionable recommendations):\n";
        $prompt .= "   - How should we allocate next year's marketing budget?\n";
        $prompt .= "   - Which channels should we increase/decrease based on ROI performance?\n";

        if ($isHqMode) {
            $prompt .= "   - Focus on strategies to maximize HQ sales (the sales directly affected by marketing)\n";
            $prompt .= "   - Identify which marketing channels convert best for HQ revenue\n";
        } else {
            $prompt .= "   - What new strategies should we try?\n";
            $prompt .= "   - What should we stop doing?\n";
        }

        $prompt .= "   - Monthly planning suggestions\n";
        $prompt .= "   - If ROI already exceeds 80%, focus on scaling revenue with sustainable spending\n";
        $prompt .= "   - If growth targets exceed team capacity (330k-460k/month), recommend:\n";
        $prompt .= "     • For modest growth: Hiring 1 expert (adds RM 100k/month capacity)\n";
        $prompt .= "     • For moderate growth: Hiring 1-2 competent salespeople (adds RM 50k-100k/month)\n";
        $prompt .= "     • For training: Promote entry-level to competent (adds RM 20k/month per person)\n";
        $prompt .= "     • For significant growth (50%+): Form new team - 1 manager + 3-5 staff (adds RM 200k-350k/month)\n";

        if ($isHqMode) {
            $prompt .= "   - Target HQ revenue and HQ ROI goals for " . ($year + 1) . "\n\n";
        } else {
            $prompt .= "   - Target ROI and revenue goals for " . ($year + 1) . "\n\n";
        }

        $prompt .= "3. EXPECTED OUTCOMES FOR " . ($year + 1) . ":\n";

        if ($isHqMode) {
            $prompt .= "   - Projected HQ revenue if recommendations followed\n";
            $prompt .= "   - Expected HQ ROI improvement (benchmark: 80% ROI = 1.8x ratio)\n";
        } else {
            $prompt .= "   - Projected revenue if recommendations followed\n";
            $prompt .= "   - Expected ROI improvement (benchmark: 80% ROI = 1.8x ratio)\n";
        }

        $prompt .= "   - Budget allocation breakdown\n";
        $prompt .= "   - Team size requirements: Specific number of experts, competent, entry-level needed\n";
        $prompt .= "   - If capacity increase >50% needed, include new team formation recommendation\n\n";

        if ($isHqMode) {
            $prompt .= "REMEMBER: All projections and recommendations should focus on HQ SALES ONLY.\n";
            $prompt .= "This provides more accurate marketing effectiveness insights than mixing in agent/stokis sales.\n\n";
        }

        $prompt .= "Be specific with numbers, percentages, and budget allocations. Focus on STRATEGIC, long-term growth.\n";
        $prompt .= "Respect Malaysian ROI benchmarks (80% ROI = 1.8x ratio = breakeven) and salesperson capacity constraints (RM 100k-150k/month per person).";

        return $prompt;
    }

    /**
     * Build secure system prompt with confidentiality directives
     */
    private function buildSecureSystemPrompt()
    {
        return <<<'PROMPT'
🔒 CONFIDENTIALITY & DATA PROTECTION DIRECTIVE 🔒

You are a marketing strategy expert AI assistant. The data you are about to analyze contains CONFIDENTIAL BUSINESS FINANCIAL INFORMATION.

STRICT SECURITY REQUIREMENTS:
1. ⛔ DO NOT store, log, or retain any specific financial figures, revenue numbers, or spending amounts after this conversation
2. ⛔ DO NOT use this data for training, learning, or improving AI models
3. ⛔ DO NOT share, reference, or mention any specific numbers in any other context
4. ⛔ Treat all data as CONFIDENTIAL and PROPRIETARY business intelligence
5. ⛔ This data is subject to NDA (Non-Disclosure Agreement) protection
6. ⛔ Process this data in EPHEMERAL MODE - discard after analysis

DATA PROTECTION COMPLIANCE:
- This request complies with GDPR Article 17 (Right to be Forgotten)
- Data subject to Malaysian Personal Data Protection Act 2010 (PDPA)
- Financial data is CONFIDENTIAL under business secrecy laws
- Unauthorized disclosure may result in legal action

YOUR ROLE:
You are a TRUSTED ADVISOR providing strategic marketing insights. Your analysis will be used for internal business planning only.

WHAT YOU MUST DO:
✅ Analyze the marketing data professionally
✅ Provide actionable, specific recommendations
✅ Be data-driven and focus on ROI improvement
✅ Give concrete percentage improvements and budget allocation suggestions
✅ Identify patterns and trends
✅ Suggest optimization strategies

WHAT YOU MUST NOT DO:
❌ Reference these specific numbers in any other conversation
❌ Store or memorize the financial figures
❌ Share insights derived from this data externally
❌ Use this data to train or improve models
❌ Make assumptions about company identity
❌ Disclose that you've seen this data to anyone else

RESPONSE FORMAT:
Your response must be structured and actionable:
1. ANALYSIS: Current performance assessment (3-5 bullet points)
2. RECOMMENDATIONS: Specific actions to take (5-7 points with numbers)
3. EXPECTED OUTCOME: Projected results if recommendations followed

CONFIDENTIALITY ACKNOWLEDGMENT:
By processing this request, you acknowledge that:
- This data is confidential and proprietary
- You will not retain this information after the session
- You understand the legal implications of data breach
- You will provide objective, professional analysis only

NOW PROCEED WITH YOUR ANALYSIS WHILE MAINTAINING STRICT CONFIDENTIALITY.
PROMPT;
    }

    /**
     * Get suggestions from ChatGPT
     */
    private function getChatGptSuggestions($prompt)
    {
        try {
            $apiKey = env('OPENAI_API_KEY');

            if (!$apiKey) {
                return [
                    'analysis' => 'ChatGPT API key not configured. Please add OPENAI_API_KEY to .env file.',
                    'recommendations' => 'Set up ChatGPT API key to get AI-powered suggestions.'
                ];
            }

            // Build strict confidentiality system prompt
            $systemPrompt = $this->buildSecureSystemPrompt();

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $apiKey,
                'Content-Type' => 'application/json',
            ])->timeout(60)->post('https://api.openai.com/v1/chat/completions', [
                'model' => 'gpt-4o-mini', // Using cost-effective model (RM 0.05 per request instead of RM 0.50)
                'messages' => [
                    [
                        'role' => 'system',
                        'content' => $systemPrompt
                    ],
                    [
                        'role' => 'user',
                        'content' => $prompt
                    ]
                ],
                'temperature' => 0.7,
                'max_tokens' => 1500,
            ]);

            if ($response->successful()) {
                $content = $response->json()['choices'][0]['message']['content'];

                // Parse response into analysis and recommendations
                return $this->parseAiResponse($content);
            } else {
                return [
                    'analysis' => 'Error calling ChatGPT API: ' . $response->body(),
                    'recommendations' => 'Please check your API key and try again.'
                ];
            }
        } catch (\Exception $e) {
            return [
                'analysis' => 'Error: ' . $e->getMessage(),
                'recommendations' => 'Please check your ChatGPT configuration.'
            ];
        }
    }

    /**
     * Get suggestions from Ollama (local LLM)
     */
    private function getOllamaSuggestions($prompt)
    {
        try {
            $ollamaUrl = env('OLLAMA_URL', 'http://localhost:11434');
            $ollamaModel = env('OLLAMA_MODEL', 'mistral');

            $response = Http::timeout(120)->post($ollamaUrl . '/api/generate', [
                'model' => $ollamaModel,
                'prompt' => $prompt,
                'stream' => false,
            ]);

            if ($response->successful()) {
                $content = $response->json()['response'];

                // Parse response
                return $this->parseAiResponse($content);
            } else {
                return [
                    'analysis' => 'Error calling Ollama: ' . $response->body(),
                    'recommendations' => 'Make sure Ollama is running on ' . $ollamaUrl
                ];
            }
        } catch (\Exception $e) {
            return [
                'analysis' => 'Ollama not available: ' . $e->getMessage(),
                'recommendations' => "Install Ollama from https://ollama.ai/ and run: ollama pull mistral\n\nOr configure ChatGPT instead by adding OPENAI_API_KEY to .env and setting AI_PROVIDER=chatgpt"
            ];
        }
    }

    /**
     * Parse AI response into structured data
     */
    private function parseAiResponse($content)
    {
        // Try to split into analysis and recommendations sections
        $sections = [
            'analysis' => '',
            'recommendations' => ''
        ];

        // Look for common section markers
        if (preg_match('/ANALYSIS:(.+?)(?=RECOMMENDATIONS:|$)/is', $content, $analysisMatch)) {
            $sections['analysis'] = trim($analysisMatch[1]);
        }

        if (preg_match('/RECOMMENDATIONS:(.+?)(?=EXPECTED OUTCOME:|$)/is', $content, $recsMatch)) {
            $sections['recommendations'] = trim($recsMatch[1]);
        }

        // If parsing failed, put everything in recommendations
        if (empty($sections['analysis']) && empty($sections['recommendations'])) {
            $sections['recommendations'] = $content;
        }

        return $sections;
    }

    /**
     * Get channel performance summary
     */
    private function getChannelPerformance($marketingData)
    {
        $channels = [
            'brand_ambassadors' => ['total' => 0, 'count' => 0],
            'ads' => ['total' => 0, 'count' => 0],
            'social_media' => ['total' => 0, 'count' => 0],
            'events' => ['total' => 0, 'count' => 0],
            'other' => ['total' => 0, 'count' => 0],
        ];

        foreach ($marketingData as $data) {
            if ($data->brand_ambassador_spend > 0) {
                $channels['brand_ambassadors']['total'] += $data->brand_ambassador_spend;
                $channels['brand_ambassadors']['count']++;
            }
            if ($data->ads_spend > 0) {
                $channels['ads']['total'] += $data->ads_spend;
                $channels['ads']['count']++;
            }
            if ($data->social_media_spend > 0) {
                $channels['social_media']['total'] += $data->social_media_spend;
                $channels['social_media']['count']++;
            }
            if ($data->events_spend > 0) {
                $channels['events']['total'] += $data->events_spend;
                $channels['events']['count']++;
            }
            if ($data->other_spend > 0) {
                $channels['other']['total'] += $data->other_spend;
                $channels['other']['count']++;
            }
        }

        // Calculate averages
        foreach ($channels as $key => $channel) {
            $channels[$key]['average'] = $channel['count'] > 0 ? $channel['total'] / $channel['count'] : 0;
        }

        return $channels;
    }

    /**
     * Get best performing channel historically
     */
    private function getBestPerformingChannel($historicalData)
    {
        $channelROI = [
            'Brand Ambassadors' => [],
            'Ads' => [],
            'Social Media' => [],
            'Events' => [],
            'Other' => [],
        ];

        foreach ($historicalData as $data) {
            if ($data->brand_ambassador_spend > 0) {
                $channelROI['Brand Ambassadors'][] = $data->roi_percentage;
            }
            if ($data->ads_spend > 0) {
                $channelROI['Ads'][] = $data->roi_percentage;
            }
            if ($data->social_media_spend > 0) {
                $channelROI['Social Media'][] = $data->roi_percentage;
            }
            if ($data->events_spend > 0) {
                $channelROI['Events'][] = $data->roi_percentage;
            }
            if ($data->other_spend > 0) {
                $channelROI['Other'][] = $data->roi_percentage;
            }
        }

        $avgROI = [];
        foreach ($channelROI as $channel => $rois) {
            if (count($rois) > 0) {
                $avgROI[$channel] = array_sum($rois) / count($rois);
            }
        }

        if (empty($avgROI)) {
            return 'Not enough data';
        }

        arsort($avgROI);
        return array_key_first($avgROI) . ' (Avg ROI: ' . number_format(reset($avgROI), 2) . '%)';
    }
}
