<?php

namespace App\Http\Controllers;

use App\Services\GoogleSheetsService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Crypt;
use Carbon\Carbon;

class GoogleAuthController extends Controller
{
    protected $googleSheetsService;

    public function __construct(GoogleSheetsService $googleSheetsService)
    {
        $this->googleSheetsService = $googleSheetsService;
    }

    /**
     * Redirect user to Google OAuth consent screen
     */
    public function redirectToGoogle()
    {
        $user = auth()->user();

        // Only sellers can use Google Sheets sync
        if (!$user->isSeller()) {
            return redirect()->back()->with('error', 'Only sellers can use Google Sheets sync.');
        }

        try {
            if (!class_exists('\Google\Client')) {
                return redirect()->back()->with('error', 'Google API Client library not installed. Please contact administrator.');
            }

            $client = new \Google\Client();
            $client->setApplicationName(config('app.name') . ' Order Sync');
            $client->setScopes([
                \Google\Service\Sheets::SPREADSHEETS,
                \Google\Service\Drive::DRIVE_FILE,
                'https://www.googleapis.com/auth/userinfo.email',
            ]);
            $client->setClientId(config('services.google.client_id'));
            $client->setClientSecret(config('services.google.client_secret'));
            $client->setRedirectUri(config('services.google.redirect'));
            $client->setAccessType('offline');
            $client->setPrompt('consent');

            // Generate state parameter for security
            $state = base64_encode(json_encode([
                'user_id' => $user->id,
                'timestamp' => now()->timestamp,
            ]));

            $client->setState($state);

            $authUrl = $client->createAuthUrl();

            return redirect($authUrl);

        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Failed to initiate Google authentication: ' . $e->getMessage());
        }
    }

    /**
     * Handle Google OAuth callback
     */
    public function handleGoogleCallback(Request $request)
    {
        $user = auth()->user();

        // Check for error from Google
        if ($request->has('error')) {
            return redirect()->route('settings.google-sheets')
                ->with('error', 'Google authentication was cancelled or failed.');
        }

        // Validate state parameter
        $state = $request->input('state');
        if (!$state) {
            return redirect()->route('settings.google-sheets')
                ->with('error', 'Invalid authentication state.');
        }

        try {
            $stateData = json_decode(base64_decode($state), true);

            // Verify user ID matches
            if ($stateData['user_id'] != $user->id) {
                return redirect()->route('settings.google-sheets')
                    ->with('error', 'Authentication state mismatch.');
            }

            // Exchange authorization code for access token
            $client = new \Google\Client();
            $client->setClientId(config('services.google.client_id'));
            $client->setClientSecret(config('services.google.client_secret'));
            $client->setRedirectUri(config('services.google.redirect'));

            $code = $request->input('code');
            $token = $client->fetchAccessTokenWithAuthCode($code);

            if (isset($token['error'])) {
                throw new \Exception($token['error_description'] ?? 'Failed to fetch access token');
            }

            // Get user email from Google
            $client->setAccessToken($token);
            $oauth2 = new \Google\Service\Oauth2($client);
            $googleUser = $oauth2->userinfo->get();

            // Store tokens (encrypted)
            $user->google_access_token = Crypt::encryptString(json_encode($token));
            $user->google_refresh_token = isset($token['refresh_token'])
                ? Crypt::encryptString($token['refresh_token'])
                : $user->google_refresh_token; // Keep existing if not provided
            $user->google_token_expires_at = isset($token['expires_in'])
                ? Carbon::now()->addSeconds($token['expires_in'])
                : null;
            $user->google_email = $googleUser->email;
            $user->google_sheets_sync_enabled = true;
            $user->save();

            // Create Google Sheet
            $spreadsheetId = $this->googleSheetsService->createSheet($user);

            return redirect()->route('settings.google-sheets')
                ->with('success', "Google Sheets connected successfully! Your orders will now sync to your Google Sheet.");

        } catch (\Exception $e) {
            \Log::error('Google OAuth callback error', [
                'user_id' => $user->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return redirect()->route('settings.google-sheets')
                ->with('error', 'Failed to complete Google authentication: ' . $e->getMessage());
        }
    }

    /**
     * Disconnect Google account
     */
    public function disconnect()
    {
        $user = auth()->user();

        if (!$user->google_sheets_sync_enabled) {
            return redirect()->back()->with('info', 'Google Sheets is not connected.');
        }

        try {
            // Revoke access and clear data
            $this->googleSheetsService->revokeAccess($user);

            return redirect()->route('settings.google-sheets')
                ->with('success', 'Google account disconnected successfully. Your existing Google Sheet is still in your Drive.');

        } catch (\Exception $e) {
            \Log::error('Google disconnect error', [
                'user_id' => $user->id,
                'error' => $e->getMessage()
            ]);

            return redirect()->back()
                ->with('error', 'Failed to disconnect Google account: ' . $e->getMessage());
        }
    }

    /**
     * Sync last 10 orders (quick test/recent sync)
     */
    public function syncRecent()
    {
        $user = auth()->user();

        if (!$user->google_sheets_sync_enabled) {
            return redirect()->back()->with('error', 'Google Sheets is not connected.');
        }

        try {
            // Queue batch sync job (limit to 10 most recent orders)
            \App\Jobs\SyncOrderToGoogleSheets::dispatch($user, null, true, 10);

            return redirect()->route('settings.google-sheets')
                ->with('success', 'Syncing your last 10 orders to Google Sheets. This will take about 10-20 seconds.');

        } catch (\Exception $e) {
            \Log::error('Recent sync error', [
                'user_id' => $user->id,
                'error' => $e->getMessage()
            ]);

            return redirect()->back()
                ->with('error', 'Failed to start sync: ' . $e->getMessage());
        }
    }

    /**
     * Manually trigger full sync of all orders
     */
    public function manualSync()
    {
        $user = auth()->user();

        if (!$user->google_sheets_sync_enabled) {
            return redirect()->back()->with('error', 'Google Sheets is not connected.');
        }

        try {
            // Queue batch sync job (limit to 100 most recent orders)
            \App\Jobs\SyncOrderToGoogleSheets::dispatch($user, null, true, 100);

            return redirect()->route('settings.google-sheets')
                ->with('success', 'Manual sync started. Your last 100 orders are being synced to Google Sheets (this may take 2-3 minutes).');

        } catch (\Exception $e) {
            \Log::error('Manual sync error', [
                'user_id' => $user->id,
                'error' => $e->getMessage()
            ]);

            return redirect()->back()
                ->with('error', 'Failed to start manual sync: ' . $e->getMessage());
        }
    }

    /**
     * Show import preview (dry-run) - disaster recovery
     */
    public function showImportPreview(Request $request)
    {
        $user = auth()->user();

        // Only admin can use import feature
        if (!$user->isAdmin()) {
            abort(403, 'Only administrators can import orders from Google Sheets.');
        }

        // Get target user (defaults to authenticated user if not admin)
        $targetUserId = $request->input('user_id', $user->id);

        if (!$user->google_sheets_sync_enabled) {
            return redirect()->back()->with('error', 'Google Sheets is not connected for this user.');
        }

        try {
            // Run dry-run to get preview
            \Artisan::call('orders:import-missing', [
                'user_id' => $targetUserId,
                '--dry-run' => true,
            ]);

            $output = \Artisan::output();

            return view('settings.google-sheets-import-preview', [
                'output' => $output,
                'targetUserId' => $targetUserId,
            ]);

        } catch (\Exception $e) {
            \Log::error('Import preview error', [
                'user_id' => $user->id,
                'target_user_id' => $targetUserId,
                'error' => $e->getMessage()
            ]);

            return redirect()->back()
                ->with('error', 'Failed to generate import preview: ' . $e->getMessage());
        }
    }

    /**
     * Import orders from Google Sheets - disaster recovery
     */
    public function importOrders(Request $request)
    {
        $user = auth()->user();

        // Only admin can use import feature
        if (!$user->isAdmin()) {
            abort(403, 'Only administrators can import orders from Google Sheets.');
        }

        $validated = $request->validate([
            'user_id' => 'required|integer|exists:users,id',
        ]);

        $targetUserId = $validated['user_id'];
        $targetUser = \App\Models\User::find($targetUserId);

        if (!$targetUser->google_sheets_sync_enabled) {
            return redirect()->back()->with('error', 'Google Sheets is not connected for this user.');
        }

        try {
            // Run actual import (not dry-run)
            \Artisan::call('orders:import-missing', [
                'user_id' => $targetUserId,
            ]);

            $output = \Artisan::output();

            \Log::info('Orders imported from Google Sheets', [
                'admin_user_id' => $user->id,
                'admin_email' => $user->email,
                'target_user_id' => $targetUserId,
                'target_email' => $targetUser->email,
            ]);

            return redirect()->route('settings.google-sheets')
                ->with('success', 'Orders imported successfully! Check the logs for details.')
                ->with('info', $output);

        } catch (\Exception $e) {
            \Log::error('Import orders error', [
                'admin_user_id' => $user->id,
                'target_user_id' => $targetUserId,
                'error' => $e->getMessage()
            ]);

            return redirect()->back()
                ->with('error', 'Failed to import orders: ' . $e->getMessage());
        }
    }
}
