<?php

namespace App\Http\Controllers;

use App\Models\WebhookSource;
use App\Models\WebhookLog;
use App\Services\WebhookOrderMapper;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class GenericWebhookController extends Controller
{
    protected $orderMapper;

    public function __construct(WebhookOrderMapper $orderMapper)
    {
        $this->orderMapper = $orderMapper;
    }

    /**
     * Handle incoming generic webhook
     */
    public function handle(Request $request, $token)
    {
        // Find webhook source by token
        $webhookSource = WebhookSource::where('webhook_token', $token)->first();

        if (!$webhookSource) {
            Log::warning('Webhook received with invalid token', ['token' => $token]);
            return response()->json(['error' => 'Invalid webhook token'], 404);
        }

        if (!$webhookSource->is_active) {
            Log::warning('Webhook received for inactive source', ['source' => $webhookSource->name]);
            return response()->json(['error' => 'Webhook source is inactive'], 403);
        }

        // Get payload (support both JSON and form data)
        $payload = $request->all();
        if (empty($payload)) {
            $payload = json_decode($request->getContent(), true) ?? [];
        }

        // Extract event type
        $eventType = $payload['event'] ?? $payload['event_type'] ?? $payload['type'] ?? 'unknown';

        // Create webhook log
        $webhookLog = WebhookLog::create([
            'webhook_source_id' => $webhookSource->id,
            'event_type' => $eventType,
            'payload' => $payload,
            'status' => 'pending',
            'ip_address' => $request->ip(),
        ]);

        try {
            // Only process payment success events (or all events if configured)
            $processableEvents = ['payment_success', 'payment.success', 'charge.succeeded', 'order.created', 'form.submit'];

            if (in_array(strtolower($eventType), $processableEvents) || $webhookSource->settings['process_all_events'] ?? false) {
                // Create order from webhook payload
                $order = $this->orderMapper->createOrder($payload, $webhookSource);

                // Update webhook log
                $webhookLog->update([
                    'status' => 'success',
                    'order_id' => $order->id,
                ]);

                Log::info('Order created from webhook', [
                    'webhook_source' => $webhookSource->name,
                    'order_id' => $order->id,
                    'order_number' => $order->order_number
                ]);

                return response()->json([
                    'success' => true,
                    'message' => 'Order created successfully',
                    'order_id' => $order->id,
                    'order_number' => $order->order_number
                ], 200);
            } else {
                // Event type not configured for processing
                $webhookLog->update([
                    'status' => 'ignored',
                    'error_message' => 'Event type not configured for processing: ' . $eventType
                ]);

                return response()->json([
                    'success' => true,
                    'message' => 'Webhook received but not processed (event type: ' . $eventType . ')'
                ], 200);
            }

        } catch (\Exception $e) {
            // Log error
            $webhookLog->update([
                'status' => 'failed',
                'error_message' => $e->getMessage()
            ]);

            Log::error('Webhook processing failed', [
                'webhook_source' => $webhookSource->name,
                'error' => $e->getMessage(),
                'payload' => $payload
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to process webhook: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Test webhook endpoint (for debugging)
     */
    public function test(Request $request, $token)
    {
        $webhookSource = WebhookSource::where('webhook_token', $token)->first();

        if (!$webhookSource) {
            return response()->json(['error' => 'Invalid webhook token'], 404);
        }

        return response()->json([
            'success' => true,
            'message' => 'Webhook endpoint is working!',
            'webhook_source' => $webhookSource->name,
            'webhook_url' => $webhookSource->webhook_url,
            'payload_received' => $request->all()
        ]);
    }
}
