<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\HistoricalSale;
use App\Models\SalesTarget;
use App\Models\Order;
use Illuminate\Support\Facades\DB;
use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Writer\Csv;

class ForecastingController extends Controller
{
    /**
     * Dashboard - Main forecasting view
     */
    public function index(Request $request)
    {
        $year = $request->get('year', date('Y'));

        // Get all historical data
        $historicalData = HistoricalSale::orderBy('year', 'desc')
            ->orderBy('month')
            ->get();

        // Get current year actual sales from orders table
        $currentYearSales = $this->getCurrentYearSales($year);

        // Get sales targets
        $targets = SalesTarget::whereNull('user_id') // company-wide only
            ->where('year', $year)
            ->orderBy('month')
            ->get();

        // Calculate summary stats
        $summary = $this->calculateSummary($year);

        // Get sales breakdown data for charts
        $salesBreakdown = $this->getSalesBreakdownForYear($year);

        // Get forecasting data
        $forecast = $this->generateForecast($year);

        return view('forecasting.index', compact(
            'historicalData',
            'currentYearSales',
            'targets',
            'summary',
            'year',
            'salesBreakdown',
            'forecast'
        ));
    }

    /**
     * Show import page
     */
    public function showImport()
    {
        $existingData = HistoricalSale::orderBy('year')->orderBy('month')->get();

        return view('forecasting.import', compact('existingData'));
    }

    /**
     * Handle CSV/XLSX import
     */
    public function import(Request $request)
    {
        $request->validate([
            'file' => 'required|file|mimes:csv,xlsx,xls|max:10240', // 10MB max
        ]);

        try {
            $file = $request->file('file');
            $extension = $file->getClientOriginalExtension();

            // Load spreadsheet
            $spreadsheet = IOFactory::load($file->getRealPath());
            $worksheet = $spreadsheet->getActiveSheet();
            $rows = $worksheet->toArray();

            // Skip header row
            $header = array_shift($rows);

            $imported = 0;
            $updated = 0;
            $errors = [];

            foreach ($rows as $index => $row) {
                // Skip empty rows
                if (empty($row[0]) || empty($row[1])) {
                    continue;
                }

                $year = (int) $row[0];
                $month = (int) $row[1];
                $revenue = (float) str_replace(',', '', $row[2] ?? 0);
                $orders = (int) $row[3];
                $expenses = (float) str_replace(',', '', $row[4] ?? 0);
                $stokisSales = (float) str_replace(',', '', $row[5] ?? 0);
                $agentSales = (float) str_replace(',', '', $row[6] ?? 0);
                $hqSales = (float) str_replace(',', '', $row[7] ?? 0);
                $notes = $row[8] ?? '';

                // Validation
                if ($year < 2000 || $year > 2030) {
                    $errors[] = "Row " . ($index + 2) . ": Invalid year ($year)";
                    continue;
                }

                if ($month < 1 || $month > 12) {
                    $errors[] = "Row " . ($index + 2) . ": Invalid month ($month)";
                    continue;
                }

                if ($revenue <= 0 || $orders <= 0) {
                    $errors[] = "Row " . ($index + 2) . ": Revenue and orders must be > 0";
                    continue;
                }

                // Insert or update
                $existing = HistoricalSale::where('year', $year)
                    ->where('month', $month)
                    ->first();

                if ($existing) {
                    $existing->update([
                        'revenue' => $revenue,
                        'orders' => $orders,
                        'expenses' => $expenses,
                        'stokis_sales' => $stokisSales,
                        'agent_sales' => $agentSales,
                        'hq_sales' => $hqSales,
                        'notes' => $notes,
                        'import_source' => $extension,
                    ]);
                    $updated++;
                } else {
                    HistoricalSale::create([
                        'year' => $year,
                        'month' => $month,
                        'revenue' => $revenue,
                        'orders' => $orders,
                        'expenses' => $expenses,
                        'stokis_sales' => $stokisSales,
                        'agent_sales' => $agentSales,
                        'hq_sales' => $hqSales,
                        'notes' => $notes,
                        'import_source' => $extension,
                        'created_by' => auth()->id(),
                    ]);
                    $imported++;
                }
            }

            $message = "Import complete! Imported: $imported, Updated: $updated";

            if (count($errors) > 0) {
                $message .= ". Errors: " . count($errors);
                return back()->with('warning', $message)->with('errors', $errors);
            }

            return back()->with('success', $message);

        } catch (\Exception $e) {
            \Log::error('Import failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return back()->with('error', 'Import failed: ' . $e->getMessage());
        }
    }

    /**
     * Download CSV template
     */
    public function downloadTemplate(Request $request)
    {
        $format = $request->get('format', 'csv'); // csv or xlsx

        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Headers
        $sheet->setCellValue('A1', 'Year');
        $sheet->setCellValue('B1', 'Month');
        $sheet->setCellValue('C1', 'Revenue');
        $sheet->setCellValue('D1', 'Orders');
        $sheet->setCellValue('E1', 'Expenses');
        $sheet->setCellValue('F1', 'Stokis_Sales');
        $sheet->setCellValue('G1', 'Agent_Sales');
        $sheet->setCellValue('H1', 'HQ_Sales');
        $sheet->setCellValue('I1', 'Notes');

        // Sample data
        $sheet->setCellValue('A2', '2022');
        $sheet->setCellValue('B2', '1');
        $sheet->setCellValue('C2', '45000');
        $sheet->setCellValue('D2', '280');
        $sheet->setCellValue('E2', '5000');
        $sheet->setCellValue('F2', '30000');
        $sheet->setCellValue('G2', '10000');
        $sheet->setCellValue('H2', '5000');
        $sheet->setCellValue('I2', 'Chinese New Year');

        $sheet->setCellValue('A3', '2022');
        $sheet->setCellValue('B3', '2');
        $sheet->setCellValue('C3', '38000');
        $sheet->setCellValue('D3', '245');
        $sheet->setCellValue('E3', '4500');
        $sheet->setCellValue('F3', '25000');
        $sheet->setCellValue('G3', '8000');
        $sheet->setCellValue('H3', '5000');
        $sheet->setCellValue('I3', '');

        // Style headers
        $sheet->getStyle('A1:I1')->getFont()->setBold(true);

        // Auto-size columns
        foreach (range('A', 'I') as $col) {
            $sheet->getColumnDimension($col)->setAutoSize(true);
        }

        // Generate file
        $filename = 'historical_sales_template.' . $format;

        if ($format === 'xlsx') {
            $writer = new Xlsx($spreadsheet);
            $contentType = 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet';
        } else {
            $writer = new Csv($spreadsheet);
            $contentType = 'text/csv';
        }

        // Stream to browser
        header('Content-Type: ' . $contentType);
        header('Content-Disposition: attachment;filename="' . $filename . '"');
        header('Cache-Control: max-age=0');

        $writer->save('php://output');
        exit;
    }

    /**
     * Get year sales from historical_sales (for past years) or orders (for current year)
     */
    private function getCurrentYearSales($year)
    {
        $sales = [];

        // Check if we have historical data for this year
        $historicalData = HistoricalSale::where('year', $year)
            ->orderBy('month')
            ->get()
            ->keyBy('month');

        for ($month = 1; $month <= 12; $month++) {
            // First, try to get from historical_sales table
            if ($historicalData->has($month)) {
                $historical = $historicalData[$month];
                $sales[$month] = [
                    'month' => $month,
                    'orders' => $historical->orders,
                    'revenue' => $historical->revenue,
                    'expenses' => $historical->expenses,
                    'stokis_sales' => $historical->stokis_sales ?? 0,
                    'agent_sales' => $historical->agent_sales ?? 0,
                    'hq_sales' => $historical->hq_sales ?? 0,
                    'aov' => $historical->aov,
                    'source' => 'historical', // Indicate data source
                ];
            } else {
                // Fallback to orders table + TikTok Shop (for current year/month data)
                $startDate = "$year-" . str_pad($month, 2, '0', STR_PAD_LEFT) . "-01";
                $endDate = date('Y-m-t', strtotime($startDate));

                // Get Orders table data
                $orderResult = Order::whereBetween('date_created', [$startDate, $endDate])
                    ->whereIn('status', ['processing', 'approval', 'printed', 'completed'])
                    ->select(
                        DB::raw('COUNT(*) as orders'),
                        DB::raw('SUM(total) as revenue')
                    )
                    ->first();

                $ordersCount = $orderResult->orders ?? 0;
                $ordersRevenue = $orderResult->revenue ?? 0;

                // Get TikTok Shop data
                $tiktokResult = \App\Models\TikTokTransaction::whereYear('settled_date', $year)
                    ->whereMonth('settled_date', $month)
                    ->where('type', 'Order')
                    ->select(
                        DB::raw('COUNT(*) as orders'),
                        DB::raw('SUM(total_revenue) as revenue')
                    )
                    ->first();

                $tiktokCount = $tiktokResult->orders ?? 0;
                $tiktokRevenue = $tiktokResult->revenue ?? 0;

                // Combine totals
                $totalOrders = $ordersCount + $tiktokCount;
                $totalRevenue = $ordersRevenue + $tiktokRevenue;

                // Note: Channel breakdown (HQ/Stokis/Agent) set to 0 for auto-calculated months
                // User should input this data manually if needed, or it can be added later
                // For now, we show total revenue from all sources (Orders + TikTok)

                $sales[$month] = [
                    'month' => $month,
                    'orders' => $totalOrders,
                    'revenue' => $totalRevenue,
                    'expenses' => 0,
                    'stokis_sales' => 0, // Can be manually added later if needed
                    'agent_sales' => 0,  // Can be manually added later if needed
                    'hq_sales' => $totalRevenue, // Default: assume all sales are HQ until specified otherwise
                    'aov' => $totalOrders > 0 ? $totalRevenue / $totalOrders : 0,
                    'source' => 'auto-calculated', // Indicate data source
                ];
            }
        }

        return $sales;
    }

    /**
     * Get sales breakdown for charting
     */
    private function getSalesBreakdownForYear($year)
    {
        // Use getCurrentYearSales to get data (which can fallback to orders table)
        $currentYearSales = $this->getCurrentYearSales($year);

        $breakdown = collect();

        foreach ($currentYearSales as $month => $data) {
            $breakdown->push([
                'month' => date('M', mktime(0, 0, 0, $month, 1)),
                'stokis' => (float) ($data['stokis_sales'] ?? 0),
                'agent' => (float) ($data['agent_sales'] ?? 0),
                'hq' => (float) ($data['hq_sales'] ?? 0),
                'total' => (float) $data['revenue'],
            ]);
        }

        return $breakdown;
    }

    /**
     * Generate forecast using historical data
     */
    private function generateForecast($year)
    {
        // Get historical data for trend analysis
        $historicalData = HistoricalSale::where('year', '<', $year)
            ->orderBy('year')
            ->orderBy('month')
            ->get();

        if ($historicalData->count() < 12) {
            return [
                'next_month_revenue' => 0,
                'next_month_orders' => 0,
                'year_end_projection' => 0,
                'confidence' => 'low',
                'trend' => 'insufficient_data',
            ];
        }

        // Calculate monthly averages and trends
        $monthlyAverages = [];
        for ($m = 1; $m <= 12; $m++) {
            $monthData = $historicalData->where('month', $m);
            if ($monthData->count() > 0) {
                $monthlyAverages[$m] = [
                    'avg_revenue' => $monthData->avg('revenue'),
                    'avg_orders' => $monthData->avg('orders'),
                    'growth_rate' => $this->calculateGrowthRate($monthData),
                ];
            }
        }

        // Get current year data
        $currentYearData = HistoricalSale::where('year', $year)
            ->orderBy('month', 'desc')
            ->get();

        // Calculate YoY growth
        $lastYear = $year - 1;
        $lastYearTotal = HistoricalSale::where('year', $lastYear)->sum('revenue');
        $currentYearTotal = $currentYearData->sum('revenue');
        $yoyGrowth = $lastYearTotal > 0 ? (($currentYearTotal - $lastYearTotal) / $lastYearTotal) : 0;

        // Predict next month
        $lastMonth = $currentYearData->first();
        $nextMonth = $lastMonth ? ($lastMonth->month % 12) + 1 : 1;

        $nextMonthForecast = [
            'revenue' => 0,
            'orders' => 0,
        ];

        if (isset($monthlyAverages[$nextMonth])) {
            $baseRevenue = $monthlyAverages[$nextMonth]['avg_revenue'];
            $growthRate = $monthlyAverages[$nextMonth]['growth_rate'];

            // Apply growth rate
            $nextMonthForecast['revenue'] = $baseRevenue * (1 + $growthRate + $yoyGrowth);
            $nextMonthForecast['orders'] = $monthlyAverages[$nextMonth]['avg_orders'] * (1 + $growthRate + $yoyGrowth);
        }

        // Year-end projection
        $yearEndProjection = $currentYearTotal;
        $remainingMonths = 12 - $currentYearData->count();

        for ($m = 1; $m <= $remainingMonths; $m++) {
            $monthNum = ($lastMonth->month + $m - 1) % 12 + 1;
            if (isset($monthlyAverages[$monthNum])) {
                $yearEndProjection += $monthlyAverages[$monthNum]['avg_revenue'] * (1 + $yoyGrowth);
            }
        }

        // Determine confidence level
        $confidence = 'medium';
        if ($historicalData->count() >= 36) {
            $confidence = 'high';
        } elseif ($historicalData->count() < 24) {
            $confidence = 'low';
        }

        // Determine trend
        $trend = $yoyGrowth > 0.05 ? 'growing' : ($yoyGrowth < -0.05 ? 'declining' : 'stable');

        return [
            'next_month_revenue' => $nextMonthForecast['revenue'],
            'next_month_orders' => $nextMonthForecast['orders'],
            'year_end_projection' => $yearEndProjection,
            'yoy_growth' => $yoyGrowth * 100,
            'confidence' => $confidence,
            'trend' => $trend,
            'monthly_averages' => $monthlyAverages,
        ];
    }

    /**
     * Calculate growth rate for a set of data
     */
    private function calculateGrowthRate($data)
    {
        if ($data->count() < 2) {
            return 0;
        }

        $values = $data->sortBy('year')->pluck('revenue')->values();
        $first = $values->first();
        $last = $values->last();

        if ($first <= 0) {
            return 0;
        }

        $periods = $values->count() - 1;
        return $periods > 0 ? (pow($last / $first, 1 / $periods) - 1) : 0;
    }

    /**
     * Show expense update form
     */
    public function showExpenseUpdate(Request $request)
    {
        $year = $request->get('year', date('Y'));
        $month = $request->get('month', date('n'));

        // Get or create historical sale record
        $record = HistoricalSale::firstOrCreate(
            ['year' => $year, 'month' => $month],
            [
                'revenue' => 0,
                'orders' => 0,
                'expenses' => 0,
                'created_by' => auth()->id(),
            ]
        );

        // If revenue is 0, try to get from orders table
        if ($record->revenue == 0) {
            $startDate = "$year-" . str_pad($month, 2, '0', STR_PAD_LEFT) . "-01";
            $endDate = date('Y-m-t', strtotime($startDate));

            $result = Order::whereBetween('date_created', [$startDate, $endDate])
                ->whereIn('status', ['processing', 'approval', 'printed', 'completed'])
                ->select(
                    DB::raw('COUNT(*) as orders'),
                    DB::raw('SUM(total) as revenue')
                )
                ->first();

            $record->orders = $result->orders ?? 0;
            $record->revenue = $result->revenue ?? 0;
        }

        // Get all months for current year
        $currentYearData = [];
        for ($m = 1; $m <= 12; $m++) {
            $existing = HistoricalSale::where('year', $year)->where('month', $m)->first();

            if ($existing) {
                $currentYearData[$m] = $existing;
            } else {
                // Get from orders table
                $startDate = "$year-" . str_pad($m, 2, '0', STR_PAD_LEFT) . "-01";
                $endDate = date('Y-m-t', strtotime($startDate));

                $result = Order::whereBetween('date_created', [$startDate, $endDate])
                    ->whereIn('status', ['processing', 'approval', 'printed', 'completed'])
                    ->select(
                        DB::raw('COUNT(*) as orders'),
                        DB::raw('SUM(total) as revenue')
                    )
                    ->first();

                $currentYearData[$m] = (object)[
                    'year' => $year,
                    'month' => $m,
                    'orders' => $result->orders ?? 0,
                    'revenue' => $result->revenue ?? 0,
                    'expenses' => 0,
                    'expense_updated_at' => null,
                    'source' => 'orders'
                ];
            }
        }

        return view('forecasting.expense-update', compact('year', 'month', 'record', 'currentYearData'));
    }

    /**
     * Update expenses for a month
     */
    public function updateExpense(Request $request)
    {
        $request->validate([
            'year' => 'required|integer|min:2000|max:2030',
            'month' => 'required|integer|min:1|max:12',
            'expenses' => 'required|numeric|min:0',
        ]);

        $year = $request->year;
        $month = $request->month;
        $expenses = $request->expenses;

        // Get or create historical sale record
        $record = HistoricalSale::where('year', $year)
            ->where('month', $month)
            ->first();

        if (!$record) {
            // Get revenue and orders from orders table
            $startDate = "$year-" . str_pad($month, 2, '0', STR_PAD_LEFT) . "-01";
            $endDate = date('Y-m-t', strtotime($startDate));

            $result = Order::whereBetween('date_created', [$startDate, $endDate])
                ->whereIn('status', ['processing', 'approval', 'printed', 'completed'])
                ->select(
                    DB::raw('COUNT(*) as orders'),
                    DB::raw('SUM(total) as revenue')
                )
                ->first();

            $orders = $result->orders ?? 0;
            $revenue = $result->revenue ?? 0;

            // If no orders yet, set minimum to avoid confusion (will show RM 0 with 0 orders)
            // The AOV calculation is now safe (returns 0 when orders = 0)

            $record = HistoricalSale::create([
                'year' => $year,
                'month' => $month,
                'revenue' => $revenue,
                'orders' => $orders,
                'expenses' => $expenses,
                'created_by' => auth()->id(),
                'expense_updated_at' => now(),
                'expense_updated_by' => auth()->id(),
            ]);

            $monthName = date('F', mktime(0, 0, 0, $month, 1));
            if ($orders == 0) {
                return back()->with('success', "Expense for $monthName $year added: RM " . number_format($expenses, 2) . " (No orders data yet for this month)");
            }

            return back()->with('success', "Expense for $monthName $year added: RM " . number_format($expenses, 2) . " (Revenue: RM " . number_format($revenue, 2) . ", Orders: $orders)");
        }

        // Update existing record
        $record->update([
            'expenses' => $expenses,
            'expense_updated_at' => now(),
            'expense_updated_by' => auth()->id(),
        ]);

        return back()->with('success', "Expense for " . date('F', mktime(0, 0, 0, $month, 1)) . " $year updated: RM " . number_format($expenses, 2));
    }

    /**
     * Update channel sales (Stokis/Agent) for a specific month
     */
    public function updateChannelSales(Request $request)
    {
        $request->validate([
            'year' => 'required|integer|min:2000|max:2030',
            'month' => 'required|integer|min:1|max:12',
            'channel' => 'required|in:stokis_sales,agent_sales',
            'amount' => 'required|numeric|min:0',
        ]);

        $year = $request->year;
        $month = $request->month;
        $channel = $request->channel;
        $amount = $request->amount;

        // Get or create historical sale record
        $record = HistoricalSale::where('year', $year)
            ->where('month', $month)
            ->first();

        if (!$record) {
            // Get revenue and orders from orders table for current month
            $startDate = "$year-" . str_pad($month, 2, '0', STR_PAD_LEFT) . "-01";
            $endDate = date('Y-m-t', strtotime($startDate));

            // Get Orders table data
            $orderResult = Order::whereBetween('date_created', [$startDate, $endDate])
                ->whereIn('status', ['processing', 'approval', 'printed', 'completed'])
                ->select(
                    DB::raw('COUNT(*) as orders'),
                    DB::raw('SUM(total) as revenue')
                )
                ->first();

            $orders = $orderResult->orders ?? 0;
            $revenue = $orderResult->revenue ?? 0;

            // Get TikTok Shop data
            $tiktokResult = \App\Models\TikTokTransaction::whereYear('settled_date', $year)
                ->whereMonth('settled_date', $month)
                ->where('type', 'Order')
                ->select(
                    DB::raw('COUNT(*) as orders'),
                    DB::raw('SUM(total_revenue) as revenue')
                )
                ->first();

            $tiktokCount = $tiktokResult->orders ?? 0;
            $tiktokRevenue = $tiktokResult->revenue ?? 0;

            // Combine totals
            $totalOrders = $orders + $tiktokCount;
            $hqRevenue = $revenue + $tiktokRevenue; // HQ = system orders

            // Create new record with channel breakdown
            // HQ sales come from system, Stokis/Agent are manual input
            $stokisSales = $channel === 'stokis_sales' ? $amount : 0;
            $agentSales = $channel === 'agent_sales' ? $amount : 0;

            // Total Revenue = HQ + Stokis + Agent
            $finalRevenue = $hqRevenue + $stokisSales + $agentSales;

            $record = HistoricalSale::create([
                'year' => $year,
                'month' => $month,
                'revenue' => $finalRevenue,
                'orders' => $totalOrders,
                'expenses' => 0,
                'stokis_sales' => $stokisSales,
                'agent_sales' => $agentSales,
                'hq_sales' => $hqRevenue,
                'created_by' => auth()->id(),
            ]);
        } else {
            // Update existing record
            $record->update([
                $channel => $amount,
            ]);
        }

        // Recalculate total revenue (Stokis + Agent + HQ)
        $record->refresh();
        $totalRevenue = ($record->stokis_sales ?? 0) + ($record->agent_sales ?? 0) + ($record->hq_sales ?? 0);

        // Update revenue
        $record->update([
            'revenue' => $totalRevenue,
        ]);

        $channelName = $channel === 'stokis_sales' ? 'Stokis' : 'Agent';
        $monthName = date('F', mktime(0, 0, 0, $month, 1));

        // Recalculate for response
        $netProfit = $totalRevenue - ($record->expenses ?? 0);
        $aov = $record->orders > 0 ? $totalRevenue / $record->orders : 0;

        return response()->json([
            'success' => true,
            'message' => "$channelName sales for $monthName $year updated: RM " . number_format($amount, 2),
            'formatted_value' => $this->formatCompactValue($amount),
            'updated_hq' => $record->hq_sales ?? 0,
            'formatted_hq' => $this->formatCompactValue($record->hq_sales ?? 0),
            'updated_revenue' => $totalRevenue,
            'formatted_revenue' => $this->formatCompactValue($totalRevenue),
            'updated_net' => $netProfit,
            'formatted_net' => $this->formatCompactValue($netProfit),
            'updated_aov' => $aov,
            'formatted_aov' => number_format($aov, 0),
        ]);
    }

    /**
     * Update orders count for a specific month
     */
    public function updateOrders(Request $request)
    {
        $request->validate([
            'year' => 'required|integer|min:2000|max:2030',
            'month' => 'required|integer|min:1|max:12',
            'orders' => 'required|integer|min:0',
        ]);

        $year = $request->year;
        $month = $request->month;
        $orders = $request->orders;

        // Get or create historical sale record
        $record = HistoricalSale::where('year', $year)
            ->where('month', $month)
            ->first();

        if (!$record) {
            // Get revenue from orders table for current month
            $startDate = "$year-" . str_pad($month, 2, '0', STR_PAD_LEFT) . "-01";
            $endDate = date('Y-m-t', strtotime($startDate));

            // Get Orders table data
            $orderResult = Order::whereBetween('date_created', [$startDate, $endDate])
                ->whereIn('status', ['processing', 'approval', 'printed', 'completed'])
                ->select(
                    DB::raw('SUM(total) as revenue')
                )
                ->first();

            $revenue = $orderResult->revenue ?? 0;

            // Get TikTok Shop data
            $tiktokResult = \App\Models\TikTokTransaction::whereYear('settled_date', $year)
                ->whereMonth('settled_date', $month)
                ->where('type', 'Order')
                ->select(
                    DB::raw('SUM(total_revenue) as revenue')
                )
                ->first();

            $tiktokRevenue = $tiktokResult->revenue ?? 0;

            // Combine totals
            $hqRevenue = $revenue + $tiktokRevenue;

            // Create new record
            $record = HistoricalSale::create([
                'year' => $year,
                'month' => $month,
                'revenue' => $hqRevenue,
                'orders' => $orders,
                'expenses' => 0,
                'stokis_sales' => 0,
                'agent_sales' => 0,
                'hq_sales' => $hqRevenue,
                'created_by' => auth()->id(),
            ]);
        } else {
            // Update existing record
            $record->update([
                'orders' => $orders,
            ]);
        }

        $record->refresh();

        // Recalculate AOV
        $aov = $orders > 0 ? $record->revenue / $orders : 0;
        $netProfit = $record->revenue - ($record->expenses ?? 0);

        $monthName = date('F', mktime(0, 0, 0, $month, 1));

        return response()->json([
            'success' => true,
            'message' => "Orders for $monthName $year updated: " . number_format($orders),
            'updated_orders' => $orders,
            'formatted_orders' => number_format($orders),
            'updated_aov' => $aov,
            'formatted_aov' => number_format($aov, 0),
            'updated_net' => $netProfit,
            'formatted_net' => $this->formatCompactValue($netProfit),
        ]);
    }

    /**
     * Format value to compact notation (K, M)
     */
    private function formatCompactValue($value)
    {
        if ($value >= 1000000) {
            return number_format($value / 1000000, 2) . 'M';
        } elseif ($value >= 1000) {
            return number_format($value / 1000, 1) . 'K';
        } else {
            return number_format($value, 2);
        }
    }

    /**
     * Get Prophet forecast using Python
     */
    public function getProphetForecast(Request $request)
    {
        $periods = $request->get('periods', 6);

        // Get historical data
        $historicalData = HistoricalSale::orderBy('year')
            ->orderBy('month')
            ->get()
            ->map(function ($record) {
                return [
                    'year' => $record->year,
                    'month' => $record->month,
                    'revenue' => (float) $record->revenue,
                ];
            });

        if ($historicalData->count() < 12) {
            return response()->json([
                'error' => 'Insufficient data. Need at least 12 months of historical data.',
                'data_points' => $historicalData->count()
            ], 400);
        }

        // Convert to JSON
        $jsonData = json_encode($historicalData->values());

        // Get Python path from config
        $pythonPath = config('prophet.python_path', env('PYTHON_PATH', '/var/www/mamadil.team/venv/bin/python3.9'));
        $pythonScript = config('prophet.script_path', env('PROPHET_SCRIPT', base_path('forecast_prophet.py')));

        if (!file_exists($pythonScript)) {
            return response()->json([
                'error' => 'Prophet script not found',
                'path' => $pythonScript,
                'suggestion' => 'Make sure forecast_prophet.py exists in the application root'
            ], 500);
        }

        if (!file_exists($pythonPath)) {
            return response()->json([
                'error' => 'Python executable not found',
                'path' => $pythonPath,
                'suggestion' => 'Check PYTHON_PATH in .env file'
            ], 500);
        }

        // Save JSON data to temporary file
        $tempFile = storage_path('app/forecast_temp_data.json');
        file_put_contents($tempFile, $jsonData);

        try {
            // Execute Python script
            // IMPORTANT: Do NOT use 2>&1 as it mixes stderr with stdout, corrupting JSON output
            $command = escapeshellcmd($pythonPath) . " " . escapeshellarg($pythonScript) . " " . (int)$periods . " < " . escapeshellarg($tempFile);

            \Log::info('Executing Prophet forecast', [
                'command' => $command,
                'python_path' => $pythonPath,
                'script_path' => $pythonScript
            ]);

            $output = shell_exec($command);

            // Clean up temp file
            @unlink($tempFile);

            if (!$output) {
                return response()->json([
                    'error' => 'Failed to execute Prophet forecast. No output received.',
                    'install_command' => 'pip install prophet pandas',
                    'debug_command' => $command,
                    'python_path' => $pythonPath
                ], 500);
            }

            // Trim any whitespace
            $output = trim($output);

            // Try to find JSON in the output (in case there are warnings before it)
            if (preg_match('/\{[\s\S]*\}/', $output, $matches)) {
                $jsonOutput = $matches[0];
            } else {
                $jsonOutput = $output;
            }

            $result = json_decode($jsonOutput, true);

            if (!$result) {
                \Log::error('Prophet forecast output is not valid JSON', [
                    'output' => $output,
                    'command' => $command
                ]);

                return response()->json([
                    'error' => 'Invalid output from Prophet script',
                    'raw_output' => substr($output, 0, 500),
                    'debug_info' => 'Check storage/logs/laravel.log for full output'
                ], 500);
            }

            // Check if there's an error in the result
            if (isset($result['error'])) {
                return response()->json($result, 400);
            }

            return response()->json($result);

        } catch (\Exception $e) {
            // Clean up temp file in case of exception
            @unlink($tempFile);

            \Log::error('Prophet forecast exception', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'error' => 'Exception during forecast: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get AI-powered insights based on forecast and historical data
     */
    public function getAIInsights(Request $request)
    {
        $periods = $request->get('periods', 6);
        $currentYear = date('Y');

        try {
            // Get historical data
            $historicalData = HistoricalSale::orderBy('year', 'desc')
                ->orderBy('month', 'desc')
                ->take(24) // Last 24 months
                ->get();

            if ($historicalData->count() < 3) {
                return response()->json([
                    'error' => 'Insufficient historical data. Need at least 3 months of data for AI analysis.'
                ], 400);
            }

            // Get forecast data (simplified version without Prophet)
            $forecast = $this->generateForecast($currentYear);

            // Get channel breakdown for current year
            $currentYearSales = $this->getCurrentYearSales($currentYear);
            $channelBreakdown = [
                'stokis' => 0,
                'agent' => 0,
                'hq' => 0,
            ];

            foreach ($currentYearSales as $monthData) {
                $channelBreakdown['stokis'] += $monthData['stokis_sales'] ?? 0;
                $channelBreakdown['agent'] += $monthData['agent_sales'] ?? 0;
                $channelBreakdown['hq'] += $monthData['hq_sales'] ?? 0;
            }

            $totalRevenue = $channelBreakdown['stokis'] + $channelBreakdown['agent'] + $channelBreakdown['hq'];

            // Build comprehensive data package for AI
            $dataPackage = [
                'forecast_period' => $periods . ' months',
                'historical_summary' => [
                    'months_of_data' => $historicalData->count(),
                    'recent_monthly_average' => number_format($historicalData->take(6)->avg('revenue'), 2),
                    'trend' => $forecast['trend'],
                    'yoy_growth' => number_format($forecast['yoy_growth'], 1) . '%',
                ],
                'current_year_performance' => [
                    'total_revenue' => number_format($totalRevenue, 2),
                    'stokis_revenue' => number_format($channelBreakdown['stokis'], 2),
                    'agent_revenue' => number_format($channelBreakdown['agent'], 2),
                    'hq_revenue' => number_format($channelBreakdown['hq'], 2),
                    'stokis_percentage' => $totalRevenue > 0 ? number_format(($channelBreakdown['stokis'] / $totalRevenue) * 100, 1) . '%' : '0%',
                    'agent_percentage' => $totalRevenue > 0 ? number_format(($channelBreakdown['agent'] / $totalRevenue) * 100, 1) . '%' : '0%',
                    'hq_percentage' => $totalRevenue > 0 ? number_format(($channelBreakdown['hq'] / $totalRevenue) * 100, 1) . '%' : '0%',
                ],
                'monthly_trends' => $historicalData->take(12)->map(function($record) {
                    return [
                        'period' => $record->month_name . ' ' . $record->year,
                        'revenue' => number_format($record->revenue, 2),
                        'orders' => $record->orders,
                    ];
                })->values()->toArray(),
            ];

            // Call OpenAI for insights
            $insights = $this->generateAIInsights($dataPackage, $periods);

            return response()->json([
                'success' => true,
                'insights' => $insights,
            ]);

        } catch (\Exception $e) {
            \Log::error('AI Insights generation failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'error' => 'Failed to generate insights: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Generate AI insights using OpenAI
     */
    private function generateAIInsights($dataPackage, $periods)
    {
        $apiKey = env('OPENAI_API_KEY');

        if (!$apiKey) {
            return "⚠️ OpenAI API key not configured. Please add OPENAI_API_KEY to .env file to enable AI insights.";
        }

        $prompt = "You are a senior management consultant preparing an EXECUTIVE BUSINESS INTELLIGENCE REPORT for board presentation. ";
        $prompt .= "Apply proven business frameworks and data-driven methodologies. Be CONCRETE, SPECIFIC, and QUANTIFIABLE.\n\n";

        $prompt .= "=== CRITICAL: COMPANY BUSINESS CONTEXT ===\n\n";
        $prompt .= "**Industry & Products:**\n";
        $prompt .= "- **Niche**: Health, Beauty, and Supplement Products\n";
        $prompt .= "- **Core Focus**: Women's Health & Wellness Products\n";
        $prompt .= "- **Product Philosophy**: Family-safe, health-oriented supplements and beauty products\n";
        $prompt .= "- **Target Market**: Malaysian women and families prioritizing health and wellness\n\n";

        $prompt .= "**Sales Channel Structure - ACTUAL CURRENT STATE:**\n\n";

        $prompt .= "1. **Stokis (Distributors)**: ⚠️ CRITICAL REALITY CHECK\n";
        $prompt .= "   - **Total Stokis**: 11-12 stokis only (NOT 100+ like other MLM companies)\n";
        $prompt .= "   - **Active Stokis**: Only 4-5 stokis are ACTIVELY performing (the rest underperforming/dormant)\n";
        $prompt .= "   - **Active Stokis Revenue**: RM 70,000 - RM 250,000/month per active stokis (these are the stars)\n";
        $prompt .= "   - **Average Per Stokis**: ~RM 64,000/month (including dormant ones)\n";
        $prompt .= "   - **Total Stokis Channel Revenue**: ~RM 7 million/year (~RM 583k/month total from all 11-12 stokis)\n";
        $prompt .= "   - **Problem**: Only 4-5 stokis carrying the entire stokis channel. If 1-2 quit, revenue drops significantly.\n";
        $prompt .= "   - **Opportunity**: Reactivate/support the 6-7 underperforming stokis, recruit new stokis\n\n";

        $prompt .= "2. **Agents (Direct Sellers)**: ⚠️ MASSIVE DORMANCY ISSUE\n";
        $prompt .= "   - **Total Agents**: ~5,000 agents registered in system\n";
        $prompt .= "   - **Active Agents**: Only 10-20 agents are ACTIVELY selling (99.6% dormancy rate!)\n";
        $prompt .= "   - **Total Agent Channel Revenue**: ~RM 400,000/year (~RM 33k/month total from all agents)\n";
        $prompt .= "   - **Average Per Active Agent**: RM 2,000-3,000/month (for the 10-20 active ones)\n";
        $prompt .= "   - **CRITICAL PROBLEM**: 4,980+ dormant agents doing ZERO sales\n";
        $prompt .= "   - **HUGE OPPORTUNITY**: If even 1% of dormant agents reactivate (50 agents x RM 3k/month = RM 150k/month = RM 1.8M/year additional revenue!)\n";
        $prompt .= "   - **Low Hanging Fruit**: Reactivating existing agents is CHEAPER than recruiting new ones\n\n";

        $prompt .= "3. **HQ (Direct/E-commerce)**: Website/online sales\n";
        $prompt .= "   - E-commerce EXISTS but NOT successful (expensive to develop vs returns)\n";
        $prompt .= "   - DO NOT suggest heavy e-commerce investment - it's proven not cost-effective for this business\n";
        $prompt .= "   - HQ channel needs to focus on OTHER growth strategies (product diversification, see below)\n\n";

        $prompt .= "**AI MUST UNDERSTAND THIS REALITY:**\n";
        $prompt .= "- We have 11-12 stokis, NOT 120+ stokis. Average per stokis is RM 64k/month.\n";
        $prompt .= "- We have 5,000 agents registered, but 99.6% are DORMANT (only 10-20 active).\n";
        $prompt .= "- Agent reactivation = BIGGEST OPPORTUNITY (thousands of trained agents not selling)\n";
        $prompt .= "- Stokis support = CRITICAL (losing 1 active stokis = losing RM 70k-250k/month)\n\n";

        $prompt .= "**CRITICAL BUSINESS SUSTAINABILITY STRATEGY:**\n";
        $prompt .= "⚠️ **Target: HQ sales must reach 50%+ for business sustainability**\n\n";
        $prompt .= "**Why:** Agents/Stokis are UNCERTAIN - they may quit, switch to competitors, or become inactive.\n";
        $prompt .= "Over-reliance on them (currently 50%+ combined) creates high business risk.\n";
        $prompt .= "If top stokis leave, revenue drops exponentially.\n\n";

        $prompt .= "**Challenge:** We support and respect local market for Agents/Stokis.\n";
        $prompt .= "We CANNOT compete with them in local Malaysian market (unethical, damages relationships).\n\n";

        $prompt .= "**Solution: PRODUCT DIVERSIFICATION for HQ Sales Growth**\n";
        $prompt .= "To grow HQ sales to 50%+ WITHOUT disturbing local agent/stokis market:\n";
        $prompt .= "✅ Focus HQ growth on PRODUCT DIVERSIFICATION (new product categories)\n";
        $prompt .= "✅ Expand beyond women-only products to FAMILY-FRIENDLY products\n";
        $prompt .= "✅ Target broad Malaysian consumer segments\n";
        $prompt .= "✅ Capture market share from existing popular product categories\n";
        $prompt .= "✅ Agents/Stokis can sell ALL products (no competition, expands their offering)\n\n";

        $prompt .= "**SUCCESSFUL EXAMPLE: Coffee Product Launch**\n";
        $prompt .= "✅ **Coffee Product (Healthy Coffee Alternative)** - POSITIVE FEEDBACK received\n";
        $prompt .= "- Not limited to women only - WHOLE FAMILY can enjoy\n";
        $prompt .= "- If women's health products not suitable for someone, coffee provides alternative purchase\n";
        $prompt .= "- No product waste - family members can consume\n";
        $prompt .= "- Broader market appeal = more sales opportunities\n\n";

        $prompt .= "**KEY INSIGHT: Product Diversification Benefits**\n";
        $prompt .= "1. **Reduced Gender Limitation**: Not all products need to be women-specific\n";
        $prompt .= "2. **Family Purchase**: One customer can buy multiple product types\n";
        $prompt .= "3. **Market Share Capture**: Join existing large consumer markets\n";
        $prompt .= "4. **Customer Retention**: If one product doesn't suit, offer alternatives\n";
        $prompt .= "5. **Resilience**: Multiple product categories = stable revenue streams\n\n";

        $prompt .= "**CRITICAL: AI MUST Suggest New Product Categories for Malaysian Market**\n";
        $prompt .= "When analyzing growth opportunities, AI should recommend product categories with:\n";
        $prompt .= "- ✅ **Broad Appeal**: Products enjoyed by multiple demographic segments (men, women, elderly, youth)\n";
        $prompt .= "- ✅ **Large Existing Markets**: Categories Malaysians already consume heavily\n";
        $prompt .= "- ✅ **Daily/Regular Use**: Products with repeat purchase potential\n";
        $prompt .= "- ✅ **Health-Oriented**: Aligns with company's health & wellness positioning\n";
        $prompt .= "- ✅ **Halal-Friendly**: Suitable for Malaysian Muslim-majority market\n\n";

        $prompt .= "**Examples of Product Categories to Consider (AI should suggest specifics):**\n\n";

        $prompt .= "1. **Functional Beverages** (like Coffee example):\n";
        $prompt .= "   - Healthy tea alternatives, functional drinks, wellness beverages\n";
        $prompt .= "   - Market Size: Malaysians consume beverages daily\n";
        $prompt .= "   - Target: Whole family, office workers, health-conscious consumers\n\n";

        $prompt .= "2. **Nutritional Foods**:\n";
        $prompt .= "   - Healthy snacks, protein bars, meal replacements, functional foods\n";
        $prompt .= "   - Market Size: RM billions annually in Malaysia\n";
        $prompt .= "   - Target: Busy professionals, fitness enthusiasts, families\n\n";

        $prompt .= "3. **Wellness Products for Men**:\n";
        $prompt .= "   - Men's health supplements (currently women-focused, expand to men)\n";
        $prompt .= "   - Energy, vitality, fitness, stress management\n";
        $prompt .= "   - Market Size: 50% of population currently underserved\n\n";

        $prompt .= "4. **Family Health Products**:\n";
        $prompt .= "   - Children's health supplements, elderly care products\n";
        $prompt .= "   - Immune support, bone health, eye health (age-agnostic)\n";
        $prompt .= "   - Market Size: Every household needs family health products\n\n";

        $prompt .= "5. **Daily Wellness Essentials**:\n";
        $prompt .= "   - Vitamins, minerals, probiotics (not gender-specific)\n";
        $prompt .= "   - Sleep support, stress relief, immune boosters\n";
        $prompt .= "   - Market Size: Growing post-COVID health awareness\n\n";

        $prompt .= "6. **Personal Care & Beauty for All**:\n";
        $prompt .= "   - Unisex skincare, oral care, hair care\n";
        $prompt .= "   - Natural/organic personal care products\n";
        $prompt .= "   - Market Size: Daily essentials for everyone\n\n";

        $prompt .= "**Product Selection Criteria (AI should evaluate against these):**\n";
        $prompt .= "- Can we capture 1-5% market share of existing large category? (calculate potential RM)\n";
        $prompt .= "- Does it appeal to multiple segments beyond women's health?\n";
        $prompt .= "- Is it consumable/replenishable? (recurring revenue)\n";
        $prompt .= "- Does it fit our health & wellness brand positioning?\n";
        $prompt .= "- Can agents/stokis easily sell it alongside current products?\n";
        $prompt .= "- Is the market already proven? (don't create new category, join existing)\n\n";

        $prompt .= "**CRITICAL: Pricing Psychology & Value Proposition Strategy**\n";
        $prompt .= "When suggesting new product categories, AI MUST structure pricing to make customers feel they're SAVING money:\n\n";

        $prompt .= "**Ideal Pricing Model:**\n";
        $prompt .= "- Box/Pack Price: RM 70-80 per box (upfront investment)\n";
        $prompt .= "- Contains: 15-20 sachets per box\n";
        $prompt .= "- Per Serving Cost: RM 4-5 per consumption/day\n";
        $prompt .= "- Duration: 15-20 days supply (nearly 3 weeks)\n\n";

        $prompt .= "**Value Proposition Formula - ALWAYS Calculate Savings:**\n";
        $prompt .= "Compare daily cost vs common Malaysian expenses:\n\n";

        $prompt .= "Example 1: Health Beverage/Meal Replacement\n";
        $prompt .= "- Your Product: RM 4-5 per serving\n";
        $prompt .= "- VS Meals outside: RM 10-20 per meal (SAVE RM 5-15/day)\n";
        $prompt .= "- VS Mamak/food court: RM 12-15 per meal (SAVE RM 7-10/day)\n";
        $prompt .= "- Daily savings: RM 5-15/day\n";
        $prompt .= "- Monthly savings: RM 150-450 (if replacing 1 meal daily)\n";
        $prompt .= "- Message: 'Replace lunch with RM 5 healthy meal instead of RM 15 outside'\n\n";

        $prompt .= "Example 2: Functional Coffee/Energy Drink\n";
        $prompt .= "- Your Product: RM 4-5 per serving\n";
        $prompt .= "- VS Starbucks/café: RM 12-18 per cup (SAVE RM 7-13/day)\n";
        $prompt .= "- VS Convenience store coffee: RM 6-8 (SAVE RM 1-3/day)\n";
        $prompt .= "- VS Energy drinks: RM 6-10 (SAVE RM 1-5/day)\n";
        $prompt .= "- Monthly savings: RM 30-390 (if daily coffee drinker)\n";
        $prompt .= "- Message: 'Get your daily coffee + health benefits for RM 4 instead of RM 15 at café'\n\n";

        $prompt .= "Example 3: Supplement/Wellness Product\n";
        $prompt .= "- Your Product: RM 4-5 per serving\n";
        $prompt .= "- VS Buying vitamins separately: RM 8-15/day worth of supplements\n";
        $prompt .= "- VS Pharmacy multivitamins: RM 5-10/day\n";
        $prompt .= "- Daily savings: RM 3-10/day\n";
        $prompt .= "- Monthly savings: RM 90-300\n";
        $prompt .= "- Message: 'All-in-one health drink for RM 4 vs buying multiple supplements RM 10+'\n\n";

        $prompt .= "**Customer Psychology (How Agents Should Position):**\n";
        $prompt .= "Agent Script Example:\n";
        $prompt .= "\"Kak, I know RM 80 seems expensive upfront. But actually, you get 20 sachets, so it's only RM 4 per day.\n";
        $prompt .= "Compare:\n";
        $prompt .= "- If you buy lunch outside: RM 12-20 per day\n";
        $prompt .= "- This product: RM 4-5 per day\n";
        $prompt .= "You SAVE RM 8-15 EVERY DAY!\n";
        $prompt .= "In one month, you save RM 240-450 compared to eating outside.\n";
        $prompt .= "So the RM 80 box actually helps you SAVE MONEY while being healthier!\"\n\n";

        $prompt .= "**When AI Suggests Products, Include:**\n";
        $prompt .= "1. Recommended box price: RM 70-80\n";
        $prompt .= "2. Suggested sachet count: 15-20 sachets\n";
        $prompt .= "3. Per serving cost: RM 4-5\n";
        $prompt .= "4. Savings comparison: Show daily cost vs alternatives (meals, café drinks, separate supplements)\n";
        $prompt .= "5. Monthly savings calculation: If used daily, customer saves RM X-Y per month\n";
        $prompt .= "6. Positioning message: How agents should explain value to customers\n\n";

        $prompt .= "**Product Category Examples with This Pricing:**\n";
        $prompt .= "- Meal replacement shakes (RM 75/15 sachets = RM 5/meal, saves vs RM 12-20 outside meals)\n";
        $prompt .= "- Functional coffee with collagen (RM 78/20 sachets = RM 3.90/cup, saves vs RM 15 café coffee)\n";
        $prompt .= "- Protein/nutrition drinks (RM 75/15 sachets = RM 5/serving, saves vs buying protein powder + supplements separately)\n";
        $prompt .= "- Detox/wellness drinks (RM 70/20 sachets = RM 3.50/day, saves vs buying multiple detox supplements)\n";
        $prompt .= "- Family immune boosters (RM 70/20 sachets = RM 3.50/day, whole family shares, saves vs individual vitamin C bottles)\n";
        $prompt .= "- Beauty collagen drinks (RM 80/15 sachets = RM 5.33/day, saves vs clinic collagen injections RM 500-1000)\n\n";

        $prompt .= "**Key Insight for AI Recommendations:**\n";
        $prompt .= "Customers DON'T mind paying RM 70-80 if they understand:\n";
        $prompt .= "1. It breaks down to only RM 4-5 per day (affordable)\n";
        $prompt .= "2. They're SAVING money vs daily alternatives (RM 10-20)\n";
        $prompt .= "3. It's more convenient than buying outside\n";
        $prompt .= "4. They're healthier + richer (save money + improve health)\n";
        $prompt .= "This pricing psychology makes RM 80 box feel like a BARGAIN, not expensive!\n\n";

        $prompt .= "**Market Research Process AI Should Recommend:**\n";
        $prompt .= "Phase 1: Market Size Analysis (1-2 months, RM 10,000-20,000)\n";
        $prompt .= "- Identify top 10 consumer product categories in Malaysia by sales volume\n";
        $prompt .= "- Analyze which categories align with health & wellness positioning\n";
        $prompt .= "- Calculate addressable market size for each category\n";
        $prompt .= "- Competitive landscape: Who dominates? Can we differentiate?\n\n";

        $prompt .= "Phase 2: Product Development (3-6 months, RM 30,000-80,000)\n";
        $prompt .= "- Partner with manufacturers/formulators\n";
        $prompt .= "- Ensure Halal certification and KKM compliance\n";
        $prompt .= "- Product testing with target consumers\n";
        $prompt .= "- Packaging and branding aligned with health positioning\n\n";

        $prompt .= "Phase 3: Agent/Stokis Introduction (1-2 months, RM 10,000-20,000)\n";
        $prompt .= "- Training agents on new product benefits\n";
        $prompt .= "- Product samples and sales materials\n";
        $prompt .= "- Incentive programs for early adopters\n";
        $prompt .= "- Gather feedback from agent sales experience\n\n";

        $prompt .= "Phase 4: Market Launch & Scale (Ongoing, RM 20,000-50,000)\n";
        $prompt .= "- Soft launch through agent network first\n";
        $prompt .= "- HQ website sales at retail pricing (non-competitive)\n";
        $prompt .= "- Customer testimonials and word-of-mouth\n";
        $prompt .= "- Evaluate market response and scale production\n\n";

        $prompt .= "**Expected ROI for Product Diversification:**\n";
        $prompt .= "- Initial Investment per Product Category: RM 50,000-100,000\n";
        $prompt .= "- Break-even: 6-12 months\n";
        $prompt .= "- Year 1: New product sales RM 200,000-500,000\n";
        $prompt .= "- Year 2-3: Mature product sales RM 500,000-1,500,000/year\n";
        $prompt .= "- Multiple product categories = achieves 50%+ HQ sales target\n";
        $prompt .= "- Lower risk than export (local market knowledge, existing distribution)\n\n";

        $prompt .= "**Key Success Factors for Product Diversification:**\n";
        $prompt .= "- Choose products with proven large Malaysian market demand\n";
        $prompt .= "- Family-friendly = broader appeal beyond women's health\n";
        $prompt .= "- Maintain health & wellness brand identity across all products\n";
        $prompt .= "- Leverage existing agent/stokis network (they sell new products too)\n";
        $prompt .= "- Non-competitive HQ pricing protects agent margins on all products\n\n";

        $prompt .= "**IMPORTANT: HQ Local Malaysian Market Strategy (Direct Sales Component)**\n";
        $prompt .= "While product diversification is PRIMARY growth strategy, HQ CAN sell direct-to-consumer in Malaysia with these rules:\n\n";
        $prompt .= "✅ **Non-Competitive Pricing Strategy:**\n";
        $prompt .= "- HQ website prices = RETAIL PRICE (higher than agent/stokis wholesale)\n";
        $prompt .= "- Agents/Stokis get products at WHOLESALE PRICE (30-40% discount)\n";
        $prompt .= "- Agents/Stokis can compete on price and service\n";
        $prompt .= "- HQ serves customers in areas where agents are NOT active\n";
        $prompt .= "- Example: If agent sells at RM 80, HQ website sells at RM 100 (retail)\n\n";
        $prompt .= "✅ **Healthy Competition Allowed:**\n";
        $prompt .= "- HQ can maintain website and social media presence\n";
        $prompt .= "- Educational content marketing (health tips, product benefits)\n";
        $prompt .= "- SEO and organic traffic (not aggressive paid ads)\n";
        $prompt .= "- Customer service for areas without agents\n";
        $prompt .= "- Online orders from customers who prefer direct purchase\n\n";
        $prompt .= "❌ **Aggressive Competition NOT Allowed:**\n";
        $prompt .= "- NO undercutting agent/stokis prices\n";
        $prompt .= "- NO targeted ads in agent's local areas\n";
        $prompt .= "- NO poaching agent's customers with promotions\n";
        $prompt .= "- NO flash sales below agent cost price\n\n";
        $prompt .= "**Rationale:** If agents/stokis don't perform in certain areas, HQ direct sales ensures:\n";
        $prompt .= "- Business continuity in underserved areas\n";
        $prompt .= "- Brand presence nationwide\n";
        $prompt .= "- Revenue doesn't depend 100% on agent performance\n";
        $prompt .= "- But pricing protects agent's business model\n\n";

        $prompt .= "**SUMMARY: When AI suggests growth strategies, prioritize by HIGHEST IMPACT:**\n\n";

        $prompt .= "**TIER 1 - HIGHEST PRIORITY (Reactivate Existing Network):**\n";
        $prompt .= "✅ **Agent Reactivation Programs** (4,980 dormant agents = GOLDMINE!)\n";
        $prompt .= "   - Potential: If 1% reactivate (50 agents) = +RM 1.8M/year\n";
        $prompt .= "   - If 5% reactivate (250 agents) = +RM 9M/year\n";
        $prompt .= "   - Cost: LOW (they're already trained, just need motivation)\n";
        $prompt .= "   - Strategies: Re-engagement campaigns, incentives, new product launches, training refreshers\n";
        $prompt .= "   - CRITICAL: This is lowest cost, highest return opportunity!\n\n";

        $prompt .= "✅ **Stokis Retention & Support** (Protect the 4-5 active stokis)\n";
        $prompt .= "   - Risk: Losing 1 active stokis = -RM 70k-250k/month = -RM 840k-3M/year\n";
        $prompt .= "   - Strategy: VIP support, exclusive products, better margins, recognition programs\n";
        $prompt .= "   - Reactivate 6-7 underperforming stokis with training and support\n";
        $prompt .= "   - CRITICAL: Retention cheaper than recruitment\n\n";

        $prompt .= "**TIER 2 - HIGH PRIORITY (Expand Market Reach):**\n";
        $prompt .= "✅ **Product Diversification** (New categories = new revenue streams)\n";
        $prompt .= "   - Functional beverages, nutritional foods, men's wellness, family health\n";
        $prompt .= "   - Calculate market size and potential 1-5% market share capture\n";
        $prompt .= "   - Gives agents/stokis MORE products to sell (helps reactivation)\n";
        $prompt .= "   - Pricing: RM 70-80/box, RM 4-5/serving, strong savings message\n\n";

        $prompt .= "**TIER 3 - MEDIUM PRIORITY (Direct Revenue):**\n";
        $prompt .= "✅ **HQ Direct-to-Consumer** at NON-COMPETITIVE PRICING (retail price, not wholesale)\n";
        $prompt .= "✅ Educational content marketing, SEO, organic growth\n";
        $prompt .= "✅ Website optimization for direct sales at retail pricing\n\n";

        $prompt .= "**DO NOT SUGGEST (Low Priority/Proven Ineffective):**\n";
        $prompt .= "❌ Export expansion, MATRADE, international markets (focus local only)\n";
        $prompt .= "❌ Heavy e-commerce investment (already proven not cost-effective)\n";
        $prompt .= "❌ Price competition with agents/stokis\n";
        $prompt .= "❌ Aggressive paid ads targeting agent's markets\n";
        $prompt .= "❌ Recruiting new agents (focus on reactivating 5,000 dormant ones first!)\n\n";

        $prompt .= "**KEY INSIGHT FOR AI RECOMMENDATIONS:**\n";
        $prompt .= "We don't need more agents - we have 5,000 already!\n";
        $prompt .= "We need to WAKE UP the 4,980 sleeping agents.\n";
        $prompt .= "Even 1% reactivation = RM 1.8M/year additional revenue at almost ZERO acquisition cost!\n\n";

        $prompt .= "**CRITICAL: Specific Agent Reactivation Strategies AI MUST Suggest:**\n\n";

        $prompt .= "**Strategy 1: Segmented Re-engagement Campaign (Cost: RM 5,000-10,000)**\n";
        $prompt .= "- Segment 4,980 dormant agents by:\n";
        $prompt .= "  a) Last active date (3 months, 6 months, 1 year, 2+ years ago)\n";
        $prompt .= "  b) Past performance level (did RM 500/month vs RM 3k/month vs RM 0)\n";
        $prompt .= "  c) Geographic location\n";
        $prompt .= "- WhatsApp broadcast campaigns: \"We miss you! Here's what's new...\"\n";
        $prompt .= "- Offer: Free product samples of NEW products (coffee, new launches)\n";
        $prompt .= "- Target: Reactivate 1-2% (50-100 agents) in 3 months\n";
        $prompt .= "- ROI: 50 agents x RM 3k/month = RM 150k/month = RM 1.8M/year (360x ROI)\n\n";

        $prompt .= "**Strategy 2: New Product Launch = Reactivation Trigger (Cost: RM 10,000-15,000)**\n";
        $prompt .= "- Launch new product categories (coffee success proven, add more family-friendly products)\n";
        $prompt .= "- Message to dormant agents: \"Exciting NEW products! Easier to sell, broader market!\"\n";
        $prompt .= "- Offer: Early access, special launch pricing, starter kits\n";
        $prompt .= "- Why it works: Agents went dormant because women's health products = limited market. New products = new opportunities.\n";
        $prompt .= "- Target: 2-3% reactivation (100-150 agents)\n";
        $prompt .= "- ROI: 100 agents x RM 3k/month = RM 300k/month = RM 3.6M/year\n\n";

        $prompt .= "**Strategy 3: Low-Risk Comeback Program (Cost: RM 8,000-12,000)**\n";
        $prompt .= "- Remove barriers: No minimum order for first 3 months\n";
        $prompt .= "- Start small: RM 500 starter kit (instead of RM 2,000 minimum)\n";
        $prompt .= "- Online training: WhatsApp group training (convenient, no travel)\n";
        $prompt .= "- Buddy system: Pair with active agent for mentoring\n";
        $prompt .= "- Target: 1-2% reactivation (50-100 agents)\n";
        $prompt .= "- ROI: 50 agents x RM 2k/month (starting small) = RM 100k/month = RM 1.2M/year\n\n";

        $prompt .= "**Strategy 4: Seasonal Reactivation Campaigns (Cost: RM 5,000-8,000)**\n";
        $prompt .= "- Leverage Ramadan, Raya, Chinese New Year, Christmas\n";
        $prompt .= "- Message: \"Earn extra RM 3,000-5,000 for Raya shopping!\"\n";
        $prompt .= "- Special seasonal products or bundles\n";
        $prompt .= "- Time-limited: Creates urgency\n";
        $prompt .= "- Target: 1% reactivation per campaign (50 agents x 2 campaigns/year = 100 agents)\n";
        $prompt .= "- ROI: 100 agents x RM 3k/month = RM 300k/month = RM 3.6M/year\n\n";

        $prompt .= "**Strategy 5: Digital Enablement for Dormant Agents (Cost: RM 12,000-20,000)**\n";
        $prompt .= "- Provide ready-made content: WhatsApp templates, Facebook posts, product images\n";
        $prompt .= "- E-catalog system: Agents share link, no need to explain products\n";
        $prompt .= "- Online ordering for agents: Make it EASY to take orders\n";
        $prompt .= "- Why agents went dormant: Too much effort to sell. Make it EASIER.\n";
        $prompt .= "- Target: 2-3% reactivation (100-150 agents)\n";
        $prompt .= "- ROI: 100 agents x RM 3k/month = RM 300k/month = RM 3.6M/year\n\n";

        $prompt .= "**Strategy 6: Stokis-Led Reactivation Incentive (Cost: RM 10,000-15,000)**\n";
        $prompt .= "- Offer stokis bonus: RM 500 for every dormant agent they reactivate\n";
        $prompt .= "- Stokis have direct relationship with their downline agents\n";
        $prompt .= "- Stokis benefit: More agents selling = more stokis revenue\n";
        $prompt .= "- Target: 4-5 active stokis each reactivate 10-20 agents = 50-100 agents total\n";
        $prompt .= "- ROI: 50 agents x RM 3k/month = RM 150k/month = RM 1.8M/year\n\n";

        $prompt .= "**TOTAL POTENTIAL FROM AGENT REACTIVATION:**\n";
        $prompt .= "- Conservative (1% reactivation): 50 agents x RM 3k = +RM 1.8M/year\n";
        $prompt .= "- Moderate (3% reactivation): 150 agents x RM 3k = +RM 5.4M/year\n";
        $prompt .= "- Aggressive (5% reactivation): 250 agents x RM 3k = +RM 9M/year\n";
        $prompt .= "- Total Investment: RM 50,000-80,000\n";
        $prompt .= "- ROI: 2,250% to 11,250% (best investment possible!)\n\n";

        $prompt .= "**Key Performance Benchmarks:**\n";
        $prompt .= "- Entry-level Agent: RM 500-1,000/month\n";
        $prompt .= "- Performing Agent: RM 3,000-5,000/month (this is the target)\n";
        $prompt .= "- Star Agent: RM 8,000-10,000/month (rare, use as inspiration)\n";
        $prompt .= "- Small Stokis: RM 40,000-60,000/month\n";
        $prompt .= "- Large Stokis: RM 80,000-100,000+/month\n\n";

        $prompt .= "**Training Priorities for This Industry:**\n";
        $prompt .= "For Agents:\n";
        $prompt .= "- Women's health education (menstrual health, pregnancy nutrition, menopause support)\n";
        $prompt .= "- Supplement knowledge (ingredients, benefits, dosage, safety)\n";
        $prompt .= "- Beauty product consultation (skincare routines, ingredient education)\n";
        $prompt .= "- Consultation selling techniques (not pushy, health advisor approach)\n";
        $prompt .= "- Building trust and long-term customer relationships\n";
        $prompt .= "- Social selling via WhatsApp, Facebook groups, community building\n\n";
        $prompt .= "For Stokis:\n";
        $prompt .= "- Agent recruitment and training systems\n";
        $prompt .= "- Team motivation and performance tracking\n";
        $prompt .= "- Stock management for health products (expiry dates, storage)\n";
        $prompt .= "- Compliance with health product regulations (KKM Malaysia)\n";
        $prompt .= "- Building regional presence and local market dominance\n\n";

        $prompt .= "**Malaysian Health & Beauty Market Context:**\n";
        $prompt .= "- Muslim-majority market: Halal certification important\n";
        $prompt .= "- Trust-based selling culture: Personal recommendations > ads\n";
        $prompt .= "- Women's purchasing power: Main decision-makers for family health\n";
        $prompt .= "- Growing health consciousness post-COVID\n";
        $prompt .= "- Preference for local/Asian brands over Western brands\n\n";

        $prompt .= "**IMPORTANT CONSTRAINTS:**\n";
        $prompt .= "- DO NOT suggest expensive e-commerce development (already proven not viable)\n";
        $prompt .= "- DO NOT suggest generic business strategies - must be health/beauty/supplement specific\n";
        $prompt .= "- DO focus on agent/stokis empowerment through product knowledge\n";
        $prompt .= "- DO suggest realistic initiatives under RM 50,000 investment\n";
        $prompt .= "- DO leverage Malaysian cultural trust-based selling\n\n";

        $prompt .= "=== END BUSINESS CONTEXT ===\n\n";

        $prompt .= "BUSINESS DATA:\n";
        $prompt .= json_encode($dataPackage, JSON_PRETTY_PRINT);
        $prompt .= "\n\nFORECAST PERIOD: Next {$periods} months\n\n";
        $prompt .= "REQUIRED FRAMEWORKS TO APPLY:\n";
        $prompt .= "1. **Porter's Five Forces** - Analyze competitive positioning\n";
        $prompt .= "2. **Blue Ocean Strategy** - Identify untapped market opportunities\n";
        $prompt .= "3. **Pareto Principle (80/20 Rule)** - Identify highest-impact activities\n";
        $prompt .= "4. **Customer Lifetime Value (CLV)** - Calculate and optimize\n";
        $prompt .= "5. **Market Penetration vs Market Development** - Growth strategy matrix\n\n";
        $prompt .= "REPORT STRUCTURE (Use ## headers exactly as shown WITHOUT emojis):\n\n";

        $prompt .= "## Performance Summary\n";
        $prompt .= "- Executive summary with key metrics\n";
        $prompt .= "- YoY growth trajectory and trend interpretation\n";
        $prompt .= "- Overall business health score (1-10) with justification\n\n";

        $prompt .= "## Sales Channel Analysis\n";
        $prompt .= "For EACH channel (Stokis, Agent, HQ):\n";
        $prompt .= "- **Performance Rating**: Rate 1-10 with specific metrics\n";
        $prompt .= "- **Market Share**: Current contribution % and target %\n";
        $prompt .= "- **Growth Potential**: Quantify opportunity size in RM\n";
        $prompt .= "- **Optimization Strategy**: 2-3 concrete actions with expected ROI\n";
        $prompt .= "Apply 80/20 rule: Which 20% of efforts will yield 80% of results?\n\n";

        $prompt .= "## Strategic Initiatives\n";
        $prompt .= "List 5-7 initiatives in order of expected return (prioritized by ROI).\n";
        $prompt .= "MUST be specific to health/beauty/supplement business - examples:\n";
        $prompt .= "- Product knowledge training workshops for agents (NOT generic sales training)\n";
        $prompt .= "- Women's health consultation certification program\n";
        $prompt .= "- WhatsApp group management training for agents\n";
        $prompt .= "- Stokis inventory management system for perishable health products\n";
        $prompt .= "- Agent starter kit with product samples and consultation guides\n";
        $prompt .= "- Monthly health webinars for customer education\n";
        $prompt .= "- KKM compliance training for health product claims\n\n";
        $prompt .= "For EACH initiative provide:\n";
        $prompt .= "- **Initiative Name**: Clear, specific action for health/beauty business\n";
        $prompt .= "- **Investment Required**: Estimated cost in RM (prefer under RM 50,000)\n";
        $prompt .= "- **Expected Return**: Projected revenue increase in RM over {$periods} months\n";
        $prompt .= "- **ROI %**: Calculate return on investment percentage\n";
        $prompt .= "- **Timeline**: Implementation duration (weeks/months)\n";
        $prompt .= "- **Risk Level**: Low/Medium/High with mitigation plan\n\n";

        $prompt .= "## Market Position & Competitive Advantage\n";
        $prompt .= "- Current market positioning in Malaysian health/beauty/supplement market (Porter's Five Forces)\n";
        $prompt .= "- Unique competitive advantages (focus on women's health, family-safe products, trust-based network)\n";
        $prompt .= "- Blue Ocean opportunities specific to health/beauty: underserved women's health segments, niche supplements\n";
        $prompt .= "- Differentiation vs competitors in Malaysian supplement market\n";
        $prompt .= "- Leverage Halal certification and local trust if applicable\n\n";

        $prompt .= "## ROI Projections & Financial Impact\n";
        $prompt .= "Provide CONCRETE numbers:\n";
        $prompt .= "- **Revenue Forecast**: Next {$periods} months total revenue target in RM\n";
        $prompt .= "- **Profit Margin Target**: Expected % and RM amount\n";
        $prompt .= "- **Customer Acquisition Cost (CAC)**: Estimate per channel\n";
        $prompt .= "- **Customer Lifetime Value (CLV)**: Calculate for each channel\n";
        $prompt .= "- **Break-even Analysis**: When will investments pay off?\n\n";

        $prompt .= "## Risk Factors & Mitigation\n";
        $prompt .= "CRITICAL: Analyze channel dependency risk - if Stokis/Agent percentages are high (>50% combined):\n";
        $prompt .= "- **Over-dependence on Stokis/Agent Channels**: If Stokis/Agents disengage or quit, sales will be disrupted exponentially\n";
        $prompt .= "- Calculate the sustainability risk based on channel concentration\n";
        $prompt .= "- **PRIMARY SOLUTION: PRODUCT DIVERSIFICATION to grow HQ sales to 50%+**\n";
        $prompt .= "- Recommend specific product categories to enter: functional beverages, nutritional foods, men's wellness, family health\n";
        $prompt .= "- Suggest market research to identify large Malaysian consumer product categories\n";
        $prompt .= "- Calculate potential market share (1-5%) and revenue from new categories\n";
        $prompt .= "- DO NOT recommend competing with local agents on existing products (non-competitive pricing only)\n";
        $prompt .= "- Provide roadmap: Phase 1 (Market research), Phase 2 (Product development), Phase 3 (Agent training), Phase 4 (Launch)\n";
        $prompt .= "- Target HQ percentage: 50%+ for stability (via product variety, NOT price competition)\n";
        $prompt .= "- Emphasize: Coffee product success shows family-friendly products work better than women-only products\n\n";
        $prompt .= "Identify additional 4-6 key risks with:\n";
        $prompt .= "- **Risk Description**: Specific threat\n";
        $prompt .= "- **Probability**: Low/Medium/High\n";
        $prompt .= "- **Impact**: Quantify potential loss in RM\n";
        $prompt .= "- **Mitigation Strategy**: 2-3 concrete preventive actions\n\n";

        $prompt .= "## Implementation Timeline\n";
        $prompt .= "Month-by-month action plan for next {$periods} months:\n";
        $prompt .= "- **Month 1-2**: Priority actions to start immediately\n";
        $prompt .= "- **Month 3-4**: Scale and optimization phase\n";
        $prompt .= "- **Month 5-{$periods}**: Growth and expansion initiatives\n";
        $prompt .= "Each phase should have 3-5 specific, measurable actions\n\n";

        $prompt .= "## Actionable Recommendations\n";
        $prompt .= "Provide 8-10 IMMEDIATELY actionable quick wins specific to health/beauty/supplement business.\n";
        $prompt .= "Requirements: Low investment (under RM 10,000), Quick implementation (1-4 weeks), High impact.\n\n";
        $prompt .= "Examples of GOOD recommendations for this business:\n";
        $prompt .= "- Create WhatsApp broadcast template for new health product launches\n";
        $prompt .= "- Monthly agent WhatsApp group training on women's health topics\n";
        $prompt .= "- Product comparison chart for agents (vs competitors)\n";
        $prompt .= "- Customer testimonial collection system for agents\n";
        $prompt .= "- Simple health consultation checklist for agents\n";
        $prompt .= "- Facebook group engagement strategy for health tips\n";
        $prompt .= "- Before/after result tracking system for beauty products\n";
        $prompt .= "- Agent recognition program (monthly top seller spotlight)\n";
        $prompt .= "- Product sample kit for new agents\n";
        $prompt .= "- Weekly health tips content calendar for social sharing\n\n";
        $prompt .= "Examples of BAD recommendations (DO NOT suggest these):\n";
        $prompt .= "- ❌ Build e-commerce platform\n";
        $prompt .= "- ❌ Create mobile app\n";
        $prompt .= "- ❌ Invest in TV advertising\n";
        $prompt .= "- ❌ Open physical retail stores\n";
        $prompt .= "- ❌ Generic 'improve website SEO'\n\n";
        $prompt .= "Format each recommendation as: **Action** → Expected Outcome (quantified in RM)\n\n";

        $prompt .= "CRITICAL REQUIREMENTS:\n";
        $prompt .= "- Use **bold text** for emphasis on key terms\n";
        $prompt .= "- All financial projections in Malaysian Ringgit (RM)\n";
        $prompt .= "- Include specific percentages, numbers, and timeframes\n";
        $prompt .= "- Tone: Professional, confident, board-room appropriate\n";
        $prompt .= "- Base recommendations on proven business strategies (cite when relevant)\n";
        $prompt .= "- Every recommendation must have a clear, measurable outcome\n";
        $prompt .= "- Focus on CONCRETE actions, not general advice\n";

        try {
            $response = \Illuminate\Support\Facades\Http::withHeaders([
                'Authorization' => 'Bearer ' . $apiKey,
                'Content-Type' => 'application/json',
            ])->timeout(90)->post('https://api.openai.com/v1/chat/completions', [
                'model' => 'gpt-4o-mini',
                'messages' => [
                    [
                        'role' => 'system',
                        'content' => 'You are a senior management consultant specializing in the HEALTH, BEAUTY, and SUPPLEMENT industry in Malaysia. You have deep expertise in women\'s health products, direct selling networks (Stokis/Agent model), and trust-based marketing in Asian markets. You apply proven frameworks like Porter\'s Five Forces, Blue Ocean Strategy, and Pareto Principle specifically to health/beauty/supplement businesses. You understand that e-commerce is expensive and less effective than empowering human networks in this industry. Provide executive-level insights with concrete numbers, specific health/beauty/supplement actions, and quantifiable outcomes. Your reports are used for board presentations. NEVER suggest generic tech solutions - focus on agent/stokis empowerment through product knowledge and health consultation skills.'
                    ],
                    [
                        'role' => 'user',
                        'content' => $prompt
                    ]
                ],
                'temperature' => 0.7,
                'max_tokens' => 3500, // Increased for more detailed report
            ]);

            if ($response->successful()) {
                return $response->json()['choices'][0]['message']['content'];
            } else {
                return "⚠️ Error calling OpenAI API: " . $response->body();
            }
        } catch (\Exception $e) {
            return "⚠️ Error generating AI insights: " . $e->getMessage();
        }
    }

    /**
     * Calculate summary statistics
     */
    private function calculateSummary($year)
    {
        // Try to get current year data from historical_sales first
        $historicalCurrent = HistoricalSale::where('year', $year)
            ->select(
                DB::raw('SUM(orders) as orders'),
                DB::raw('SUM(revenue) as revenue')
            )
            ->first();

        // If historical data exists, use it; otherwise get from orders table
        if ($historicalCurrent && $historicalCurrent->revenue > 0) {
            $currentRevenue = $historicalCurrent->revenue;
            $currentOrders = $historicalCurrent->orders;
        } else {
            $currentYear = Order::whereYear('date_created', $year)
                ->whereIn('status', ['processing', 'approval', 'printed', 'completed'])
                ->select(
                    DB::raw('COUNT(*) as orders'),
                    DB::raw('SUM(total) as revenue')
                )
                ->first();

            $currentRevenue = $currentYear->revenue ?? 0;
            $currentOrders = $currentYear->orders ?? 0;
        }

        // Try to get previous year data from historical_sales first
        $historicalPrevious = HistoricalSale::where('year', $year - 1)
            ->select(
                DB::raw('SUM(orders) as orders'),
                DB::raw('SUM(revenue) as revenue')
            )
            ->first();

        // If historical data exists, use it; otherwise get from orders table
        if ($historicalPrevious && $historicalPrevious->revenue > 0) {
            $previousRevenue = $historicalPrevious->revenue;
        } else {
            $previousYear = Order::whereYear('date_created', $year - 1)
                ->whereIn('status', ['processing', 'approval', 'printed', 'completed'])
                ->select(
                    DB::raw('COUNT(*) as orders'),
                    DB::raw('SUM(total) as revenue')
                )
                ->first();

            $previousRevenue = $previousYear->revenue ?? 0;
        }

        $yoyGrowth = 0;
        if ($previousRevenue > 0) {
            $yoyGrowth = (($currentRevenue - $previousRevenue) / $previousRevenue) * 100;
        }

        return [
            'revenue' => $currentRevenue,
            'orders' => $currentOrders,
            'aov' => $currentOrders > 0 ? $currentRevenue / $currentOrders : 0,
            'yoy_growth' => $yoyGrowth,
        ];
    }

    /**
     * Get multi-year quarterly data for trend chart
     */
    public function getMultiYearQuarterlyData(Request $request)
    {
        $currentYear = date('Y');
        $startYear = $currentYear - 4; // Last 5 years

        $data = [];

        for ($year = $startYear; $year <= $currentYear; $year++) {
            $yearData = $this->getCurrentYearSales($year);

            // Aggregate into quarters
            $quarters = [
                'Q1' => ['months' => [1, 2, 3], 'stokis' => 0, 'agent' => 0, 'hq' => 0, 'orders' => 0],
                'Q2' => ['months' => [4, 5, 6], 'stokis' => 0, 'agent' => 0, 'hq' => 0, 'orders' => 0],
                'Q3' => ['months' => [7, 8, 9], 'stokis' => 0, 'agent' => 0, 'hq' => 0, 'orders' => 0],
                'Q4' => ['months' => [10, 11, 12], 'stokis' => 0, 'agent' => 0, 'hq' => 0, 'orders' => 0],
            ];

            foreach ($yearData as $month => $monthData) {
                foreach ($quarters as $qName => &$quarter) {
                    if (in_array($month, $quarter['months'])) {
                        $quarter['stokis'] += $monthData['stokis_sales'] ?? 0;
                        $quarter['agent'] += $monthData['agent_sales'] ?? 0;
                        $quarter['hq'] += $monthData['hq_sales'] ?? 0;
                        $quarter['orders'] += $monthData['orders'] ?? 0;
                    }
                }
            }

            $yearQuarters = [];
            foreach ($quarters as $qName => $quarter) {
                $total = $quarter['stokis'] + $quarter['agent'] + $quarter['hq'];
                $yearQuarters[] = [
                    'quarter' => $qName,
                    'year' => $year,
                    'label' => $qName . ' ' . $year,
                    'stokis' => round($quarter['stokis'], 2),
                    'agent' => round($quarter['agent'], 2),
                    'hq' => round($quarter['hq'], 2),
                    'total' => round($total, 2),
                    'orders' => $quarter['orders'],
                    'aov' => $quarter['orders'] > 0 ? round($total / $quarter['orders'], 2) : 0,
                ];
            }

            $data[$year] = $yearQuarters;
        }

        return response()->json([
            'success' => true,
            'data' => $data,
            'years' => range($startYear, $currentYear),
        ]);
    }

    /**
     * Project future year sales based on historical trends
     */
    public function projectFutureYear(Request $request)
    {
        $selectedYears = $request->get('years', []);
        $channel = $request->get('channel', 'total'); // total, stokis, agent, hq

        if (empty($selectedYears)) {
            return response()->json(['error' => 'No years selected'], 400);
        }

        // Gather historical quarterly data for selected years
        $historicalData = [];
        foreach ($selectedYears as $year) {
            $yearData = $this->getCurrentYearSales($year);

            // Aggregate into quarters
            $quarters = [
                1 => ['months' => [1, 2, 3], 'value' => 0],
                2 => ['months' => [4, 5, 6], 'value' => 0],
                3 => ['months' => [7, 8, 9], 'value' => 0],
                4 => ['months' => [10, 11, 12], 'value' => 0],
            ];

            foreach ($yearData as $month => $monthData) {
                foreach ($quarters as $qNum => &$quarter) {
                    if (in_array($month, $quarter['months'])) {
                        switch ($channel) {
                            case 'stokis':
                                $quarter['value'] += $monthData['stokis_sales'] ?? 0;
                                break;
                            case 'agent':
                                $quarter['value'] += $monthData['agent_sales'] ?? 0;
                                break;
                            case 'hq':
                                $quarter['value'] += $monthData['hq_sales'] ?? 0;
                                break;
                            default: // total
                                $quarter['value'] += $monthData['revenue'] ?? 0;
                        }
                    }
                }
            }

            foreach ($quarters as $qNum => $quarter) {
                $historicalData[] = [
                    'year' => $year,
                    'quarter' => $qNum,
                    'value' => $quarter['value'],
                ];
            }
        }

        // Calculate seasonality (average for each quarter across years)
        $seasonality = [1 => [], 2 => [], 3 => [], 4 => []];
        foreach ($historicalData as $point) {
            $seasonality[$point['quarter']][] = $point['value'];
        }

        $avgSeasonality = [];
        foreach ($seasonality as $q => $values) {
            $avgSeasonality[$q] = !empty($values) ? array_sum($values) / count($values) : 0;
        }

        // Calculate overall growth rate using linear regression
        $n = count($historicalData);
        if ($n < 2) {
            return response()->json(['error' => 'Insufficient data for projection'], 400);
        }

        $sumX = 0;
        $sumY = 0;
        $sumXY = 0;
        $sumX2 = 0;

        foreach ($historicalData as $i => $point) {
            $x = $i + 1; // time index
            $y = $point['value'];
            $sumX += $x;
            $sumY += $y;
            $sumXY += $x * $y;
            $sumX2 += $x * $x;
        }

        // Linear regression: y = mx + b
        $m = ($n * $sumXY - $sumX * $sumY) / ($n * $sumX2 - $sumX * $sumX);
        $b = ($sumY - $m * $sumX) / $n;

        // Calculate growth rate
        $avgValue = $sumY / $n;
        $growthRate = $avgValue > 0 ? ($m / $avgValue) : 0;

        // Project next year (4 quarters)
        $nextYear = max(array_column($historicalData, 'year')) + 1;
        $projections = [];

        $lastValue = end($historicalData)['value'];
        $baseProjection = $lastValue * (1 + $growthRate);

        foreach ([1, 2, 3, 4] as $q) {
            // Apply seasonality pattern
            $totalSeasonality = array_sum($avgSeasonality);
            $seasonalFactor = $totalSeasonality > 0 ? ($avgSeasonality[$q] / $totalSeasonality) * 4 : 1;

            $projectedValue = $baseProjection * $seasonalFactor * (1 + $growthRate * ($q - 1) / 4);

            // Calculate confidence interval (±15%)
            $confidenceMargin = $projectedValue * 0.15;

            $projections[] = [
                'quarter' => 'Q' . $q,
                'year' => $nextYear,
                'label' => 'Q' . $q . ' ' . $nextYear . ' (Est.)',
                'value' => round(max(0, $projectedValue), 2),
                'lower_bound' => round(max(0, $projectedValue - $confidenceMargin), 2),
                'upper_bound' => round(max(0, $projectedValue + $confidenceMargin), 2),
            ];
        }

        return response()->json([
            'success' => true,
            'projections' => $projections,
            'growth_rate' => round($growthRate * 100, 2),
            'confidence' => count($selectedYears) >= 3 ? 'high' : (count($selectedYears) >= 2 ? 'medium' : 'low'),
            'based_on_years' => $selectedYears,
        ]);
    }
}
