<?php

namespace App\Http\Controllers;

use App\Models\Order;
use App\Models\Store;
use App\Models\WebhookSource;
use App\Services\DashboardStatsService;
use App\Exports\AnalyticsExport;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use Maatwebsite\Excel\Facades\Excel;

class DashboardController extends Controller
{
    protected $statsService;

    public function __construct(DashboardStatsService $statsService)
    {
        $this->statsService = $statsService;
    }

    public function index(Request $request)
    {
        $perPage = $request->get('per_page', 25);
        $user = $request->user();

        // Get date range parameters
        $period = $request->get('period', 'today');
        $startDate = $request->get('start_date');
        $endDate = $request->get('end_date');

        // Get analytics filter parameters
        $analyticsSource = $request->get('analytics_source'); // 'all', 'store_X', 'webhook_X'

        // For sellers and managers, override analytics source to filter by their stores only
        if ($user->isSeller() || $user->isManager()) {
            $analyticsSource = null; // Will be filtered in queries
        }

        // Calculate date range based on period
        $dateRange = $this->calculateDateRange($period, $startDate, $endDate);

        // Build orders query with role-based filtering
        $ordersQuery = Order::with(['store:id,name,store_prefix', 'webhookSource:id,name,type'])
            ->select(['id', 'store_id', 'webhook_source_id', 'order_number', 'global_order_id', 'woo_order_id', 'status', 'total', 'currency', 'payment_method', 'payment_method_title', 'billing', 'shipping', 'date_created'])
            ->whereIn('status', ['processing', 'pending', 'on-hold']);

        // Filter by user's stores and created orders if seller or manager
        if ($user->isSeller()) {
            $storeIds = $user->getStoreIds();
            $ordersQuery->where(function($q) use ($storeIds, $user) {
                $q->whereIn('store_id', $storeIds)
                  ->orWhere('created_by', $user->id);
            });
        } elseif ($user->isManager()) {
            $storeIds = $user->getStoreIds();
            $sellerIds = $user->getManagedSellerIds();
            $ordersQuery->where(function($q) use ($storeIds, $sellerIds) {
                $q->whereIn('store_id', $storeIds)
                  ->orWhereIn('created_by', $sellerIds);
            });
        }

        $orders = $ordersQuery->orderBy('date_created', 'desc')->paginate($perPage);

        $stats = $this->statsService->getTodayStats($user);

        // Period-based analytics
        $periodStats = $this->statsService->getPeriodStats($dateRange, $user);

        // Advanced Analytics
        $advancedAnalytics = $this->getAdvancedAnalytics($dateRange, $analyticsSource, $user);

        // Top Sales (Admin and Manager only)
        $topSales = null;
        if ($user->isAdmin() || $user->isManager()) {
            $topSales = $this->getTopSales($dateRange, $user);
        }

        // Shipping Cost Estimation
        $shippingCosts = $this->statsService->getShippingCostEstimation($dateRange, $user);

        return view('dashboard.index', compact('orders', 'stats', 'perPage', 'periodStats', 'period', 'startDate', 'endDate', 'dateRange', 'advancedAnalytics', 'analyticsSource', 'topSales', 'shippingCosts'));
    }

    private function getAdvancedAnalytics($dateRange, $analyticsSource = null, $user = null, $includeOrderDetails = false)
    {
        $startDate = $dateRange['start'];
        $endDate = $dateRange['end'];

        // Skip caching for exports (since they need fresh detailed data)
        if ($includeOrderDetails) {
            return $this->calculateAdvancedAnalytics($startDate, $endDate, $analyticsSource, $user, $includeOrderDetails);
        }

        // OPTIMIZED: Add caching for dashboard view
        $userKey = $user ? ($user->isSeller() ? 'seller_' . $user->id : ($user->isManager() ? 'manager_' . $user->id : 'admin')) : 'all';
        $sourceKey = $analyticsSource ?? 'all';
        $cacheKey = "dashboard_advanced_analytics_{$userKey}_{$sourceKey}_{$startDate->format('Y-m-d')}_{$endDate->format('Y-m-d')}";

        // Use shorter cache for "today" analytics, longer for historical
        $isToday = $startDate->isToday() && $endDate->isToday();
        $cacheTTL = $isToday ? 60 : 1800; // 1 minute for today, 30 minutes for historical

        return Cache::remember($cacheKey, $cacheTTL, function () use ($startDate, $endDate, $analyticsSource, $user, $includeOrderDetails) {
            return $this->calculateAdvancedAnalytics($startDate, $endDate, $analyticsSource, $user, $includeOrderDetails);
        });
    }

    private function calculateAdvancedAnalytics($startDate, $endDate, $analyticsSource, $user, $includeOrderDetails = false)
    {
        // Build query for ALL orders in the period
        $query = Order::whereBetween('date_created', [$startDate, $endDate]);

        // Apply role-based filtering first (for sellers and managers)
        if ($user && $user->isSeller()) {
            $storeIds = $user->getStoreIds();
            $query->where(function($q) use ($storeIds, $user) {
                $q->whereIn('store_id', $storeIds)
                  ->orWhere('created_by', $user->id);
            });
        } elseif ($user && $user->isManager()) {
            $storeIds = $user->getStoreIds();
            $sellerIds = $user->getManagedSellerIds();
            $query->where(function($q) use ($storeIds, $sellerIds) {
                $q->whereIn('store_id', $storeIds)
                  ->orWhereIn('created_by', $sellerIds);
            });
        }
        // Apply source filter if specified (for admins)
        elseif ($analyticsSource && $analyticsSource !== 'all') {
            if (strpos($analyticsSource, 'store_') === 0) {
                // Filter by store
                $storeId = str_replace('store_', '', $analyticsSource);
                $query->where('store_id', $storeId);
            } elseif (strpos($analyticsSource, 'webhook_') === 0) {
                // Filter by webhook source
                $webhookId = str_replace('webhook_', '', $analyticsSource);
                $query->where('webhook_source_id', $webhookId);
            }
        }

        // OPTIMIZED: Pass query builder instead of loading all data
        // 1. Delivery by State (uses database aggregation)
        $deliveryByState = $this->getDeliveryByState(clone $query);

        // 2. Top Products (by SKU, needs line_items - load efficiently)
        $topProducts = $this->getTopProducts(clone $query);

        // 3. Units Out (by SKU, needs line_items - load efficiently)
        $unitsOut = $this->getUnitsOut(clone $query);

        // 4. Repeat Customers (by email/phone from filtered orders)
        $repeatCustomers = $this->getRepeatCustomers($startDate, $endDate, $analyticsSource, $user);

        // 5. Customer Database (all customers with full contact info)
        $customerDatabase = $this->getCustomerDatabase($startDate, $endDate, $analyticsSource, $user, $includeOrderDetails);

        return [
            'delivery_by_state' => $deliveryByState,
            'top_products' => $topProducts,
            'units_out' => $unitsOut,
            'repeat_customers' => $repeatCustomers,
            'customer_database' => $customerDatabase,
        ];
    }

    private function getDeliveryByState($orders)
    {
        // OPTIMIZED: Use database aggregation with JSON extraction
        // Note: $orders is now a query builder, not a collection
        $query = is_a($orders, 'Illuminate\Database\Eloquent\Builder') ? $orders : Order::query();

        $stateCounts = DB::table(DB::raw('(' . $query->toSql() . ') as filtered_orders'))
            ->mergeBindings($query->getQuery())
            ->select(
                DB::raw("COALESCE(
                    TRIM(JSON_UNQUOTE(JSON_EXTRACT(shipping, '$.state'))),
                    TRIM(JSON_UNQUOTE(JSON_EXTRACT(billing, '$.state'))),
                    'Unknown'
                ) as state"),
                DB::raw('COUNT(*) as count')
            )
            ->groupBy('state')
            ->orderByDesc('count')
            ->get();

        // Normalize and combine states (e.g., SGR + Selangor = Selangor)
        $normalizedStates = [];
        foreach ($stateCounts as $item) {
            $normalizedStateName = $this->normalizeStateName($item->state);

            if (!isset($normalizedStates[$normalizedStateName])) {
                $normalizedStates[$normalizedStateName] = 0;
            }
            $normalizedStates[$normalizedStateName] += (int)$item->count;
        }

        $totalOrders = array_sum($normalizedStates);

        // Convert to array and calculate percentages
        $stateData = [];
        foreach ($normalizedStates as $state => $count) {
            $stateData[] = [
                'state' => $state,
                'count' => $count,
                'percentage' => $totalOrders > 0 ? round(($count / $totalOrders) * 100, 1) : 0,
            ];
        }

        // Sort by count descending
        usort($stateData, function($a, $b) {
            return $b['count'] - $a['count'];
        });

        return [
            'data' => $stateData,
            'total' => $totalOrders,
        ];
    }

    /**
     * Normalize state names to standardized full names
     * Maps state abbreviations and variations to consistent full names
     */
    private function normalizeStateName($state)
    {
        if (empty($state)) {
            return 'Unknown';
        }

        // Convert to lowercase for comparison
        $stateLower = strtolower(trim($state));

        // State abbreviation and variation mapping
        $stateMapping = [
            // Johor
            'jhr' => 'Johor',
            'johor' => 'Johor',

            // Kedah
            'kdh' => 'Kedah',
            'kedah' => 'Kedah',

            // Kelantan
            'ktn' => 'Kelantan',
            'kelantan' => 'Kelantan',

            // Malacca / Melaka
            'mlk' => 'Melaka',
            'melaka' => 'Melaka',
            'malacca' => 'Melaka',

            // Negeri Sembilan
            'nsn' => 'Negeri Sembilan',
            'ns' => 'Negeri Sembilan',
            'negeri sembilan' => 'Negeri Sembilan',

            // Pahang
            'phg' => 'Pahang',
            'pahang' => 'Pahang',

            // Penang / Pulau Pinang
            'png' => 'Pulau Pinang',
            'penang' => 'Pulau Pinang',
            'pulau pinang' => 'Pulau Pinang',

            // Perak
            'prk' => 'Perak',
            'perak' => 'Perak',

            // Perlis
            'pls' => 'Perlis',
            'perlis' => 'Perlis',

            // Sabah
            'sbh' => 'Sabah',
            'sabah' => 'Sabah',

            // Sarawak
            'swk' => 'Sarawak',
            'sarawak' => 'Sarawak',
            'srwk' => 'Sarawak',

            // Selangor
            'sgr' => 'Selangor',
            'sel' => 'Selangor',
            'selangor' => 'Selangor',

            // Terengganu
            'trg' => 'Terengganu',
            'tganu' => 'Terengganu',
            'terengganu' => 'Terengganu',
            'trengganu' => 'Terengganu',

            // Federal Territories
            'kul' => 'Kuala Lumpur',
            'wp kuala lumpur' => 'Kuala Lumpur',
            'kuala lumpur' => 'Kuala Lumpur',
            'kl' => 'Kuala Lumpur',

            'lbn' => 'Labuan',
            'wp labuan' => 'Labuan',
            'labuan' => 'Labuan',

            'pjy' => 'Putrajaya',
            'wp putrajaya' => 'Putrajaya',
            'putrajaya' => 'Putrajaya',
        ];

        // Return mapped state name or original if not found
        return $stateMapping[$stateLower] ?? ucwords($state);
    }

    private function getTopProducts($query)
    {
        // OPTIMIZED: Use chunking to avoid loading all orders into memory
        $productStats = [];

        $query->select('id', 'line_items')->chunk(1000, function ($orders) use (&$productStats) {
            foreach ($orders as $order) {
                if (isset($order->line_items) && is_array($order->line_items)) {
                    foreach ($order->line_items as $item) {
                        $lineItemQuantity = isset($item['quantity']) ? intval($item['quantity']) : 1;

                        // Check if this is a bundle item with component products
                        if (isset($item['bundle_items']) && is_array($item['bundle_items']) && count($item['bundle_items']) > 0) {
                            // Bundle: Count each component SKU separately
                            foreach ($item['bundle_items'] as $bundleItem) {
                                $sku = isset($bundleItem['sku']) && !empty(trim($bundleItem['sku'])) ? trim($bundleItem['sku']) : 'NO-SKU';
                                $name = $bundleItem['name'] ?? 'Unknown Product';
                                $bundleItemQuantity = isset($bundleItem['quantity']) ? intval($bundleItem['quantity']) : 1;

                                // Total quantity = line item quantity × bundle item quantity
                                $totalQuantity = $lineItemQuantity * $bundleItemQuantity;

                                if (!isset($productStats[$sku])) {
                                    $productStats[$sku] = [
                                        'sku' => $sku,
                                        'name' => $name,
                                        'units' => 0,
                                        'orders' => 0,
                                    ];
                                }

                                $productStats[$sku]['units'] += $totalQuantity;
                                $productStats[$sku]['orders']++;
                            }
                        } else {
                            // Single product: Count normally
                            $sku = isset($item['sku']) && !empty(trim($item['sku'])) ? trim($item['sku']) : 'NO-SKU';
                            $name = $item['name'] ?? 'Unknown Product';

                            if (!isset($productStats[$sku])) {
                                $productStats[$sku] = [
                                    'sku' => $sku,
                                    'name' => $name,
                                    'units' => 0,
                                    'orders' => 0,
                                ];
                            }

                            $productStats[$sku]['units'] += $lineItemQuantity;
                            $productStats[$sku]['orders']++;
                        }
                    }
                }
            }
        });

        // Sort by units sold descending
        usort($productStats, function($a, $b) {
            return $b['units'] - $a['units'];
        });

        return $productStats; // Return all products
    }

    private function getUnitsOut($query)
    {
        // OPTIMIZED: Use chunking to avoid loading all orders into memory
        $skuUnits = [];
        $totalUnits = 0;

        $query->select('id', 'line_items')->chunk(1000, function ($orders) use (&$skuUnits, &$totalUnits) {
            foreach ($orders as $order) {
                if (isset($order->line_items) && is_array($order->line_items)) {
                    foreach ($order->line_items as $item) {
                        $lineItemQuantity = isset($item['quantity']) ? intval($item['quantity']) : 1;

                        // Check if this is a bundle item with component products
                        if (isset($item['bundle_items']) && is_array($item['bundle_items']) && count($item['bundle_items']) > 0) {
                            // Bundle: Count each component SKU separately
                            foreach ($item['bundle_items'] as $bundleItem) {
                                $sku = isset($bundleItem['sku']) && !empty(trim($bundleItem['sku'])) ? trim($bundleItem['sku']) : 'NO-SKU';
                                $bundleItemQuantity = isset($bundleItem['quantity']) ? intval($bundleItem['quantity']) : 1;

                                // Total quantity = line item quantity × bundle item quantity
                                $totalQuantity = $lineItemQuantity * $bundleItemQuantity;

                                $skuUnits[$sku] = ($skuUnits[$sku] ?? 0) + $totalQuantity;
                                $totalUnits += $totalQuantity;
                            }
                        } else {
                            // Single product: Count normally
                            $sku = isset($item['sku']) && !empty(trim($item['sku'])) ? trim($item['sku']) : 'NO-SKU';

                            $skuUnits[$sku] = ($skuUnits[$sku] ?? 0) + $lineItemQuantity;
                            $totalUnits += $lineItemQuantity;
                        }
                    }
                }
            }
        });

        // Sort by units descending
        arsort($skuUnits);

        $unitsData = [];
        foreach ($skuUnits as $sku => $units) {
            $unitsData[] = [
                'sku' => $sku,
                'units' => $units,
            ];
        }

        return [
            'data' => $unitsData, // Return all SKUs
            'total_units' => $totalUnits,
        ];
    }

    private function getRepeatCustomers($startDate, $endDate, $analyticsSource = null, $user = null)
    {
        // Build query for ALL orders in the period
        $query = Order::whereBetween('date_created', [$startDate, $endDate]);

        // Apply role-based filtering first (for sellers and managers)
        if ($user && $user->isSeller()) {
            $storeIds = $user->getStoreIds();
            $query->where(function($q) use ($storeIds, $user) {
                $q->whereIn('store_id', $storeIds)
                  ->orWhere('created_by', $user->id);
            });
        } elseif ($user && $user->isManager()) {
            $storeIds = $user->getStoreIds();
            $sellerIds = $user->getManagedSellerIds();
            $query->where(function($q) use ($storeIds, $sellerIds) {
                $q->whereIn('store_id', $storeIds)
                  ->orWhereIn('created_by', $sellerIds);
            });
        }
        // Apply source filter if specified (for admins)
        elseif ($analyticsSource && $analyticsSource !== 'all') {
            if (strpos($analyticsSource, 'store_') === 0) {
                $storeId = str_replace('store_', '', $analyticsSource);
                $query->where('store_id', $storeId);
            } elseif (strpos($analyticsSource, 'webhook_') === 0) {
                $webhookId = str_replace('webhook_', '', $analyticsSource);
                $query->where('webhook_source_id', $webhookId);
            }
        }

        // OPTIMIZED: Use database aggregation with GROUP BY on phone/email
        $customerStats = DB::table(DB::raw('(' . $query->toSql() . ') as filtered_orders'))
            ->mergeBindings($query->getQuery())
            ->select(
                DB::raw("COALESCE(
                    JSON_UNQUOTE(JSON_EXTRACT(billing, '$.phone')),
                    LOWER(JSON_UNQUOTE(JSON_EXTRACT(billing, '$.email')))
                ) as customer_identifier"),
                DB::raw("TRIM(CONCAT(
                    COALESCE(JSON_UNQUOTE(JSON_EXTRACT(billing, '$.first_name')), ''),
                    ' ',
                    COALESCE(JSON_UNQUOTE(JSON_EXTRACT(billing, '$.last_name')), '')
                )) as name"),
                DB::raw("LOWER(JSON_UNQUOTE(JSON_EXTRACT(billing, '$.email'))) as email"),
                DB::raw("JSON_UNQUOTE(JSON_EXTRACT(billing, '$.phone')) as phone"),
                DB::raw('COUNT(*) as order_count'),
                DB::raw('SUM(total) as total_spent')
            )
            ->whereNotNull(DB::raw("COALESCE(
                JSON_UNQUOTE(JSON_EXTRACT(billing, '$.phone')),
                JSON_UNQUOTE(JSON_EXTRACT(billing, '$.email'))
            )"))
            ->groupBy('customer_identifier', 'name', 'email', 'phone')
            ->havingRaw('COUNT(*) > 1') // Only repeat customers
            ->orderByDesc('order_count')
            ->get();

        $totalCustomers = DB::table(DB::raw('(' . $query->toSql() . ') as filtered_orders'))
            ->mergeBindings($query->getQuery())
            ->selectRaw("COUNT(DISTINCT COALESCE(
                JSON_UNQUOTE(JSON_EXTRACT(billing, '$.phone')),
                LOWER(JSON_UNQUOTE(JSON_EXTRACT(billing, '$.email')))
            )) as total")
            ->value('total');

        $repeatCustomers = $customerStats->map(function ($customer) {
            return [
                'name' => $customer->name,
                'email' => $customer->email,
                'phone' => $customer->phone,
                'order_count' => (int)$customer->order_count,
                'total_spent' => (float)$customer->total_spent,
            ];
        })->toArray();

        return [
            'data' => $repeatCustomers,
            'total_repeat_customers' => count($repeatCustomers),
            'total_customers' => $totalCustomers,
            'repeat_rate' => $totalCustomers > 0 ? round((count($repeatCustomers) / $totalCustomers) * 100, 1) : 0,
        ];
    }

    private function getCustomerDatabase($startDate, $endDate, $analyticsSource = null, $user = null, $includeOrderDetails = false)
    {
        // For dashboard view (not export), return empty array to save memory/time
        // Customer database is ONLY used for exports, never displayed on dashboard
        if (!$includeOrderDetails) {
            return [
                'customers' => [],
                'order_details' => null,
                'total_customers' => 0
            ];
        }

        // Build query for ALL orders in the period
        $query = Order::whereBetween('date_created', [$startDate, $endDate]);

        // Apply role-based filtering first (for sellers and managers)
        if ($user && $user->isSeller()) {
            $storeIds = $user->getStoreIds();
            $query->where(function($q) use ($storeIds, $user) {
                $q->whereIn('store_id', $storeIds)
                  ->orWhere('created_by', $user->id);
            });
        } elseif ($user && $user->isManager()) {
            $storeIds = $user->getStoreIds();
            $sellerIds = $user->getManagedSellerIds();
            $query->where(function($q) use ($storeIds, $sellerIds) {
                $q->whereIn('store_id', $storeIds)
                  ->orWhereIn('created_by', $sellerIds);
            });
        }
        // Apply source filter if specified (for admins)
        elseif ($analyticsSource && $analyticsSource !== 'all') {
            if (strpos($analyticsSource, 'store_') === 0) {
                $storeId = str_replace('store_', '', $analyticsSource);
                $query->where('store_id', $storeId);
            } elseif (strpos($analyticsSource, 'webhook_') === 0) {
                $webhookId = str_replace('webhook_', '', $analyticsSource);
                $query->where('webhook_source_id', $webhookId);
            }
        }

        // Process order details ONLY when exporting (not for dashboard view)
        $orderDetails = null;
        if ($includeOrderDetails) {
            // Increase memory limit for large exports
            ini_set('memory_limit', '512M');
            set_time_limit(300); // 5 minutes for exports

            // Clone query for chunking (to preserve original for later use)
            $chunkQuery = clone $query;

            // Process orders in chunks to avoid memory exhaustion
            $customerOrders = [];
            $orderDetails = [];

            $chunkQuery->orderBy('date_created', 'desc')->chunk(500, function ($orders) use (&$customerOrders, &$orderDetails) {
            // First pass: build customer stats
            foreach ($orders as $order) {
                $billing = is_string($order->billing) ? json_decode($order->billing, true) : $order->billing;
                $phone = $billing['phone'] ?? null;
                $email = strtolower($billing['email'] ?? '');
                $customerId = $phone ?: $email;

                if ($customerId) {
                    if (!isset($customerOrders[$customerId])) {
                        $customerOrders[$customerId] = [
                            'first_date' => $order->date_created,
                            'last_date' => $order->date_created,
                            'count' => 0
                        ];
                    }
                    $customerOrders[$customerId]['count']++;
                    if ($order->date_created < $customerOrders[$customerId]['first_date']) {
                        $customerOrders[$customerId]['first_date'] = $order->date_created;
                    }
                    if ($order->date_created > $customerOrders[$customerId]['last_date']) {
                        $customerOrders[$customerId]['last_date'] = $order->date_created;
                    }
                }
            }

            // Second pass: build order details
            foreach ($orders as $order) {
                $billing = is_string($order->billing) ? json_decode($order->billing, true) : $order->billing;
                $shipping = is_string($order->shipping) ? json_decode($order->shipping, true) : $order->shipping;
                $lineItems = is_string($order->line_items) ? json_decode($order->line_items, true) : $order->line_items;

                $customerName = trim(($billing['first_name'] ?? '') . ' ' . ($billing['last_name'] ?? ''));
                $phone = $billing['phone'] ?? null;
                $email = strtolower($billing['email'] ?? '');
                $customerId = $phone ?: $email;

                // Get customer stats
                $customerStats = $customerOrders[$customerId] ?? ['first_date' => $order->date_created, 'last_date' => $order->date_created, 'count' => 1];

                // Extract products, SKUs, and quantities
                $products = [];
                $skus = [];
                $quantities = [];
                if (is_array($lineItems)) {
                    foreach ($lineItems as $item) {
                        $lineItemQuantity = isset($item['quantity']) ? intval($item['quantity']) : 1;

                        // Check if this is a bundle item with component products
                        if (isset($item['bundle_items']) && is_array($item['bundle_items']) && count($item['bundle_items']) > 0) {
                            // Bundle: Expand to show each component SKU
                            foreach ($item['bundle_items'] as $bundleItem) {
                                $products[] = $bundleItem['name'] ?? 'Unknown Product';
                                $skus[] = $bundleItem['sku'] ?? '';
                                $bundleItemQuantity = isset($bundleItem['quantity']) ? intval($bundleItem['quantity']) : 1;
                                $quantities[] = $lineItemQuantity * $bundleItemQuantity;
                            }
                        } else {
                            // Single product: Add normally
                            $products[] = $item['name'] ?? 'Unknown Product';
                            $skus[] = $item['sku'] ?? '';
                            $quantities[] = $lineItemQuantity;
                        }
                    }
                }

                $orderDetails[] = [
                    'customer_name' => $customerName,
                    'email' => $billing['email'] ?? '',
                    'phone' => $phone ?? '',
                    'address' => ($shipping['address_1'] ?? $billing['address_1'] ?? ''),
                    'city' => ($shipping['city'] ?? $billing['city'] ?? ''),
                    'state' => ($shipping['state'] ?? $billing['state'] ?? ''),
                    'postcode' => ($shipping['postcode'] ?? $billing['postcode'] ?? ''),
                    'first_order_date' => $customerStats['first_date']->format('Y-m-d'),
                    'last_order_date' => $customerStats['last_date']->format('Y-m-d'),
                    'products' => implode(', ', $products),
                    'skus' => implode(', ', $skus),
                    'quantities' => implode(', ', $quantities),
                    'total_orders' => $customerStats['count'],
                    'order_total' => (float)$order->total,
                ];
            }
        });
        }

        // Calculate summary for dashboard view (always needed, fast SQL aggregation)
        $customers = DB::table(DB::raw('(' . $query->toSql() . ') as filtered_orders'))
            ->mergeBindings($query->getQuery())
            ->select(
                DB::raw("COALESCE(
                    JSON_UNQUOTE(JSON_EXTRACT(billing, '$.phone')),
                    LOWER(JSON_UNQUOTE(JSON_EXTRACT(billing, '$.email')))
                ) as customer_identifier"),
                DB::raw("TRIM(CONCAT(
                    COALESCE(JSON_UNQUOTE(JSON_EXTRACT(billing, '$.first_name')), ''),
                    ' ',
                    COALESCE(JSON_UNQUOTE(JSON_EXTRACT(billing, '$.last_name')), '')
                )) as name"),
                DB::raw("LOWER(JSON_UNQUOTE(JSON_EXTRACT(billing, '$.email'))) as email"),
                DB::raw("JSON_UNQUOTE(JSON_EXTRACT(billing, '$.phone')) as phone"),
                DB::raw("COALESCE(JSON_UNQUOTE(JSON_EXTRACT(shipping, '$.address_1')), JSON_UNQUOTE(JSON_EXTRACT(billing, '$.address_1'))) as address"),
                DB::raw("COALESCE(JSON_UNQUOTE(JSON_EXTRACT(shipping, '$.city')), JSON_UNQUOTE(JSON_EXTRACT(billing, '$.city'))) as city"),
                DB::raw("COALESCE(JSON_UNQUOTE(JSON_EXTRACT(shipping, '$.state')), JSON_UNQUOTE(JSON_EXTRACT(billing, '$.state'))) as state"),
                DB::raw("COALESCE(JSON_UNQUOTE(JSON_EXTRACT(shipping, '$.postcode')), JSON_UNQUOTE(JSON_EXTRACT(billing, '$.postcode'))) as postcode"),
                DB::raw('MIN(DATE(date_created)) as first_order_date'),
                DB::raw('MAX(DATE(date_created)) as last_order_date'),
                DB::raw('COUNT(*) as order_count'),
                DB::raw('SUM(total) as total_spent')
            )
            ->whereNotNull(DB::raw("COALESCE(
                JSON_UNQUOTE(JSON_EXTRACT(billing, '$.phone')),
                JSON_UNQUOTE(JSON_EXTRACT(billing, '$.email'))
            )"))
            ->groupBy('customer_identifier', 'name', 'email', 'phone', 'address', 'city', 'state', 'postcode')
            ->orderBy('name')
            ->get()
            ->map(function ($customer) {
                return [
                    'name' => $customer->name,
                    'email' => $customer->email,
                    'phone' => $customer->phone,
                    'address' => $customer->address ?? '',
                    'city' => $customer->city ?? '',
                    'state' => $customer->state ?? '',
                    'postcode' => $customer->postcode ?? '',
                    'first_order_date' => $customer->first_order_date,
                    'last_order_date' => $customer->last_order_date,
                    'order_count' => (int)$customer->order_count,
                    'total_spent' => (float)$customer->total_spent,
                ];
            })
            ->toArray();

        return [
            'customers' => $customers,
            'total_customers' => count($customers),
            'order_details' => $orderDetails, // For export with product details
        ];
    }

    private function calculateDateRange($period, $startDate = null, $endDate = null)
    {
        $now = Carbon::now();

        return match($period) {
            'today' => [
                'start' => $now->copy()->startOfDay(),
                'end' => $now->copy()->endOfDay(),
                'label' => 'Today'
            ],
            'yesterday' => [
                'start' => $now->copy()->subDay()->startOfDay(),
                'end' => $now->copy()->subDay()->endOfDay(),
                'label' => 'Yesterday'
            ],
            'this_week' => [
                'start' => $now->copy()->startOfWeek(),
                'end' => $now->copy()->endOfWeek(),
                'label' => 'This Week'
            ],
            'last_week' => [
                'start' => $now->copy()->startOfWeek()->subWeek(),
                'end' => $now->copy()->startOfWeek()->subWeek()->endOfWeek(),
                'label' => 'Last Week'
            ],
            'this_month' => [
                'start' => $now->copy()->startOfMonth(),
                'end' => $now->copy()->endOfMonth(),
                'label' => 'This Month'
            ],
            'last_month' => [
                'start' => $now->copy()->startOfMonth()->subMonth(),
                'end' => $now->copy()->startOfMonth()->subMonth()->endOfMonth(),
                'label' => 'Last Month'
            ],
            'this_year' => [
                'start' => $now->copy()->startOfYear(),
                'end' => $now->copy()->endOfYear(),
                'label' => 'This Year'
            ],
            'last_year' => [
                'start' => $now->copy()->startOfYear()->subYear(),
                'end' => $now->copy()->startOfYear()->subYear()->endOfYear(),
                'label' => 'Last Year'
            ],
            'custom' => [
                'start' => $startDate ? Carbon::parse($startDate)->startOfDay() : $now->copy()->startOfMonth(),
                'end' => $endDate ? Carbon::parse($endDate)->endOfDay() : $now->copy()->endOfDay(),
                'label' => 'Custom Range'
            ],
            default => [
                'start' => $now->copy()->startOfDay(),
                'end' => $now->copy()->endOfDay(),
                'label' => 'Today'
            ]
        };
    }

    public function exportAnalytics(Request $request)
    {
        $user = $request->user();

        // Get date range parameters
        $period = $request->get('period', 'today');
        $startDate = $request->get('start_date');
        $endDate = $request->get('end_date');
        $analyticsSource = $request->get('analytics_source');

        // For sellers and managers, override analytics source
        if ($user->isSeller() || $user->isManager()) {
            $analyticsSource = null;
        }

        // Calculate date range
        $dateRange = $this->calculateDateRange($period, $startDate, $endDate);

        // Determine source name for the report
        $sourceName = 'All Sources';
        if ($user->isSeller()) {
            $storeNames = $user->stores()->pluck('name')->toArray();
            $sourceName = 'My Stores (' . implode(', ', $storeNames) . ')';
        } elseif ($user->isManager()) {
            $sellerNames = $user->managedSellers()->pluck('name')->toArray();
            $sourceName = 'Managed Sellers (' . implode(', ', $sellerNames) . ')';
        } elseif ($analyticsSource && $analyticsSource !== 'all') {
            if (strpos($analyticsSource, 'store_') === 0) {
                $store = Store::find(str_replace('store_', '', $analyticsSource));
                $sourceName = $store ? $store->name : 'Unknown Store';
            } elseif (strpos($analyticsSource, 'webhook_') === 0) {
                $webhook = WebhookSource::find(str_replace('webhook_', '', $analyticsSource));
                $sourceName = $webhook ? $webhook->name : 'Unknown Webhook';
            }
        }

        // Check if date range exceeds 31 days
        $days = $dateRange['start']->diffInDays($dateRange['end']) + 1;

        if ($days > 31) {
            // Calculate number of months
            $monthCount = $dateRange['start']->diffInMonths($dateRange['end']) + 1;

            // Add flash message to inform user
            session()->flash('info', "Generating {$monthCount} monthly reports... This may take 1-2 minutes. Your download will start automatically.");

            // Multi-month export: Generate separate files and ZIP them
            return $this->exportMultiMonth($dateRange, $analyticsSource, $user, $sourceName);
        }

        // Single export for date ranges <= 31 days
        $advancedAnalytics = $this->getAdvancedAnalytics($dateRange, $analyticsSource, $user, true);
        $periodLabel = $dateRange['label'] . ' (' . $dateRange['start']->format('Y-m-d') . ' to ' . $dateRange['end']->format('Y-m-d') . ')';
        $filename = 'Analytics_Report_' . str_replace(' ', '_', $dateRange['label']) . '_' . now()->format('Y-m-d_His') . '.xlsx';

        return Excel::download(
            new AnalyticsExport($advancedAnalytics, $periodLabel, $sourceName),
            $filename
        );
    }

    private function exportMultiMonth($dateRange, $analyticsSource, $user, $sourceName)
    {
        // Increase limits for multi-month export
        ini_set('memory_limit', '1G');
        set_time_limit(600); // 10 minutes

        // Create unique temp directory
        $tempId = uniqid('export_', true);
        $tempDir = storage_path("app/temp/exports/{$tempId}");
        mkdir($tempDir, 0755, true);

        $files = [];
        $currentDate = $dateRange['start']->copy()->startOfMonth();
        $endDate = $dateRange['end']->copy()->endOfMonth();

        try {
            // Generate one Excel file per month
            while ($currentDate <= $endDate) {
                $monthStart = $currentDate->copy()->startOfMonth();
                $monthEnd = $currentDate->copy()->endOfMonth();

                // Don't go beyond user's selected end date
                if ($monthEnd > $dateRange['end']) {
                    $monthEnd = $dateRange['end']->copy();
                }
                if ($monthStart < $dateRange['start']) {
                    $monthStart = $dateRange['start']->copy();
                }

                // Get analytics for this month
                $monthRange = [
                    'start' => $monthStart,
                    'end' => $monthEnd,
                    'label' => $monthStart->format('F Y')
                ];

                $advancedAnalytics = $this->getAdvancedAnalytics($monthRange, $analyticsSource, $user, true);
                $periodLabel = $monthStart->format('F Y') . ' (' . $monthStart->format('Y-m-d') . ' to ' . $monthEnd->format('Y-m-d') . ')';

                // Generate filename for this month
                $monthFilename = 'Analytics_' . $monthStart->format('Y-m') . '.xlsx';
                $monthFilePath = $tempDir . DIRECTORY_SEPARATOR . $monthFilename;

                try {
                    // Generate Excel file content in memory
                    $export = new AnalyticsExport($advancedAnalytics, $periodLabel, $sourceName);
                    $fileContent = Excel::raw($export, \Maatwebsite\Excel\Excel::XLSX);

                    // Write directly to filesystem
                    file_put_contents($monthFilePath, $fileContent);

                    if (file_exists($monthFilePath) && filesize($monthFilePath) > 0) {
                        $files[] = $monthFilePath;
                    } else {
                        throw new \Exception("Failed to create export file: {$monthFilename}");
                    }
                } catch (\Exception $e) {
                    \Log::error("Excel export error for month {$monthStart->format('Y-m')}", [
                        'error' => $e->getMessage(),
                        'file' => $e->getFile(),
                        'line' => $e->getLine()
                    ]);
                    throw new \Exception("Export failed for {$monthStart->format('F Y')}: {$e->getMessage()}");
                }

                // Move to next month
                $currentDate->addMonth();
            }

            // Create ZIP file
            $zipFilename = 'Analytics_Report_' . $dateRange['start']->format('Y-m') . '_to_' . $dateRange['end']->format('Y-m') . '.zip';
            $zipPath = $tempDir . DIRECTORY_SEPARATOR . $zipFilename;

            $zip = new \ZipArchive();
            $zipOpen = $zip->open($zipPath, \ZipArchive::CREATE);

            if ($zipOpen !== true) {
                throw new \Exception("Failed to create ZIP file. Error code: {$zipOpen}");
            }

            foreach ($files as $file) {
                if (!$zip->addFile($file, basename($file))) {
                    \Log::error("Failed to add file to ZIP: {$file}");
                }
            }

            $zip->close();

            // Download ZIP and schedule cleanup
            return response()->download($zipPath, $zipFilename)->deleteFileAfterSend(true);

        } finally {
            // Register shutdown function to cleanup temp directory
            register_shutdown_function(function() use ($tempDir) {
                if (file_exists($tempDir)) {
                    // Delete all files in directory
                    array_map('unlink', glob("$tempDir/*.*"));
                    // Delete directory
                    rmdir($tempDir);
                }
            });
        }
    }

    /**
     * Get top sales/sellers performance for Admin and Manager
     * Includes historical data from SellerForecast table
     */
    private function getTopSales($dateRange, $user)
    {
        $startDate = $dateRange['start'];
        $endDate = $dateRange['end'];

        // Get all sellers to analyze
        $sellersQuery = \App\Models\User::where('role', 'seller');

        // Filter for managers - only their assigned sellers
        if ($user->isManager()) {
            $sellerIds = $user->managedSellers()->pluck('users.id')->toArray();
            $sellersQuery->whereIn('id', $sellerIds);
        }

        $sellers = $sellersQuery->get();

        // Calculate stats for each seller
        $sellerStats = [];
        foreach ($sellers as $seller) {
            // Get live orders from seller's stores + manually created orders
            $storeIds = $seller->stores()->pluck('stores.id')->toArray();

            $ordersQuery = \App\Models\Order::whereBetween('date_created', [$startDate, $endDate])
                ->where(function($q) use ($storeIds, $seller) {
                    if (count($storeIds) > 0) {
                        $q->whereIn('store_id', $storeIds);
                    }
                    $q->orWhere('created_by', $seller->id);
                })
                ->whereIn('status', ['completed', 'processing', 'approval', 'printed']); // Count all active order statuses

            // OPTIMIZED: Use database aggregation instead of loading all orders
            $totalOrders = $ordersQuery->count();
            $totalRevenue = $ordersQuery->sum('total');

            // Add historical data from SellerForecast (manually entered past data)
            // IMPORTANT: Exclude current month to avoid double counting with live orders
            $historicalQuery = \App\Models\SellerForecast::where('user_id', $seller->id);

            // Apply date filters for historical data
            $historicalQuery->where(function($q) use ($startDate, $endDate) {
                $q->where(function($sq) use ($startDate, $endDate) {
                    $sq->where('year', '>=', $startDate->year)
                       ->where('year', '<=', $endDate->year);
                });
            });

            // Further filter by months if needed
            if ($startDate->year === $endDate->year) {
                $historicalQuery->whereBetween('month', [$startDate->month, $endDate->month]);
            }

            // Exclude current month to prevent double counting
            $currentYear = now()->year;
            $currentMonth = now()->month;
            $historicalQuery->where(function($q) use ($currentYear, $currentMonth) {
                $q->where('year', '<', $currentYear)
                  ->orWhere(function($sq) use ($currentYear, $currentMonth) {
                      $sq->where('year', '=', $currentYear)
                         ->where('month', '<', $currentMonth);
                  });
            });

            $historicalOrders = $historicalQuery->sum('orders_count') ?? 0;
            $historicalRevenue = $historicalQuery->sum('actual_sales') ?? 0;

            // Combine live + historical data
            $totalOrders += $historicalOrders;
            $totalRevenue += $historicalRevenue;

            if ($totalOrders > 0) {
                $sellerStats[] = (object)[
                    'id' => $seller->id,
                    'name' => $seller->name,
                    'email' => $seller->email,
                    'total_orders' => $totalOrders,
                    'total_revenue' => $totalRevenue,
                    'avg_order_value' => $totalRevenue / $totalOrders,
                    'has_historical_data' => $historicalRevenue > 0, // Flag for UI
                ];
            }
        }

        // Sort by revenue and get top 10
        $topSellers = collect($sellerStats)
            ->sortByDesc('total_revenue')
            ->take(10)
            ->values();

        return $topSellers;
    }
}
