<?php

namespace App\Http\Controllers;

use App\Models\Customer;
use App\Helpers\PhoneHelper;
use Illuminate\Http\Request;

class CustomerController extends Controller
{
    /**
     * Lookup customer by phone number for auto-fill
     * Privacy: Only returns customers this seller has orders with
     */
    public function lookup(Request $request)
    {
        $request->validate([
            'phone' => 'required|string|min:10'
        ]);

        $user = auth()->user();

        // Check if user has auto-fill enabled
        if (!$user->auto_fill_customer_info) {
            return response()->json([
                'found' => false,
                'message' => 'Auto-fill is disabled'
            ]);
        }

        // Format phone number using PhoneHelper
        $phone = PhoneHelper::formatMalaysian($request->phone);

        // Find customer by phone number
        $customer = Customer::where('phone', $phone)->first();

        if (!$customer) {
            return response()->json([
                'found' => false
            ]);
        }

        // PRIVACY CHECK: Only return if current user has orders for this customer
        // Check 3 sources:
        // 1. Orders with customer_id (new checkout orders)
        // 2. Orders with phone in billing JSON (old checkout orders)
        // 3. Sales page orders from seller's sales pages

        $hasCheckoutOrders = \App\Models\Order::where(function($query) use ($customer, $user) {
            // New orders with customer_id
            $query->where('customer_id', $customer->id)
                  ->where('created_by', $user->id);
        })->orWhere(function($query) use ($phone, $user) {
            // Old orders with phone in billing JSON
            $query->whereRaw("JSON_EXTRACT(billing, '$.phone') = ?", [$phone])
                  ->where('created_by', $user->id);
        })->exists();

        $hasSalesPageOrders = \App\Models\SalesPageOrder::where('customer_phone', $phone)
            ->whereHas('salesPage', function($query) use ($user) {
                $query->where('user_id', $user->id);
            })->exists();

        if (!$hasCheckoutOrders && !$hasSalesPageOrders) {
            return response()->json([
                'found' => false
            ]);
        }

        return response()->json([
            'found' => true,
            'customer' => [
                'name' => $customer->name,
                'email' => $customer->email,
                'address' => $customer->address,
                'city' => $customer->city,
                'state' => $customer->state,
                'postcode' => $customer->postcode,
            ]
        ]);
    }
}
