<?php

namespace App\Http\Controllers;

use App\Models\SystemSetting;
use App\Services\CloudflareService;
use Illuminate\Http\Request;

class CloudflareSettingsController extends Controller
{
    /**
     * Display Cloudflare settings page
     */
    public function index()
    {
        $settings = [
            'cloudflare_api_token' => SystemSetting::getCloudflareApiToken(),
            'cloudflare_zone_id' => SystemSetting::getCloudflareZoneId(),
            'cloudflare_email' => SystemSetting::getCloudflareEmail(),
            'server_ip' => SystemSetting::getServerIp(),
            'auto_dns_enabled' => SystemSetting::isAutoDnsEnabled(),
            'auto_verification_enabled' => SystemSetting::isAutoVerificationEnabled(),
        ];

        $isConfigured = SystemSetting::isCloudflareEnabled();

        return view('cloudflare-settings.index', compact('settings', 'isConfigured'));
    }

    /**
     * Update Cloudflare settings
     */
    public function update(Request $request)
    {
        $request->validate([
            'cloudflare_api_token' => 'nullable|string|max:255',
            'cloudflare_zone_id' => 'nullable|string|max:255',
            'cloudflare_email' => 'nullable|email|max:255',
            'server_ip' => 'nullable|ip',
            'auto_dns_enabled' => 'boolean',
            'auto_verification_enabled' => 'boolean',
        ]);

        // Save settings
        SystemSetting::set('cloudflare_api_token', $request->cloudflare_api_token, 'string');
        SystemSetting::set('cloudflare_zone_id', $request->cloudflare_zone_id, 'string');
        SystemSetting::set('cloudflare_email', $request->cloudflare_email, 'string');
        SystemSetting::set('server_ip', $request->server_ip, 'string');
        SystemSetting::set('auto_dns_enabled', $request->auto_dns_enabled ? 'true' : 'false', 'string');
        SystemSetting::set('auto_verification_enabled', $request->auto_verification_enabled ? 'true' : 'false', 'string');

        return redirect()->route('cloudflare-settings.index')
            ->with('success', 'Cloudflare settings updated successfully!');
    }

    /**
     * Test Cloudflare connection
     */
    public function testConnection()
    {
        $cloudflare = new CloudflareService();

        $result = $cloudflare->testConnection();

        if ($result['success']) {
            return back()->with('success', $result['message'] . (isset($result['zone_name']) ? ' (Zone: ' . $result['zone_name'] . ')' : ''));
        }

        return back()->with('error', $result['message']);
    }

    /**
     * Get zone details
     */
    public function getZoneDetails()
    {
        $cloudflare = new CloudflareService();

        $result = $cloudflare->getZoneDetails();

        if ($result['success']) {
            $zone = $result['zone'];
            return response()->json([
                'success' => true,
                'zone_name' => $zone['name'] ?? 'N/A',
                'status' => $zone['status'] ?? 'N/A',
                'name_servers' => $zone['name_servers'] ?? [],
            ]);
        }

        return response()->json([
            'success' => false,
            'message' => $result['message'],
        ], 400);
    }

    /**
     * List DNS records
     */
    public function listDnsRecords(Request $request)
    {
        $cloudflare = new CloudflareService();

        $result = $cloudflare->listDnsRecords(
            $request->get('type'),
            $request->get('name')
        );

        if ($result['success']) {
            return response()->json([
                'success' => true,
                'records' => $result['records'],
            ]);
        }

        return response()->json([
            'success' => false,
            'message' => $result['message'],
        ], 400);
    }
}
