<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;

class ManagerController extends Controller
{
    /**
     * Display a listing of managers.
     */
    public function index()
    {
        $managers = User::where('role', 'manager')
            ->withCount('managedSellers')
            ->orderBy('name')
            ->paginate(25);

        return view('admin.managers.index', compact('managers'));
    }

    /**
     * Show the form for assigning sellers to a manager.
     */
    public function assignSellers($id)
    {
        $manager = User::where('role', 'manager')->findOrFail($id);
        $manager->load('managedSellers');

        // Get all sellers
        $allSellers = User::where('role', 'seller')
            ->orderBy('name')
            ->get();

        // Get currently assigned seller IDs
        $assignedSellerIds = $manager->managedSellers->pluck('id')->toArray();

        return view('admin.managers.assign-sellers', compact('manager', 'allSellers', 'assignedSellerIds'));
    }

    /**
     * Store seller assignments for a manager.
     */
    public function storeSellers(Request $request, $id)
    {
        $manager = User::where('role', 'manager')->findOrFail($id);

        $validated = $request->validate([
            'seller_ids' => 'nullable|array',
            'seller_ids.*' => 'exists:users,id',
        ]);

        $sellerIds = $validated['seller_ids'] ?? [];

        // Sync the sellers (add new, remove unselected)
        $manager->managedSellers()->sync($sellerIds);

        return redirect()->route('admin.managers.assign-sellers', $manager)
            ->with('success', 'Sellers assigned successfully!');
    }

    /**
     * Remove a seller from a manager.
     */
    public function removeSeller($managerId, $sellerId)
    {
        $manager = User::where('role', 'manager')->findOrFail($managerId);
        $manager->managedSellers()->detach($sellerId);

        return redirect()->route('admin.managers.assign-sellers', $manager)
            ->with('success', 'Seller removed successfully!');
    }
}
