<?php

namespace App\Helpers;

class PhoneHelper
{
    /**
     * Format phone number to Malaysian format with +60 country code
     *
     * Examples:
     * - 0123456789 -> +60123456789
     * - 123456789 -> +60123456789
     * - +60123456789 -> +60123456789
     * - 60123456789 -> +60123456789
     *
     * @param string|null $phone
     * @return string|null
     */
    public static function formatMalaysian(?string $phone): ?string
    {
        if (empty($phone)) {
            return null;
        }

        // Remove all non-numeric characters except leading +
        $phone = preg_replace('/[^\d+]/', '', $phone);

        // If empty after cleanup, return null
        if (empty($phone)) {
            return null;
        }

        // Remove leading + temporarily for processing
        $hasPlus = str_starts_with($phone, '+');
        $phone = ltrim($phone, '+');

        // Remove leading 00 (international prefix)
        if (str_starts_with($phone, '00')) {
            $phone = substr($phone, 2);
        }

        // Handle different formats
        if (str_starts_with($phone, '60')) {
            // Already has country code
            $phone = '60' . ltrim(substr($phone, 2), '0');
        } elseif (str_starts_with($phone, '0')) {
            // Malaysian number with leading 0
            $phone = '60' . substr($phone, 1);
        } else {
            // No country code, no leading 0
            $phone = '60' . $phone;
        }

        return '+' . $phone;
    }

    /**
     * Get WhatsApp URL for a phone number
     *
     * @param string|null $phone
     * @param string|null $message
     * @return string|null
     */
    public static function getWhatsAppUrl(?string $phone, ?string $message = null): ?string
    {
        $formattedPhone = self::formatMalaysian($phone);

        if (empty($formattedPhone)) {
            return null;
        }

        // Remove + for WhatsApp URL (wa.me uses numbers without +)
        $phoneForUrl = ltrim($formattedPhone, '+');

        $url = "https://wa.me/{$phoneForUrl}";

        if ($message) {
            $url .= '?text=' . urlencode($message);
        }

        return $url;
    }

    /**
     * Display formatted phone number (for UI display)
     *
     * @param string|null $phone
     * @return string|null
     */
    public static function display(?string $phone): ?string
    {
        $formatted = self::formatMalaysian($phone);

        if (empty($formatted)) {
            return null;
        }

        // Format as +60 12-345 6789 for better readability
        if (preg_match('/^\+60(\d{2})(\d{3,4})(\d{4})$/', $formatted, $matches)) {
            return "+60 {$matches[1]}-{$matches[2]} {$matches[3]}";
        }

        return $formatted;
    }

    /**
     * Validate if phone number is valid Malaysian format
     *
     * @param string|null $phone
     * @return bool
     */
    public static function isValid(?string $phone): bool
    {
        if (empty($phone)) {
            return false;
        }

        $formatted = self::formatMalaysian($phone);

        if (empty($formatted)) {
            return false;
        }

        // Malaysian phone numbers should be +60 followed by 9-10 digits
        // Mobile: +60 1x-xxx-xxxx (10-11 digits after +60)
        // Landline: +60 x-xxx-xxxx (9-10 digits after +60)
        return preg_match('/^\+60\d{9,11}$/', $formatted) === 1;
    }
}
