<?php

namespace App\Exports;

use Maatwebsite\Excel\Concerns\WithMultipleSheets;

class SalesReportExport implements WithMultipleSheets
{
    protected $orders;
    protected $dateRange;
    protected $user;

    public function __construct($orders, $dateRange, $user)
    {
        $this->orders = $orders;
        $this->dateRange = $dateRange;
        $this->user = $user;
    }

    public function sheets(): array
    {
        return [
            new SummarySheet($this->orders, $this->dateRange, $this->user),
            new DetailedTransactionsSheet($this->orders),
        ];
    }
}

/**
 * Summary Sheet - Contains all summary information
 */
class SummarySheet implements
    \Maatwebsite\Excel\Concerns\FromCollection,
    \Maatwebsite\Excel\Concerns\WithTitle,
    \Maatwebsite\Excel\Concerns\WithStyles,
    \Maatwebsite\Excel\Concerns\WithColumnWidths
{
    protected $orders;
    protected $dateRange;
    protected $user;

    public function __construct($orders, $dateRange, $user)
    {
        $this->orders = $orders;
        $this->dateRange = $dateRange;
        $this->user = $user;
    }

    public function collection()
    {
        $totalOrders = $this->orders->count();
        $totalRevenue = $this->orders->sum('total');
        $avgOrderValue = $totalOrders > 0 ? $totalRevenue / $totalOrders : 0;

        // Find top platform
        $platformCounts = $this->orders->groupBy(function($order) {
            return $this->getPlatformName($order);
        })->map->count();
        $topPlatform = $platformCounts->sortDesc()->keys()->first() ?? 'N/A';

        // Platform breakdown
        $platformBreakdown = $this->orders->groupBy(function($order) {
            return $this->getPlatformName($order);
        })->map(function($platformOrders) {
            $count = $platformOrders->count();
            $revenue = $platformOrders->sum('total');
            $percentage = $this->orders->count() > 0 ? ($count / $this->orders->count()) * 100 : 0;

            return [
                'platform' => $platformOrders->first() ? $this->getPlatformName($platformOrders->first()) : 'N/A',
                'count' => $count,
                'revenue' => $revenue,
                'percentage' => $percentage,
            ];
        })->sortByDesc('revenue')->values();

        // Payment breakdown
        $paymentBreakdown = $this->orders->groupBy(function($order) {
            return $this->getPaymentMethodName($order->payment_method);
        })->map(function($paymentOrders) {
            return [
                'method' => $this->getPaymentMethodName($paymentOrders->first()->payment_method),
                'count' => $paymentOrders->count(),
                'revenue' => $paymentOrders->sum('total'),
            ];
        })->sortByDesc('revenue')->values();

        // Status breakdown
        $statusBreakdown = $this->orders->groupBy('status')->map(function($statusOrders, $status) {
            return [
                'status' => ucfirst($status),
                'count' => $statusOrders->count(),
                'revenue' => $statusOrders->sum('total'),
            ];
        })->sortByDesc('count')->values();

        $data = collect();

        // Header
        $data->push(['SALES REPORT', '', '', '']);
        $data->push(['Report Period:', $this->dateRange['label'], '', '']);
        $data->push(['Generated By:', $this->user->name, '', '']);
        $data->push(['Generated On:', now()->format('M d, Y h:i A'), '', '']);
        $data->push(['', '', '', '']);

        // Row 1: Summary Statistics (3 cards)
        $data->push(['SUMMARY STATISTICS', '', '', '']);
        $data->push(['Total Orders', 'Total Revenue', 'Avg Order Value', '']);
        $data->push([$totalOrders, 'RM ' . number_format($totalRevenue, 2), 'RM ' . number_format($avgOrderValue, 2), '']);
        $data->push(['', '', '', '']);

        // Row 2: Top Platform + Platform Breakdown
        $data->push(['TOP PLATFORM & BREAKDOWN', '', '', '']);
        $data->push(['Top Platform', 'Platform', 'Orders', 'Revenue']);
        $firstPlatform = true;
        foreach ($platformBreakdown as $item) {
            if ($firstPlatform) {
                $data->push([
                    $topPlatform,
                    $item['platform'],
                    $item['count'] . ' (' . number_format($item['percentage'], 1) . '%)',
                    'RM ' . number_format($item['revenue'], 2)
                ]);
                $firstPlatform = false;
            } else {
                $data->push([
                    '',
                    $item['platform'],
                    $item['count'] . ' (' . number_format($item['percentage'], 1) . '%)',
                    'RM ' . number_format($item['revenue'], 2)
                ]);
            }
        }
        $data->push(['', '', '', '']);

        // Row 3: Payment Methods + Order Status
        $data->push(['PAYMENT METHODS & ORDER STATUS', '', '', '']);
        $data->push(['Payment Method', 'Orders', 'Order Status', 'Orders']);

        $maxRows = max($paymentBreakdown->count(), $statusBreakdown->count());
        for ($i = 0; $i < $maxRows; $i++) {
            $payment = $paymentBreakdown->get($i);
            $status = $statusBreakdown->get($i);

            $data->push([
                $payment ? $payment['method'] : '',
                $payment ? $payment['count'] . ' orders (RM ' . number_format($payment['revenue'], 2) . ')' : '',
                $status ? $status['status'] : '',
                $status ? $status['count'] . ' orders (RM ' . number_format($status['revenue'], 2) . ')' : '',
            ]);
        }

        return $data;
    }

    public function title(): string
    {
        return 'Summary';
    }

    public function styles(\PhpOffice\PhpSpreadsheet\Worksheet\Worksheet $sheet)
    {
        return [
            // Header row
            1 => [
                'font' => ['bold' => true, 'size' => 16, 'color' => ['rgb' => 'FFFFFF']],
                'fill' => [
                    'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                    'startColor' => ['rgb' => '4472C4']
                ],
            ],
            // Report info rows
            2 => ['font' => ['bold' => true]],
            3 => ['font' => ['bold' => true]],
            4 => ['font' => ['bold' => true]],

            // Section headers
            6 => [
                'font' => ['bold' => true, 'size' => 12],
                'fill' => [
                    'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                    'startColor' => ['rgb' => 'E7E6E6']
                ],
            ],
            7 => ['font' => ['bold' => true]],
            10 => [
                'font' => ['bold' => true, 'size' => 12],
                'fill' => [
                    'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                    'startColor' => ['rgb' => 'E7E6E6']
                ],
            ],
            11 => ['font' => ['bold' => true]],
        ];
    }

    public function columnWidths(): array
    {
        return [
            'A' => 25,
            'B' => 25,
            'C' => 30,
            'D' => 25,
        ];
    }

    private function getPlatformName($order)
    {
        if (str_starts_with($order->order_number, 'CK-')) {
            return 'Checkout Form';
        } elseif (str_starts_with($order->order_number, 'SP-')) {
            return 'Sales Page';
        } elseif (str_starts_with($order->order_number, 'WOO-') || $order->store_id) {
            return 'WooCommerce';
        } elseif (str_starts_with($order->order_number, 'CH-')) {
            return 'Channel';
        } else {
            return 'Manual';
        }
    }

    private function getPaymentMethodName($method)
    {
        if (str_contains(strtolower($method ?? ''), 'cod')) {
            return 'COD';
        } elseif (in_array(strtolower($method ?? ''), ['bayarcash', 'bayarcash-wc', 'online', 'fpx'])) {
            return 'Online Payment';
        } else {
            return ucfirst($method ?? 'N/A');
        }
    }
}

/**
 * Detailed Transactions Sheet
 */
class DetailedTransactionsSheet implements
    \Maatwebsite\Excel\Concerns\FromCollection,
    \Maatwebsite\Excel\Concerns\WithTitle,
    \Maatwebsite\Excel\Concerns\WithHeadings,
    \Maatwebsite\Excel\Concerns\WithStyles,
    \Maatwebsite\Excel\Concerns\WithColumnWidths
{
    protected $orders;

    public function __construct($orders)
    {
        $this->orders = $orders;
    }

    public function collection()
    {
        $data = collect();

        foreach ($this->orders as $order) {
            $products = $this->getProductsList($order);
            $productNames = collect($products)->map(function($p) {
                return $p['name'] . ' x' . $p['quantity'];
            })->join("\n");

            $data->push([
                \Carbon\Carbon::parse($order->created_at)->format('M d, Y'),
                \Carbon\Carbon::parse($order->created_at)->format('h:i A'),
                $order->order_number,
                $this->getCustomerName($order),
                $this->getCustomerPhone($order),
                $this->getCustomerAddress($order),
                $productNames,
                ucfirst($order->status),
                'RM ' . number_format($order->total, 2),
            ]);
        }

        // Add total row
        $data->push([
            '', '', '', '', '', '', '', 'TOTAL:',
            'RM ' . number_format($this->orders->sum('total'), 2),
        ]);

        return $data;
    }

    public function headings(): array
    {
        return [
            'Date',
            'Time',
            'Order ID',
            'Customer Name',
            'Phone',
            'Address',
            'Products',
            'Status',
            'Total',
        ];
    }

    public function title(): string
    {
        return 'Detailed Transactions';
    }

    public function styles(\PhpOffice\PhpSpreadsheet\Worksheet\Worksheet $sheet)
    {
        $lastRow = $this->orders->count() + 2;

        // Enable text wrapping for all cells
        $sheet->getStyle('A2:I' . $lastRow)->getAlignment()->setWrapText(true);
        $sheet->getStyle('A2:I' . $lastRow)->getAlignment()->setVertical(\PhpOffice\PhpSpreadsheet\Style\Alignment::VERTICAL_TOP);

        return [
            1 => [
                'font' => ['bold' => true, 'size' => 11, 'color' => ['rgb' => 'FFFFFF']],
                'fill' => [
                    'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                    'startColor' => ['rgb' => '4472C4']
                ],
                'alignment' => [
                    'horizontal' => \PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_CENTER,
                    'vertical' => \PhpOffice\PhpSpreadsheet\Style\Alignment::VERTICAL_CENTER,
                ],
            ],
            $lastRow => [
                'font' => ['bold' => true],
                'fill' => [
                    'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                    'startColor' => ['rgb' => 'E7E6E6']
                ],
            ],
        ];
    }

    public function columnWidths(): array
    {
        return [
            'A' => 12,  // Date
            'B' => 10,  // Time
            'C' => 15,  // Order ID
            'D' => 20,  // Customer Name
            'E' => 15,  // Phone
            'F' => 35,  // Address
            'G' => 35,  // Products
            'H' => 12,  // Status
            'I' => 15,  // Total
        ];
    }

    private function getPlatformName($order)
    {
        if (str_starts_with($order->order_number, 'CK-')) {
            return 'Checkout Form';
        } elseif (str_starts_with($order->order_number, 'SP-')) {
            return 'Sales Page';
        } elseif (str_starts_with($order->order_number, 'WOO-') || $order->store_id) {
            return 'WooCommerce';
        } elseif (str_starts_with($order->order_number, 'CH-')) {
            return 'Channel';
        } else {
            return 'Manual';
        }
    }

    private function getPaymentMethodName($method)
    {
        if (str_contains(strtolower($method ?? ''), 'cod')) {
            return 'COD';
        } elseif (in_array(strtolower($method ?? ''), ['bayarcash', 'bayarcash-wc', 'online', 'fpx'])) {
            return 'Online Payment';
        } else {
            return ucfirst($method ?? 'N/A');
        }
    }

    private function getCustomerName($order)
    {
        if (is_array($order->billing) && isset($order->billing['first_name'])) {
            return trim(($order->billing['first_name'] ?? '') . ' ' . ($order->billing['last_name'] ?? ''));
        }
        return $order->customer_name ?? 'N/A';
    }

    private function getCustomerPhone($order)
    {
        if (is_array($order->billing) && isset($order->billing['phone'])) {
            return $order->billing['phone'];
        }
        return $order->customer_phone ?? 'N/A';
    }

    private function getCustomerAddress($order)
    {
        if (is_array($order->billing)) {
            $parts = array_filter([
                $order->billing['address_1'] ?? '',
                $order->billing['address_2'] ?? '',
                $order->billing['city'] ?? '',
                $order->billing['state'] ?? '',
                $order->billing['postcode'] ?? ''
            ]);
            return implode(', ', $parts) ?: 'N/A';
        }
        return $order->customer_address ?? 'N/A';
    }

    private function getProductsList($order)
    {
        $products = [];

        if (is_array($order->line_items) && count($order->line_items) > 0) {
            foreach ($order->line_items as $item) {
                $products[] = [
                    'name' => $item['name'] ?? 'Product',
                    'quantity' => $item['quantity'] ?? 1,
                    'price' => $item['total'] ?? 0
                ];
            }
        } else {
            $products[] = [
                'name' => 'Order Items',
                'quantity' => 1,
                'price' => $order->total
            ];
        }

        return $products;
    }
}
