<?php

namespace App\Console\Commands;

use App\Models\Store;
use App\Services\WooCommerceService;
use Illuminate\Console\Command;

class SyncWooOrders extends Command
{
    protected $signature = 'woo:sync-orders
                            {--store= : Specific store ID to sync}
                            {--historical : Sync historical orders (last 12 months)}
                            {--full : Sync ALL orders (no date limit)}
                            {--months= : Number of months to sync back (for historical)}';

    protected $description = 'Sync orders from WooCommerce stores';

    public function handle()
    {
        $this->info('Starting order synchronization...');

        // Determine sync type
        $fullSync = $this->option('full');
        $historical = $this->option('historical');
        $months = $this->option('months');

        // Show sync mode
        if ($fullSync) {
            $this->warn('FULL SYNC MODE - Syncing ALL orders (no date limit)');
            $this->warn('This may take a while...');
        } elseif ($historical) {
            $monthsBack = $months ?: 12;
            $this->info("HISTORICAL SYNC MODE - Syncing last {$monthsBack} months");
        } else {
            $this->info('REGULAR SYNC MODE - Syncing last 30 days');
        }

        // Get stores to sync
        if ($this->option('store')) {
            $stores = Store::where('id', $this->option('store'))
                          ->where('is_active', true)
                          ->get();
        } else {
            $stores = Store::where('is_active', true)->get();
        }

        if ($stores->isEmpty()) {
            $this->warn('No active stores found to sync.');
            return 0;
        }

        $totalSynced = 0;
        $bar = $this->output->createProgressBar($stores->count());
        $bar->start();

        foreach ($stores as $store) {
            try {
                $service = new WooCommerceService($store);

                // Choose sync method based on options
                if ($fullSync) {
                    // Full sync - all orders, no date limit
                    $count = $service->syncOrders(true, null);
                } elseif ($historical || $months) {
                    // Historical sync with custom months
                    $monthsBack = $months ?: 12;
                    $count = $service->syncHistoricalOrders($monthsBack);
                } else {
                    // Regular sync - last 30 days
                    $count = $service->syncOrders();
                }

                $totalSynced += $count;

                $this->info("\n✓ Synced {$count} orders from {$store->name}");

                // Update last sync timestamp
                $store->update(['last_sync' => now()]);

            } catch (\Exception $e) {
                $this->error("\n✗ Failed to sync {$store->name}: " . $e->getMessage());
            }

            $bar->advance();
        }

        $bar->finish();

        $this->newLine(2);
        $this->info("Synchronization completed!");
        $this->info("Total orders synced: {$totalSynced}");

        return 0;
    }
}
