<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Store;
use App\Services\WooCommerceService;
use Carbon\Carbon;

class SyncStoreMonthly extends Command
{
    protected $signature = 'store:sync-monthly {store_id} {--months=24 : Number of months to sync back}';
    protected $description = 'Sync WooCommerce orders month by month (bypasses pagination limits)';

    public function handle()
    {
        $storeId = $this->argument('store_id');
        $monthsBack = $this->option('months');

        // Find store
        $store = Store::find($storeId);
        if (!$store) {
            $this->error("Store ID {$storeId} not found!");
            return 1;
        }

        $this->info("🚀 Starting monthly sync for: {$store->name}");
        $this->info("📅 Syncing last {$monthsBack} months of data\n");

        // Increase limits
        ini_set('memory_limit', '3G');
        set_time_limit(3600); // 60 minutes

        $service = new WooCommerceService($store);
        $totalSynced = 0;
        $startTime = time();

        // ALL statuses
        $statuses = ['processing', 'completed', 'pending', 'failed', 'on-hold', 'cancelled', 'refunded'];

        // Loop through each month from oldest to newest
        for ($i = $monthsBack - 1; $i >= 0; $i--) {
            $monthStart = Carbon::now()->subMonths($i)->startOfMonth();
            $monthEnd = Carbon::now()->subMonths($i)->endOfMonth();

            // Don't go into the future
            if ($monthEnd->isFuture()) {
                $monthEnd = Carbon::now();
            }

            $monthName = $monthStart->format('F Y');
            $this->line("📦 Syncing: {$monthName}");

            $monthSynced = 0;

            foreach ($statuses as $status) {
                $page = 1;
                $perPage = 100;

                do {
                    $params = [
                        'status' => $status,
                        'page' => $page,
                        'per_page' => $perPage,
                        'after' => $monthStart->toIso8601String(),
                        'before' => $monthEnd->toIso8601String(),
                        'orderby' => 'date',
                        'order' => 'desc'
                    ];

                    $orders = $service->getOrders($params);

                    if (empty($orders)) {
                        break;
                    }

                    // Process orders
                    foreach ($orders as $wooOrder) {
                        try {
                            \App\Models\Order::updateOrCreate(
                                [
                                    'store_id' => $store->id,
                                    'woo_order_id' => $wooOrder['id']
                                ],
                                [
                                    'order_number' => $wooOrder['number'],
                                    'status' => $wooOrder['status'],
                                    'total' => $wooOrder['total'],
                                    'currency' => $wooOrder['currency'],
                                    'payment_method' => $wooOrder['payment_method'],
                                    'payment_method_title' => $wooOrder['payment_method_title'],
                                    'billing' => $wooOrder['billing'],
                                    'shipping' => $wooOrder['shipping'],
                                    'line_items' => $wooOrder['line_items'],
                                    'meta_data' => $wooOrder['meta_data'],
                                    'notes' => $wooOrder['customer_note'] ?? null,
                                    'date_created' => $wooOrder['date_created'],
                                    'date_modified' => $wooOrder['date_modified'],
                                ]
                            );
                            $monthSynced++;
                            $totalSynced++;
                        } catch (\Exception $e) {
                            $this->error("  ✗ Failed to sync order {$wooOrder['id']}: " . $e->getMessage());
                        }
                    }

                    $page++;

                } while (count($orders) === $perPage);
            }

            if ($monthSynced > 0) {
                $this->info("  ✓ {$monthName}: {$monthSynced} orders synced");
            } else {
                $this->line("  - {$monthName}: No orders");
            }

            // Show progress
            $elapsed = time() - $startTime;
            $this->line("  ⏱️  Elapsed: " . gmdate('i:s', $elapsed) . " | Total: {$totalSynced} orders\n");
        }

        // Update last sync
        $store->update(['last_sync' => now()]);

        $totalTime = time() - $startTime;
        $this->newLine();
        $this->info("✅ Sync completed!");
        $this->info("📊 Total synced: {$totalSynced} orders");
        $this->info("⏱️  Total time: " . gmdate('i:s', $totalTime));

        return 0;
    }
}
