<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;

class OptimizePerformance extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'app:optimize-performance {--production : Run production optimizations}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Optimize application performance for speed (MySQL/MariaDB compatible)';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $isProduction = $this->option('production');

        $this->info('🚀 Starting Performance Optimization...');
        $this->newLine();

        // 1. Clear all caches first
        $this->task('Clearing existing caches', function () {
            Artisan::call('cache:clear');
            Artisan::call('config:clear');
            Artisan::call('route:clear');
            Artisan::call('view:clear');
            return true;
        });

        // 2. Run migrations (including performance indexes)
        $this->task('Running database migrations & indexes', function () {
            Artisan::call('migrate', ['--force' => true]);
            return true;
        });

        // 3. Optimize database (MySQL/MariaDB)
        $this->task('Optimizing database tables', function () {
            return $this->optimizeDatabase();
        });

        // 4. Cache configurations (production only)
        if ($isProduction) {
            $this->task('Caching configuration', function () {
                Artisan::call('config:cache');
                return true;
            });

            $this->task('Caching routes', function () {
                Artisan::call('route:cache');
                return true;
            });

            $this->task('Caching views', function () {
                Artisan::call('view:cache');
                return true;
            });
        }

        // 5. Optimize Composer autoloader
        $this->task('Optimizing Composer autoloader', function () use ($isProduction) {
            $command = $isProduction ? 'composer dump-autoload -o' : 'composer dump-autoload';
            exec($command, $output, $returnCode);
            return $returnCode === 0;
        });

        // 6. Show database info
        $this->newLine();
        $this->showDatabaseInfo();

        // 7. Performance recommendations
        $this->newLine();
        $this->showPerformanceRecommendations($isProduction);

        $this->newLine();
        $this->info('✅ Performance optimization complete!');
    }

    /**
     * Optimize database tables (MySQL/MariaDB compatible)
     */
    protected function optimizeDatabase(): bool
    {
        try {
            $tables = DB::select('SHOW TABLES');
            $databaseName = DB::getDatabaseName();
            $tableKey = "Tables_in_{$databaseName}";

            foreach ($tables as $table) {
                $tableName = $table->$tableKey;
                DB::statement("OPTIMIZE TABLE {$tableName}");
            }

            return true;
        } catch (\Exception $e) {
            $this->error("Database optimization failed: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Show database information
     */
    protected function showDatabaseInfo(): void
    {
        try {
            $version = DB::select('SELECT VERSION() as version')[0]->version;
            $isMariaDB = stripos($version, 'MariaDB') !== false;

            $this->info('📊 Database Information:');
            $this->line("   Database: " . DB::getDatabaseName());
            $this->line("   Engine: " . ($isMariaDB ? 'MariaDB' : 'MySQL'));
            $this->line("   Version: " . $version);

            // Count indexes
            $tables = DB::select('SHOW TABLES');
            $databaseName = DB::getDatabaseName();
            $tableKey = "Tables_in_{$databaseName}";
            $totalIndexes = 0;

            foreach ($tables as $table) {
                $tableName = $table->$tableKey;
                $indexes = DB::select("SHOW INDEX FROM {$tableName}");
                $totalIndexes += count($indexes);
            }

            $this->line("   Total Indexes: {$totalIndexes}");
        } catch (\Exception $e) {
            $this->error("   Unable to retrieve database info");
        }
    }

    /**
     * Show performance recommendations
     */
    protected function showPerformanceRecommendations(bool $isProduction): void
    {
        $this->info('💡 Performance Recommendations:');

        if (!$isProduction) {
            $this->warn('   ⚠️  Run with --production flag for production optimizations');
        }

        $this->line('   1. Enable OPcache in php.ini for production');
        $this->line('   2. Use Redis/Memcached for cache and sessions');
        $this->line('   3. Enable Gzip compression in web server');
        $this->line('   4. Use CDN for static assets');
        $this->line('   5. Monitor slow queries with EXPLAIN');

        // Check if important extensions are loaded
        $this->newLine();
        $this->line('   PHP Extensions:');
        $this->line('      OPcache: ' . (extension_loaded('Zend OPcache') ? '✅' : '❌'));
        $this->line('      Redis: ' . (extension_loaded('redis') ? '✅' : '❌'));
        $this->line('      APCu: ' . (extension_loaded('apcu') ? '✅' : '❌'));
    }
}
