<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Order;

class MigrateProcessingOrdersToApproval extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'orders:migrate-to-approval {--dry-run : Preview changes without updating}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Migrate existing orders with AWB from Processing to Approval status';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🔍 Scanning for orders with AWB still in Processing status...');
        $this->newLine();

        // Find orders that have tracking number but are still in processing status
        $orders = Order::where('status', 'processing')
            ->whereNotNull('tracking_number')
            ->get();

        $count = $orders->count();

        if ($count === 0) {
            $this->info('✅ No orders need migration. All orders with AWB are already in correct status.');
            return Command::SUCCESS;
        }

        $this->warn("Found {$count} orders with AWB still in Processing status:");
        $this->newLine();

        // Show preview
        $table = [];
        foreach ($orders as $order) {
            $table[] = [
                'Order #' => $order->order_number,
                'Tracking' => $order->tracking_number,
                'AWB Date' => $order->awb_generated_at ? $order->awb_generated_at->format('Y-m-d H:i') : 'N/A',
                'Current Status' => $order->status,
                'New Status' => 'approval',
            ];
        }

        $this->table(
            ['Order #', 'Tracking', 'AWB Date', 'Current Status', 'New Status'],
            $table
        );

        // Dry run mode
        if ($this->option('dry-run')) {
            $this->newLine();
            $this->info('🔍 DRY RUN MODE - No changes were made.');
            $this->info('Run without --dry-run to apply changes.');
            return Command::SUCCESS;
        }

        // Confirm before proceeding
        $this->newLine();
        if (!$this->confirm("Update {$count} orders from 'processing' to 'approval' status?", true)) {
            $this->warn('❌ Migration cancelled.');
            return Command::SUCCESS;
        }

        // Update orders
        $this->newLine();
        $progressBar = $this->output->createProgressBar($count);
        $progressBar->start();

        $updated = 0;
        foreach ($orders as $order) {
            $order->update(['status' => 'approval']);
            $updated++;
            $progressBar->advance();
        }

        $progressBar->finish();
        $this->newLine(2);

        $this->info("✅ Successfully migrated {$updated} orders to Approval status!");
        $this->newLine();
        $this->info('📊 Summary:');
        $this->line("   • Orders updated: {$updated}");
        $this->line("   • These orders now appear in the Approval tab");
        $this->line("   • Processing tab now only shows orders without AWB");

        return Command::SUCCESS;
    }
}
