<?php

namespace App\Console\Commands;

use App\Models\Order;
use App\Services\OrderNumberService;
use Illuminate\Console\Command;

class InitializeOrderSequences extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'orders:initialize-sequences';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Initialize order_sequences table based on existing orders';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info("=== Initializing Order Sequences ===\n");

        $orderNumberService = app(OrderNumberService::class);

        // Get all order numbers from database
        $orders = Order::select('order_number')->whereNotNull('order_number')->get();

        $this->info("Found " . $orders->count() . " orders in database\n");

        // Group by prefix and find max sequence
        $prefixGroups = [];

        foreach ($orders as $order) {
            $orderNumber = $order->order_number;

            // Extract prefix and sequence from order number
            // Patterns: CK-XXX-000123, SP-XXX-000456, WH-000789
            if (preg_match('/^((?:CK|SP)-[A-Z]{3}|WH)-(\d+)$/', $orderNumber, $matches)) {
                $prefix = $matches[1];
                $sequence = (int) $matches[2];

                if (!isset($prefixGroups[$prefix]) || $sequence > $prefixGroups[$prefix]) {
                    $prefixGroups[$prefix] = $sequence;
                }
            }
        }

        $this->info("Found " . count($prefixGroups) . " unique order prefixes:\n");

        // Initialize sequences
        foreach ($prefixGroups as $prefix => $maxSequence) {
            // Set sequence to max + 1 (ready for next order)
            $orderNumberService->initializeSequence($prefix, $maxSequence);

            $this->info("  ✓ {$prefix}: {$maxSequence} (next will be " . ($maxSequence + 1) . ")");
        }

        $this->info("\n✓ Order sequences initialized successfully!");
        $this->info("  New orders will continue from the correct sequence number.\n");

        return 0;
    }
}
