<?php

namespace App\Console\Commands;

use App\Models\Store;
use App\Services\WooCommerceService;
use Illuminate\Console\Command;

class EnableWebhooks extends Command
{
    protected $signature = 'woo:enable-webhooks {--store= : Specific store ID} {--all : Enable for all stores}';

    protected $description = 'Enable real-time webhooks for WooCommerce stores';

    public function handle()
    {
        $this->info('Setting up real-time webhooks...');
        $this->newLine();

        // Get stores
        if ($this->option('store')) {
            $stores = Store::where('id', $this->option('store'))
                          ->where('is_active', true)
                          ->get();
        } elseif ($this->option('all')) {
            $stores = Store::where('is_active', true)->get();
        } else {
            $stores = $this->selectStores();
        }

        if ($stores->isEmpty()) {
            $this->warn('No stores selected or found.');
            return 0;
        }

        foreach ($stores as $store) {
            $this->info("Setting up webhooks for: {$store->name}");

            try {
                $service = new WooCommerceService($store);

                // Generate webhook secret
                $secret = bin2hex(random_bytes(32));

                // Get webhook URL
                $webhookUrl = route('webhook.handle', ['store' => $store->id]);

                $this->line("  Webhook URL: {$webhookUrl}");

                // Create webhooks on WooCommerce
                $webhookIds = $service->createWebhooks($webhookUrl, $secret);

                // Save webhook config
                $store->update([
                    'webhooks_enabled' => true,
                    'webhook_secret' => $secret,
                    'webhook_ids' => $webhookIds,
                ]);

                $this->info("  ✓ Webhooks enabled successfully!");
                $this->line("  Created " . count($webhookIds) . " webhooks");

            } catch (\Exception $e) {
                $this->error("  ✗ Failed: " . $e->getMessage());
            }

            $this->newLine();
        }

        $this->info('Webhook setup completed!');
        $this->info('Orders will now sync in REAL-TIME when created or updated.');

        return 0;
    }

    protected function selectStores()
    {
        $stores = Store::where('is_active', true)->get();

        if ($stores->isEmpty()) {
            return collect();
        }

        $this->table(
            ['ID', 'Name', 'URL', 'Webhooks'],
            $stores->map(fn($s) => [
                $s->id,
                $s->name,
                $s->url,
                $s->webhooks_enabled ? '✓ Enabled' : '✗ Disabled'
            ])
        );

        $storeIds = $this->ask('Enter store IDs to enable webhooks (comma-separated, or "all")');

        if (strtolower($storeIds) === 'all') {
            return $stores;
        }

        $ids = array_map('trim', explode(',', $storeIds));
        return $stores->whereIn('id', $ids);
    }
}
