<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Store;
use App\Services\WooCommerceService;

class DiagnoseStores extends Command
{
    protected $signature = 'stores:diagnose';
    protected $description = 'Diagnose WooCommerce API connectivity for all stores';

    public function handle()
    {
        $stores = Store::where('is_active', true)->get();

        $this->info("Testing API connectivity for " . $stores->count() . " active stores...\n");

        $results = [];

        foreach ($stores as $store) {
            $this->line("Testing: {$store->name} (ID: {$store->id})");

            try {
                $service = new WooCommerceService($store);

                // Test 1: Fetch just 1 order
                $orders = $service->getOrders(['per_page' => 1, 'page' => 1]);

                if (!empty($orders)) {
                    $this->info("  ✓ API Connection: SUCCESS");

                    // Test 2: Count total orders
                    $completedCount = $service->getOrders(['status' => 'completed', 'per_page' => 1]);
                    $processingCount = $service->getOrders(['status' => 'processing', 'per_page' => 1]);

                    $this->info("  ✓ Can fetch orders");

                    $results[] = [
                        'id' => $store->id,
                        'name' => $store->name,
                        'status' => 'OK',
                        'url' => $store->url,
                    ];
                } else {
                    $this->error("  ✗ API Connection: NO ORDERS RETURNED");
                    $this->warn("    Check: WooCommerce REST API enabled?");

                    $results[] = [
                        'id' => $store->id,
                        'name' => $store->name,
                        'status' => 'NO ORDERS',
                        'url' => $store->url,
                    ];
                }

            } catch (\Exception $e) {
                $this->error("  ✗ API Connection: FAILED");
                $this->error("    Error: " . $e->getMessage());

                $results[] = [
                    'id' => $store->id,
                    'name' => $store->name,
                    'status' => 'ERROR',
                    'url' => $store->url,
                    'error' => $e->getMessage(),
                ];
            }

            $this->newLine();
        }

        // Summary table
        $this->newLine();
        $this->info("=== SUMMARY ===");
        $this->table(
            ['ID', 'Store Name', 'Status', 'URL'],
            collect($results)->map(function($r) {
                return [
                    $r['id'],
                    $r['name'],
                    $r['status'],
                    $r['url']
                ];
            })
        );

        $okCount = collect($results)->where('status', 'OK')->count();
        $failedCount = count($results) - $okCount;

        $this->newLine();
        $this->info("✓ Working: {$okCount}");
        $this->error("✗ Failed: {$failedCount}");

        return 0;
    }
}
