<?php

namespace App\Console\Commands;

use App\Models\User;
use App\Models\Order;
use Illuminate\Console\Command;

class DiagnoseGoogleSheetsSync extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'googlesheets:diagnose';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Diagnose Google Sheets sync issues for all sellers';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🔍 Diagnosing Google Sheets Sync for All Sellers...');
        $this->newLine();

        // Get all sellers
        $sellers = User::where('role', 'seller')->get();

        if ($sellers->isEmpty()) {
            $this->error('No sellers found!');
            return 1;
        }

        $this->info('Found ' . $sellers->count() . ' seller(s)');
        $this->newLine();

        $tableData = [];

        foreach ($sellers as $seller) {
            // Check Google Sheets status
            $syncEnabled = $seller->google_sheets_sync_enabled ? '✓' : '✗';
            $hasSheetId = $seller->google_sheet_id ? '✓' : '✗';
            $hasAccessToken = $seller->google_access_token ? '✓' : '✗';
            $hasRefreshToken = $seller->google_refresh_token ? '✓' : '✗';

            // Check orders created by this seller
            $totalOrders = Order::where('created_by', $seller->id)->count();
            $recentOrders = Order::where('created_by', $seller->id)
                ->where('created_at', '>=', now()->subDays(7))
                ->count();

            // Token expiration
            $tokenStatus = 'N/A';
            if ($seller->google_token_expires_at) {
                if ($seller->google_token_expires_at->isPast()) {
                    $tokenStatus = '❌ EXPIRED';
                } else {
                    $tokenStatus = '✓ Valid (' . $seller->google_token_expires_at->diffForHumans() . ')';
                }
            }

            $tableData[] = [
                'id' => $seller->id,
                'name' => $seller->name,
                'email' => $seller->email,
                'google_email' => $seller->google_email ?? 'Not connected',
                'sync_enabled' => $syncEnabled,
                'has_sheet_id' => $hasSheetId,
                'sheet_id' => $seller->google_sheet_id ? substr($seller->google_sheet_id, 0, 20) . '...' : 'None',
                'has_access_token' => $hasAccessToken,
                'has_refresh_token' => $hasRefreshToken,
                'token_status' => $tokenStatus,
                'total_orders' => $totalOrders,
                'recent_orders' => $recentOrders,
            ];
        }

        // Display table
        $this->table([
            'ID',
            'Name',
            'Email',
            'Google Email',
            'Sync Enabled',
            'Has Sheet ID',
            'Sheet ID',
            'Access Token',
            'Refresh Token',
            'Token Status',
            'Total Orders',
            'Recent (7d)',
        ], $tableData);

        $this->newLine();

        // Summary
        $enabledCount = $sellers->where('google_sheets_sync_enabled', true)->count();
        $withSheets = $sellers->filter(fn($s) => $s->google_sheet_id)->count();

        $this->info('📊 Summary:');
        $this->line("  • Sellers with sync enabled: {$enabledCount}/{$sellers->count()}");
        $this->line("  • Sellers with sheet ID: {$withSheets}/{$sellers->count()}");

        // Check for potential issues
        $this->newLine();
        $this->info('⚠️  Potential Issues:');

        $issues = [];

        foreach ($sellers as $seller) {
            if ($seller->google_sheets_sync_enabled && !$seller->google_sheet_id) {
                $issues[] = "  • {$seller->name} (ID: {$seller->id}): Sync enabled but NO sheet ID";
            }

            if ($seller->google_sheet_id && !$seller->google_sheets_sync_enabled) {
                $issues[] = "  • {$seller->name} (ID: {$seller->id}): Has sheet ID but sync DISABLED";
            }

            if ($seller->google_sheets_sync_enabled && !$seller->google_access_token) {
                $issues[] = "  • {$seller->name} (ID: {$seller->id}): Sync enabled but NO access token";
            }

            if ($seller->google_token_expires_at && $seller->google_token_expires_at->isPast()) {
                $issues[] = "  • {$seller->name} (ID: {$seller->id}): Access token EXPIRED";
            }
        }

        if (empty($issues)) {
            $this->info('  ✓ No obvious issues detected');
        } else {
            foreach ($issues as $issue) {
                $this->warn($issue);
            }
        }

        $this->newLine();

        // Check last sync attempts (from logs if available)
        $this->info('💡 Next Steps:');
        $this->line('  1. Check logs: tail -f storage/logs/laravel.log | grep "Google Sheets"');
        $this->line('  2. Check queue: php artisan queue:work --once');
        $this->line('  3. Create test order as each seller and monitor logs');

        return 0;
    }
}
