<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\User;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;

class CreateAdminUser extends Command
{
    protected $signature = 'make:admin 
                            {--name= : Admin name}
                            {--email= : Admin email}
                            {--password= : Admin password}';

    protected $description = 'Create a new admin user';

    public function handle()
    {
        $name = $this->option('name') ?? $this->ask('What is the admin name?', 'Admin');
        $email = $this->option('email') ?? $this->ask('What is the admin email?');
        $password = $this->option('password') ?? $this->secret('What is the admin password?');

        // Validate email
        $validator = Validator::make(['email' => $email], [
            'email' => 'required|email|unique:users,email'
        ]);

        if ($validator->fails()) {
            $this->error('Invalid email or email already exists!');
            return 1;
        }

        // Validate password
        if (strlen($password) < 6) {
            $this->error('Password must be at least 6 characters!');
            return 1;
        }

        // Create user
        $user = User::create([
            'name' => $name,
            'email' => $email,
            'password' => Hash::make($password),
            'email_verified_at' => now(),
        ]);

        $this->info('Admin user created successfully!');
        $this->table(
            ['Name', 'Email'],
            [[$user->name, $user->email]]
        );

        return 0;
    }
}