<?php

namespace App\Console\Commands;

use App\Models\OrderLog;
use Illuminate\Console\Command;
use Carbon\Carbon;

class CleanupOldOrderLogs extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'orders:cleanup-logs
                            {--days=180 : Delete logs older than X days}
                            {--keep=10 : Keep at least X logs per order}
                            {--dry-run : Preview what would be deleted without actually deleting}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Cleanup old order logs to save database space';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $days = (int) $this->option('days');
        $keepPerOrder = (int) $this->option('keep');
        $dryRun = $this->option('dry-run');

        $cutoffDate = Carbon::now()->subDays($days);

        $this->info("🧹 Order Logs Cleanup");
        $this->info("━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━");
        $this->info("Delete logs older than: {$cutoffDate->format('Y-m-d H:i:s')} ({$days} days ago)");
        $this->info("Keep per order: Latest {$keepPerOrder} logs");
        if ($dryRun) {
            $this->warn("🔍 DRY RUN MODE - No changes will be made");
        }
        $this->newLine();

        // Get orders with logs older than cutoff date
        $totalBefore = OrderLog::count();
        $this->info("📊 Current total logs: " . number_format($totalBefore));

        // Find logs to delete (older than cutoff AND not in the latest X per order)
        $toDelete = OrderLog::where('created_at', '<', $cutoffDate)
            ->whereNotIn('id', function($query) use ($keepPerOrder) {
                $query->select('id')
                    ->from('order_logs as ol')
                    ->whereRaw('ol.order_id = order_logs.order_id')
                    ->orderBy('created_at', 'desc')
                    ->limit($keepPerOrder);
            });

        $deleteCount = $toDelete->count();

        if ($deleteCount === 0) {
            $this->info("✅ No logs to cleanup!");
            return Command::SUCCESS;
        }

        $this->warn("⚠️  Found {$deleteCount} logs to delete");

        if ($dryRun) {
            $this->table(
                ['Order ID', 'Action', 'Created At', 'Age (days)'],
                $toDelete->take(10)->get()->map(function($log) {
                    return [
                        $log->order_id,
                        $log->action,
                        $log->created_at->format('Y-m-d H:i:s'),
                        $log->created_at->diffInDays(now())
                    ];
                })
            );

            if ($deleteCount > 10) {
                $this->info("... and " . ($deleteCount - 10) . " more");
            }

            $this->newLine();
            $this->info("💡 Run without --dry-run to actually delete these logs");
            return Command::SUCCESS;
        }

        // Confirm deletion
        if (!$this->confirm("Are you sure you want to delete {$deleteCount} logs?", false)) {
            $this->info("❌ Cleanup cancelled");
            return Command::SUCCESS;
        }

        // Delete the logs
        $deleted = $toDelete->delete();

        $totalAfter = OrderLog::count();
        $savedSpace = $this->estimateSavedSpace($deleted);

        $this->newLine();
        $this->info("✅ Cleanup completed!");
        $this->info("━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━");
        $this->info("Deleted: " . number_format($deleted) . " logs");
        $this->info("Remaining: " . number_format($totalAfter) . " logs");
        $this->info("Estimated space saved: ~{$savedSpace}");

        return Command::SUCCESS;
    }

    /**
     * Estimate space saved by deletion
     */
    private function estimateSavedSpace($count)
    {
        // Estimate ~0.8 KB per log on average
        $kb = $count * 0.8;

        if ($kb > 1024) {
            return number_format($kb / 1024, 2) . ' MB';
        }

        return number_format($kb, 2) . ' KB';
    }
}
