#!/bin/bash
# Clean Prophet Installation for mamadil.team
# Run: bash install_prophet_clean.sh

set -e

APP_PATH="/var/www/mamadil.team"
VENV_PATH="$APP_PATH/venv"

# Colors
GREEN='\033[0;32m'
RED='\033[0;31m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m'

print_success() {
    echo -e "${GREEN}✓ $1${NC}"
}

print_error() {
    echo -e "${RED}✗ $1${NC}"
}

print_info() {
    echo -e "${YELLOW}➜ $1${NC}"
}

print_step() {
    echo -e "${BLUE}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
    echo -e "${BLUE}$1${NC}"
    echo -e "${BLUE}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
}

echo ""
print_step "🤖 Prophet Installation for mamadil.team"
echo ""

# Check if venv exists
if [ ! -d "$VENV_PATH" ]; then
    print_error "Virtual environment not found at $VENV_PATH"
    print_info "Creating virtual environment..."
    python3.9 -m venv $VENV_PATH
    print_success "Virtual environment created"
fi

print_step "Fix: Remove pip --user Config"
print_info "Removing any pip config that causes '--user' error..."

# Remove pip.conf files that cause --user install error
rm -f ~/.pip/pip.conf 2>/dev/null || true
rm -f ~/.config/pip/pip.conf 2>/dev/null || true
rm -f $HOME/.pip/pip.conf 2>/dev/null || true
rm -f $HOME/.config/pip/pip.conf 2>/dev/null || true

# Verify no user config exists
if pip config list 2>/dev/null | grep -q "user.*true"; then
    print_info "Unsetting pip user config..."
    pip config unset global.user 2>/dev/null || true
fi

print_success "pip config cleaned (no --user flag will be used)"

print_info "Activating virtual environment..."
source $VENV_PATH/bin/activate
print_success "Virtual environment activated"

print_step "Step 1: Clean Installation"
print_info "Removing any existing Prophet installations..."
pip uninstall prophet fbprophet pystan cmdstanpy -y 2>/dev/null || true
print_success "Cleaned up old installations"

print_step "Step 2: Update pip"
print_info "Upgrading pip, setuptools, and wheel..."
pip install --upgrade pip setuptools wheel
print_success "pip upgraded"

print_step "Step 3: Install Dependencies"
print_info "Installing numpy and pandas..."
pip install numpy==1.24.3
pip install pandas==2.0.3
print_success "Dependencies installed"

print_step "Step 4: Install cmdstanpy (Prophet Backend)"
print_info "Installing cmdstanpy..."
pip install cmdstanpy==1.2.0
print_success "cmdstanpy installed"

print_step "Step 5: Install CmdStan Compiler"
print_info "This may take 5-10 minutes. Please wait..."
python3.9 << 'PYEOF'
import cmdstanpy
import sys

print("Installing CmdStan compiler...")
try:
    cmdstanpy.install_cmdstan(version='2.32.2', verbose=True, overwrite=True)
    print("\n✓ CmdStan installed successfully!")
except Exception as e:
    print(f"\n✗ Error installing CmdStan: {e}")
    sys.exit(1)
PYEOF

if [ $? -eq 0 ]; then
    print_success "CmdStan compiler installed"
else
    print_error "CmdStan installation failed"
    print_info "Trying alternative method..."
    python3.9 -m cmdstanpy.install_cmdstan --version 2.32.2 --verbose
fi

print_step "Step 6: Install Prophet"
print_info "Installing Prophet..."
pip install prophet==1.1.5
print_success "Prophet installed"

print_step "Step 7: Verify Installation"
print_info "Testing Prophet import..."

python3.9 << 'PYEOF'
import sys

try:
    from prophet import Prophet
    import pandas as pd
    import numpy as np

    print("\n✓ Prophet imported successfully!")

    # Try creating a Prophet model
    m = Prophet()
    print("✓ Prophet model created successfully!")

    # Show versions
    import prophet
    print(f"✓ Prophet version: {prophet.__version__}")
    print(f"✓ Pandas version: {pd.__version__}")
    print(f"✓ NumPy version: {np.__version__}")

except ImportError as e:
    print(f"\n✗ Import Error: {e}")
    sys.exit(1)
except Exception as e:
    print(f"\n✗ Error: {e}")
    sys.exit(1)
PYEOF

if [ $? -eq 0 ]; then
    print_success "Prophet verification successful!"
else
    print_error "Prophet verification failed"
    exit 1
fi

print_step "Step 8: Test with Sample Data"
print_info "Testing Prophet with sample forecast..."

python3.9 << 'PYEOF'
from prophet import Prophet
import pandas as pd
from datetime import datetime

# Create sample data
dates = pd.date_range(start='2023-01-01', end='2023-12-01', freq='MS')
data = pd.DataFrame({
    'ds': dates,
    'y': [50000, 55000, 60000, 58000, 62000, 65000, 68000, 70000, 72000, 75000, 78000, 80000]
})

print("\nSample data:")
print(data.head())

# Create and fit model
print("\nFitting Prophet model...")
m = Prophet()
m.fit(data)

# Make forecast
print("Making 6-month forecast...")
future = m.make_future_dataframe(periods=6, freq='MS')
forecast = m.predict(future)

print("\nForecast for next 6 months:")
print(forecast[['ds', 'yhat', 'yhat_lower', 'yhat_upper']].tail(6))
print("\n✓ Prophet forecasting works!")
PYEOF

if [ $? -eq 0 ]; then
    print_success "Prophet test successful!"
else
    print_error "Prophet test failed"
    exit 1
fi

print_step "Step 9: Test forecast_prophet.py Script"
print_info "Testing your forecast script..."

if [ -f "$APP_PATH/forecast_prophet.py" ]; then
    TEST_DATA='[{"year":2023,"month":1,"revenue":50000},{"year":2023,"month":2,"revenue":55000},{"year":2023,"month":3,"revenue":60000},{"year":2023,"month":4,"revenue":58000},{"year":2023,"month":5,"revenue":62000},{"year":2023,"month":6,"revenue":65000},{"year":2023,"month":7,"revenue":68000},{"year":2023,"month":8,"revenue":70000},{"year":2023,"month":9,"revenue":72000},{"year":2023,"month":10,"revenue":75000},{"year":2023,"month":11,"revenue":78000},{"year":2023,"month":12,"revenue":80000}]'

    echo "$TEST_DATA" | $VENV_PATH/bin/python3.9 $APP_PATH/forecast_prophet.py 6 > /tmp/prophet_output.json 2>&1

    if [ $? -eq 0 ]; then
        print_success "forecast_prophet.py works!"
        echo ""
        echo "Output preview:"
        head -30 /tmp/prophet_output.json
        rm /tmp/prophet_output.json
    else
        print_error "forecast_prophet.py failed"
        echo ""
        echo "Error output:"
        cat /tmp/prophet_output.json
        rm /tmp/prophet_output.json
        exit 1
    fi
else
    print_info "forecast_prophet.py not found (upload it first)"
fi

print_step "Step 10: Show Installed Packages"
print_info "Installed Prophet-related packages:"
pip list | grep -E "(prophet|cmdstan|numpy|pandas)"

deactivate

echo ""
print_step "✅ Installation Complete!"
echo ""
echo "Installed packages:"
echo "  - Prophet: 1.1.5"
echo "  - cmdstanpy: 1.2.0"
echo "  - numpy: 1.24.3"
echo "  - pandas: 2.0.3"
echo ""
echo "Python path: $VENV_PATH/bin/python3.9"
echo ""
echo "Next steps:"
echo "  1. Make sure .env has: PYTHON_PATH=$VENV_PATH/bin/python3.9"
echo "  2. Clear cache: php artisan config:clear"
echo "  3. Test web interface: https://mamadil.team/forecasting"
echo ""
