'use client'

import { clsx } from 'clsx'
import { motion } from 'framer-motion'
import { createContext, useContext } from 'react'

const KeyboardContext = createContext<{ highlighted: string[] }>({
  highlighted: [],
})

function Row(props: { children: React.ReactNode }) {
  return <div {...props} className="group flex h-9 gap-2" />
}

function Key({
  name,
  width = 36,
  className,
  children,
}: {
  name: string
  width?: number
  className?: string
  children?: React.ReactNode
}) {
  let { highlighted } = useContext(KeyboardContext)

  return (
    <motion.div
      variants={
        highlighted.includes(name)
          ? {
              idle: {
                scale: 1,
                boxShadow: `rgb(255, 255, 255) 0px 0px 0px 0px, rgba(47, 127, 221, 0.8) 0px 0px 0px 1px, rgba(47, 127, 221, 0.15) 0px 0px 3px 3px,rgba(47, 127, 221, 0.1) 0px 0px 0px 0px`,
              },
              active: {
                scale: [1, 1.02, 1.02, 1],
                boxShadow: [
                  `rgb(255, 255, 255) 0px 0px 0px 0px, rgba(47, 127, 221, 0.8) 0px 0px 0px 1px, rgba(47, 127, 221, 0.15) 0px 0px 3px 3px, rgba(47, 127, 221, 0.05) 0px 0px 0px 0px`,
                  `rgb(255, 255, 255) 0px 0px 0px 0px, rgba(47, 127, 221, 0.8) 0px 0px 0px 1px, rgba(47, 127, 221, 0.1) 0px 0px 1px 6px, rgba(47, 127, 221, 0.05) 0px 0px 1px 15px`,
                  `rgb(255, 255, 255) 0px 0px 0px 0px, rgba(47, 127, 221, 0.8) 0px 0px 0px 1px, rgba(47, 127, 221, 0.0) 0px 0px 1px 6px, rgba(47, 127, 221, 0.01) 0px 0px 1px 15px`,
                  `rgb(255, 255, 255) 0px 0px 0px 0px, rgba(47, 127, 221, 0.8) 0px 0px 0px 1px, rgba(47, 127, 221, 0.0) 0px 0px 3px 3px, rgba(47, 127, 221, 0.00) 0px 0px 0px 0px`,
                ],
                transition: {
                  repeat: Infinity,
                  ease: 'easeInOut',
                  duration: 1.75 - highlighted.indexOf(name) * 0.35,
                  delay: highlighted.indexOf(name) * 0.35,
                  repeatDelay: 1 + highlighted.indexOf(name) * 0.35,
                },
              },
            }
          : undefined
      }
      style={{ width: `${width / 16}rem` }}
      className={clsx(
        className,
        'flex flex-col items-center justify-center gap-0.5 px-1 py-px',
        'rounded-sm bg-white bg-linear-to-t from-black/[3%] ring-1 shadow-[0_1px_rgb(0_0_0_/_0.05)] ring-black/10',
        '[:where(&_svg)]:h-3.5 [:where(&_svg)]:fill-gray-600',
      )}
    >
      {children}
    </motion.div>
  )
}

function KeyGroup(props: { children: React.ReactNode }) {
  return (
    <div
      {...props}
      className="grid gap-px rounded-sm bg-black/10 ring-1 ring-black/10 *:ring-0"
    />
  )
}

function EscapeKey() {
  return (
    <Key name="Escape" width={62} className="rounded-tl-xl">
      <svg viewBox="0 0 13 14" className="mt-auto mr-auto">
        <path d="M8.95 8.273v-.671c0-.36.068-.672.206-.938.14-.268.339-.475.594-.621.258-.146.563-.219.914-.219.344 0 .632.063.863.188.232.122.408.28.528.472.12.19.182.388.187.594h-.687a.887.887 0 0 0-.11-.27.668.668 0 0 0-.265-.257c-.123-.073-.29-.11-.504-.11-.334 0-.588.107-.762.32-.172.214-.258.502-.258.864v.656c0 .354.09.638.27.852.182.21.432.316.75.316.203 0 .366-.027.488-.082a.683.683 0 0 0 .285-.215.803.803 0 0 0 .133-.293h.688c-.008.2-.07.395-.188.582a1.283 1.283 0 0 1-.527.454c-.232.114-.529.171-.89.171-.352 0-.657-.073-.915-.218a1.501 1.501 0 0 1-.594-.621 2.058 2.058 0 0 1-.207-.954ZM4.883 7.07c0-.291.071-.528.215-.71.145-.186.334-.32.566-.407.234-.086.484-.129.75-.129.297 0 .555.048.774.145.22.096.393.23.515.402a1 1 0 0 1 .184.602H7.23a.576.576 0 0 0-.105-.282.67.67 0 0 0-.277-.222 1.02 1.02 0 0 0-.457-.09c-.24 0-.438.053-.594.16a.527.527 0 0 0-.23.461.45.45 0 0 0 .093.297.677.677 0 0 0 .242.176c.102.041.213.074.332.097.12.024.237.046.352.067.258.044.492.105.703.183.214.076.384.19.512.34.127.149.191.357.191.625 0 .274-.069.507-.207.7-.135.19-.327.334-.574.433a2.284 2.284 0 0 1-.86.148c-.304 0-.572-.044-.804-.132a1.21 1.21 0 0 1-.543-.387 1.005 1.005 0 0 1-.195-.625H5.5c.018.122.06.228.125.316a.63.63 0 0 0 .285.207c.125.047.283.07.473.07.216 0 .39-.03.523-.09a.657.657 0 0 0 .297-.234.583.583 0 0 0 .094-.324c0-.153-.046-.27-.137-.351a.902.902 0 0 0-.37-.184 6.194 6.194 0 0 0-.52-.121 5.53 5.53 0 0 1-.7-.188 1.165 1.165 0 0 1-.5-.332c-.125-.145-.187-.352-.187-.62ZM2.254 10.066c-.367 0-.682-.073-.945-.218a1.5 1.5 0 0 1-.606-.63 2.092 2.092 0 0 1-.207-.96v-.625c0-.373.073-.693.219-.961.148-.271.351-.48.61-.625.257-.149.549-.223.874-.223.334 0 .628.074.883.223.258.148.46.358.606.629.145.27.218.59.218.957v.496H1.2v.207c0 .318.09.586.27.805.18.218.44.328.781.328.159 0 .3-.022.422-.067a.863.863 0 0 0 .3-.18.538.538 0 0 0 .153-.238h.727a.984.984 0 0 1-.133.352c-.07.125-.17.243-.301.355-.13.11-.292.2-.484.27-.193.07-.42.105-.68.105ZM1.199 7.617h2.004v-.066a1.33 1.33 0 0 0-.125-.594.937.937 0 0 0-.351-.398.96.96 0 0 0-.524-.141.98.98 0 0 0-.527.14.937.937 0 0 0-.352.399 1.33 1.33 0 0 0-.125.594v.066Z" />
      </svg>
    </Key>
  )
}

function F1Key() {
  return (
    <Key name="F1">
      <svg viewBox="0 0 10 14">
        <path d="M4.948 9.147c-.293 0-.57-.055-.83-.167a2.291 2.291 0 0 1-.686-.465 2.291 2.291 0 0 1-.466-.686A2.093 2.093 0 0 1 2.8 7c0-.293.056-.569.167-.827a2.22 2.22 0 0 1 1.152-1.151c.26-.114.537-.171.83-.171.293 0 .569.057.826.171.261.111.49.267.686.466.2.196.354.425.466.685.114.258.171.534.171.827 0 .293-.057.57-.171.83A2.22 2.22 0 0 1 5.775 8.98a2.061 2.061 0 0 1-.827.167Zm0-.707c.264 0 .504-.065.72-.193.22-.132.395-.308.524-.528.132-.22.198-.46.198-.72a1.36 1.36 0 0 0-.198-.721 1.444 1.444 0 0 0-.523-.523 1.36 1.36 0 0 0-.72-.198c-.261 0-.502.066-.721.198-.22.129-.396.303-.528.523-.129.22-.193.46-.193.72 0 .261.064.502.193.721.132.22.308.396.528.528.22.128.46.193.72.193Zm-3.7-.936a.499.499 0 0 1-.505-.505c0-.141.048-.26.145-.356.1-.1.22-.15.36-.15.138 0 .255.05.352.15.1.096.149.215.149.356 0 .14-.05.26-.15.36a.479.479 0 0 1-.35.145Zm1.081-2.62a.499.499 0 0 1-.505-.505c0-.14.048-.259.145-.356.1-.1.22-.149.36-.149.14 0 .26.05.356.15.1.096.15.215.15.355 0 .141-.05.261-.15.36a.484.484 0 0 1-.356.146ZM4.95 3.8a.499.499 0 0 1-.506-.5c0-.141.048-.26.145-.357.1-.1.22-.149.36-.149.14 0 .26.05.356.15.1.096.15.215.15.355 0 .138-.05.257-.15.356a.484.484 0 0 1-.356.145Zm2.618 1.086a.493.493 0 0 1-.356-.145c-.1-.1-.149-.22-.149-.36 0-.141.05-.26.15-.357.099-.1.218-.149.355-.149.141 0 .26.05.356.15.1.096.15.215.15.355 0 .141-.05.261-.15.36a.484.484 0 0 1-.356.146Zm1.081 2.619a.488.488 0 0 1-.351-.145c-.1-.1-.15-.22-.15-.36 0-.141.05-.26.15-.356.1-.1.217-.15.351-.15.141 0 .26.05.356.15.1.096.15.215.15.356 0 .14-.05.26-.15.36a.484.484 0 0 1-.356.145Zm-1.08 2.619a.493.493 0 0 1-.357-.145c-.1-.1-.149-.22-.149-.36 0-.141.05-.26.15-.356.099-.1.218-.15.355-.15.141 0 .26.05.356.15.1.096.15.215.15.356 0 .14-.05.26-.15.36a.484.484 0 0 1-.356.145Zm-2.62 1.081c-.14 0-.26-.05-.36-.15a.484.484 0 0 1-.145-.355.493.493 0 0 1 .505-.501c.14 0 .26.048.356.145.1.1.15.218.15.356 0 .14-.05.259-.15.356a.477.477 0 0 1-.356.15ZM2.33 10.123a.499.499 0 0 1-.505-.505c0-.141.048-.26.145-.356.1-.1.22-.15.36-.15.14 0 .26.05.356.15.1.096.15.215.15.356 0 .14-.05.26-.15.36a.484.484 0 0 1-.356.145Z" />
      </svg>
      <svg viewBox="0 0 7 14">
        <path d="M5.847 8.999h-.785V5.762h-.039l-.301.214-.305.216-.302.214V5.68l.475-.34.472-.34h.785v3.999ZM1.722 8.999H.93V5h2.548v.653H1.722V6.77h1.605v.638H1.722V9Z" />
      </svg>
    </Key>
  )
}

function F2Key() {
  return (
    <Key name="F2">
      <svg viewBox="0 0 11 14">
        <path d="M5.309 9.157c-.293 0-.57-.056-.831-.167a2.251 2.251 0 0 1-1.156-1.156 2.104 2.104 0 0 1-.167-.835c0-.296.056-.573.167-.83a2.235 2.235 0 0 1 1.156-1.156c.26-.115.538-.172.83-.172.294 0 .57.057.831.172.26.111.49.266.686.466.199.199.354.429.466.69.114.257.17.534.17.83 0 .296-.056.574-.17.835a2.188 2.188 0 0 1-.466.686 2.093 2.093 0 0 1-1.516.637Zm0-.712c.263 0 .505-.065.725-.194.22-.131.395-.307.527-.527.132-.22.198-.461.198-.725 0-.264-.066-.505-.198-.725a1.478 1.478 0 0 0-.527-.523 1.383 1.383 0 0 0-.725-.198c-.264 0-.506.066-.726.198a1.49 1.49 0 0 0-.522.523c-.132.22-.198.461-.198.725 0 .264.066.505.198.725.131.22.306.396.522.527.22.13.462.194.726.194Zm.395-5.63v.862c0 .102-.04.193-.119.272a.377.377 0 0 1-.276.114.382.382 0 0 1-.273-.114.382.382 0 0 1-.114-.272v-.862c0-.108.038-.2.114-.276a.375.375 0 0 1 .55 0 .37.37 0 0 1 .118.276ZM7.378 4.38l.616-.615a.37.37 0 0 1 .272-.11.39.39 0 0 1 .281.11.39.39 0 0 1 .11.28.37.37 0 0 1-.11.273l-.61.616a.401.401 0 0 1-.558 0 .385.385 0 0 1-.11-.277c0-.112.036-.204.11-.277Zm2.11 3.01H8.63a.387.387 0 0 1-.277-.114.387.387 0 0 1-.114-.277c0-.108.038-.2.114-.277a.38.38 0 0 1 .277-.118h.857c.105 0 .196.04.272.118a.38.38 0 0 1 0 .554.372.372 0 0 1-.272.114ZM7.932 9.07l.615.61c.074.077.11.17.11.278a.37.37 0 0 1-.11.272.377.377 0 0 1-.276.114.349.349 0 0 1-.273-.11l-.62-.61a.385.385 0 0 1-.11-.277.385.385 0 0 1 .387-.391c.109 0 .201.038.277.114ZM5.704 10.32v.862c0 .108-.04.2-.119.277a.375.375 0 0 1-.55 0 .377.377 0 0 1-.113-.277v-.862c0-.102.038-.193.114-.272a.382.382 0 0 1 .273-.114c.108 0 .2.038.276.114.08.079.12.17.12.272Zm-3.63-.646.615-.606a.36.36 0 0 1 .273-.11.39.39 0 0 1 .281.11.39.39 0 0 1 .11.281c0 .109-.038.2-.114.277l-.611.602a.368.368 0 0 1-.281.11.363.363 0 0 1-.273-.114.363.363 0 0 1-.114-.273.401.401 0 0 1 .114-.277Zm-.936-3.071h.853c.105 0 .196.04.272.118a.38.38 0 0 1 0 .554.372.372 0 0 1-.272.114h-.853a.392.392 0 0 1-.281-.668.384.384 0 0 1 .281-.118Zm1.547-1.67-.606-.616a.37.37 0 0 1-.11-.272.382.382 0 0 1 .386-.391c.112 0 .204.036.277.11l.611.615c.073.073.11.165.11.277a.368.368 0 0 1-.114.277.39.39 0 0 1-.281.11.37.37 0 0 1-.273-.11Z" />
      </svg>
      <svg viewBox="0 0 7 14">
        <path d="M4.522 6.194h-.75v-.038a1.174 1.174 0 0 1 .337-.844c.114-.117.255-.21.425-.28.172-.071.374-.106.607-.106.275 0 .515.049.72.146.205.098.364.232.475.402.113.168.17.358.17.57 0 .185-.035.349-.106.493a1.825 1.825 0 0 1-.263.401c-.106.124-.217.25-.334.378l-.908 1.002v.038h1.678v.645H3.811v-.492l1.461-1.576c.073-.077.143-.156.211-.238.069-.084.125-.174.17-.27a.715.715 0 0 0 .068-.307.534.534 0 0 0-.29-.486.634.634 0 0 0-.308-.073.608.608 0 0 0-.322.082.562.562 0 0 0-.208.22.618.618 0 0 0-.07.29v.043ZM1.396 8.999h-.79V5h2.548v.653H1.396V6.77h1.606v.638H1.396V9Z" />
      </svg>
    </Key>
  )
}

function F3Key() {
  return (
    <Key name="F3">
      <svg viewBox="0 0 14 14">
        <path d="M2.104 6.84c-.33 0-.58-.08-.747-.24-.167-.165-.25-.414-.25-.748V4.13c0-.33.083-.578.25-.742.167-.164.416-.246.747-.246h3.824c.33 0 .58.082.747.246.167.164.25.411.25.742v1.723c0 .334-.083.583-.25.747-.167.161-.416.242-.747.242H2.104Zm.075-.707h3.665c.123 0 .216-.03.277-.092.062-.062.092-.154.092-.277V4.226c0-.129-.03-.223-.092-.281-.061-.062-.154-.092-.277-.092H2.18c-.117 0-.206.03-.268.092-.061.058-.092.152-.092.281v1.538c0 .123.03.215.092.277.062.062.151.092.268.092Zm.976 4.702c-.328 0-.576-.082-.743-.246-.167-.164-.25-.411-.25-.742V8.673c0-.328.083-.572.25-.734.167-.164.415-.246.743-.246h3.26c.332 0 .58.082.743.246.167.162.25.406.25.734v1.174c0 .33-.083.578-.25.742-.164.164-.411.246-.742.246H3.155Zm.074-.707h3.103c.123 0 .215-.032.277-.097.061-.061.092-.154.092-.277V8.77c0-.123-.03-.215-.092-.277-.062-.064-.154-.097-.277-.097H3.229a.362.362 0 0 0-.268.097c-.061.062-.092.154-.092.277v.984c0 .123.03.216.092.277a.362.362 0 0 0 .268.097Zm5.977.193c-.331 0-.58-.082-.747-.246-.164-.16-.246-.407-.246-.738V4.723c0-.331.082-.577.246-.739.167-.164.416-.246.747-.246h2.69c.33 0 .58.082.747.246.167.162.25.408.25.739v4.614c0 .331-.083.577-.25.738-.167.164-.416.246-.748.246H9.207Zm.075-.707h2.535c.12 0 .211-.031.273-.093.061-.061.092-.153.092-.276v-4.43c0-.123-.03-.214-.092-.273-.062-.061-.152-.092-.273-.092H9.281c-.117 0-.207.031-.268.092-.062.06-.093.15-.093.273v4.43c0 .123.031.215.093.276.061.062.15.093.268.093Z" />
      </svg>
      <svg viewBox="0 0 7 14">
        <path d="M4.657 7.244v-.61h.422a.688.688 0 0 0 .322-.072.557.557 0 0 0 .22-.2.505.505 0 0 0 .076-.284.49.49 0 0 0-.176-.392.652.652 0 0 0-.442-.15.741.741 0 0 0-.252.041.627.627 0 0 0-.193.112.496.496 0 0 0-.179.349H3.72c.006-.157.04-.303.102-.437.063-.135.153-.252.27-.352.117-.101.26-.18.428-.237.17-.057.364-.086.583-.088.279-.002.52.042.723.132.203.09.36.214.472.372a.91.91 0 0 1 .173.539.833.833 0 0 1-.12.477.96.96 0 0 1-.619.44v.04a1.007 1.007 0 0 1 .718.434.909.909 0 0 1 .144.522c.002.19-.037.358-.117.507a1.104 1.104 0 0 1-.329.378c-.14.101-.302.18-.486.234-.182.053-.376.08-.583.08-.3 0-.558-.052-.77-.153a1.206 1.206 0 0 1-.487-.41 1.094 1.094 0 0 1-.178-.563h.726a.457.457 0 0 0 .106.258.664.664 0 0 0 .249.179.979.979 0 0 0 .357.067.903.903 0 0 0 .384-.076.598.598 0 0 0 .252-.217c.06-.094.09-.2.088-.32a.556.556 0 0 0-.334-.52.811.811 0 0 0-.372-.08h-.443ZM1.358 8.999h-.79V5h2.548v.653H1.358V6.77h1.606v.638H1.358V9Z" />
      </svg>
    </Key>
  )
}

function F4Key() {
  return (
    <Key name="F4">
      <svg viewBox="0 0 9 14">
        <path d="M.871 6.648a2.937 2.937 0 0 1 .871-2.105c.274-.273.59-.487.95-.64.359-.157.744-.235 1.156-.235.414 0 .8.078 1.16.234a2.994 2.994 0 0 1 1.59 1.59c.153.36.23.745.23 1.156a2.891 2.891 0 0 1-.582 1.754l1.746 1.75a.54.54 0 0 1 .137.348c0 .096-.02.181-.063.254a.472.472 0 0 1-.422.242.54.54 0 0 1-.355-.137L5.531 9.098c-.24.166-.501.297-.785.39-.284.094-.583.14-.898.14a2.91 2.91 0 0 1-1.157-.23 2.994 2.994 0 0 1-1.59-1.59 2.938 2.938 0 0 1-.23-1.16Zm.711 0a2.235 2.235 0 0 0 .664 1.606c.208.206.45.367.723.484.273.117.566.176.879.176a2.26 2.26 0 0 0 2.266-2.266c0-.312-.06-.605-.177-.878a2.228 2.228 0 0 0-.488-.723 2.228 2.228 0 0 0-1.601-.664c-.313 0-.606.058-.88.176a2.285 2.285 0 0 0-1.21 1.21 2.207 2.207 0 0 0-.176.88Z" />
      </svg>
      <svg viewBox="0 0 7 14">
        <path d="M5.117 5h1.14v2.62h.49v.638h-.49v.741h-.741v-.741H3.579V7.61c.234-.44.483-.876.747-1.31.266-.433.53-.866.791-1.3Zm-.82 2.62h1.219V5.597h-.041c-.202.318-.404.646-.607.984-.201.338-.391.677-.571 1.017v.02ZM1.303 8.999H.512V5H3.06v.653H1.303V6.77h1.605v.638H1.303V9Z" />
      </svg>
    </Key>
  )
}

function F5Key() {
  return (
    <Key name="F5">
      <svg viewBox="0 0 9 14">
        <path d="M1.428 6.637v-.83c0-.097.034-.18.101-.246a.347.347 0 0 1 .255-.106c.097 0 .179.035.246.106.07.067.106.149.106.246v.804c0 .472.098.885.294 1.24.2.354.476.629.83.825.355.194.768.29 1.24.29s.885-.096 1.24-.29c.354-.196.63-.471.825-.826.2-.354.3-.767.3-1.24v-.803c0-.097.033-.18.1-.246a.347.347 0 0 1 .255-.106c.097 0 .179.035.246.106.07.067.106.149.106.246v.83c0 .56-.116 1.055-.347 1.486a2.727 2.727 0 0 1-.958 1.032c-.408.261-.88.415-1.415.462v.905h1.533c.097 0 .179.034.246.101.07.07.106.155.106.255s-.035.183-.106.25a.335.335 0 0 1-.246.101H2.62a.355.355 0 0 1-.255-.606.355.355 0 0 1 .255-.101h1.53v-.905a3.107 3.107 0 0 1-1.42-.462 2.774 2.774 0 0 1-.958-1.032c-.229-.431-.343-.926-.343-1.486Zm1.516-.167V3.873c0-.331.066-.621.198-.87.135-.252.318-.448.55-.589.234-.14.503-.21.808-.21a1.46 1.46 0 0 1 1.354.8 1.8 1.8 0 0 1 .202.869V6.47c0 .325-.068.614-.202.866a1.46 1.46 0 0 1-.55.589c-.231.14-.5.21-.804.21a1.54 1.54 0 0 1-.809-.21 1.501 1.501 0 0 1-.549-.59 1.84 1.84 0 0 1-.198-.865Zm.717.005c0 .296.076.533.228.712a.768.768 0 0 0 .611.263.768.768 0 0 0 .61-.263c.156-.18.234-.416.234-.712v-2.61c0-.294-.078-.528-.233-.704a.762.762 0 0 0-.611-.268.762.762 0 0 0-.61.268c-.153.176-.23.41-.23.703v2.61Z" />
      </svg>
      <svg viewBox="0 0 7 14">
        <path d="M3.729 7.974h.726c.014.08.042.15.085.21a.59.59 0 0 0 .164.147.84.84 0 0 0 .454.12.72.72 0 0 0 .624-.354.795.795 0 0 0 .103-.413.71.71 0 0 0-.1-.381.655.655 0 0 0-.264-.252.757.757 0 0 0-.363-.088.752.752 0 0 0-.656.37h-.691l.167-2.333h2.449v.639H4.61l-.088 1.06h.041a.615.615 0 0 1 .144-.143.97.97 0 0 1 .267-.138c.111-.04.242-.059.392-.059.23 0 .443.051.636.153.195.1.352.246.469.44.119.193.178.428.178.705 0 .287-.063.537-.19.75-.127.211-.304.375-.53.492a1.674 1.674 0 0 1-.785.176 1.85 1.85 0 0 1-.569-.082 1.38 1.38 0 0 1-.442-.231 1.137 1.137 0 0 1-.293-.352 1.04 1.04 0 0 1-.111-.436ZM1.323 8.999h-.79V5H3.08v.653H1.323V6.77H2.93v.638H1.323V9Z" />
      </svg>
    </Key>
  )
}

function F6Key() {
  return (
    <Key name="F6">
      <svg viewBox="0 0 9 14">
        <path d="M6.57 8.559c.144 0 .285-.008.426-.024a4.38 4.38 0 0 0 .406-.07c.128-.029.24-.063.336-.102a.49.49 0 0 1 .121-.043.618.618 0 0 1 .102-.008.27.27 0 0 1 .184.079c.057.05.085.117.085.203a.567.567 0 0 1-.062.242c-.2.463-.474.862-.82 1.195-.347.334-.75.59-1.211.77a3.99 3.99 0 0 1-1.489.273c-.56 0-1.076-.096-1.55-.289a3.702 3.702 0 0 1-1.23-.812 3.7 3.7 0 0 1-.813-1.23A4.12 4.12 0 0 1 .77 7.186c0-.515.098-1.01.296-1.484.198-.474.473-.89.825-1.25.354-.362.76-.637 1.218-.824a.906.906 0 0 1 .141-.043.491.491 0 0 1 .086-.012c.086 0 .156.031.21.094a.298.298 0 0 1 .083.2.63.63 0 0 1-.07.258 2.1 2.1 0 0 0-.168.542c-.042.23-.063.482-.063.758 0 .643.132 1.199.395 1.668a2.7 2.7 0 0 0 1.12 1.086c.485.252 1.06.379 1.727.379ZM1.422 7.168c0 .471.08.906.242 1.305.162.395.387.74.676 1.03.292.293.635.52 1.031.68a3.43 3.43 0 0 0 1.3.243 3.47 3.47 0 0 0 1.083-.168c.346-.112.659-.274.937-.485.282-.21.513-.462.696-.753-.138.049-.284.084-.438.105a3.55 3.55 0 0 1-.484.031c-.781 0-1.452-.146-2.012-.437A3.04 3.04 0 0 1 3.168 7.46c-.3-.547-.45-1.203-.45-1.969 0-.195.015-.381.044-.558.028-.177.07-.347.125-.508-.3.198-.56.441-.782.73a3.366 3.366 0 0 0-.507.95 3.27 3.27 0 0 0-.176 1.062Z" />
      </svg>
      <svg viewBox="0 0 8 14">
        <path d="M7.08 5.926h-.754a.518.518 0 0 0-.102-.173.552.552 0 0 0-.2-.14.722.722 0 0 0-.298-.056.678.678 0 0 0-.524.249c-.07.082-.128.18-.177.296a1.74 1.74 0 0 0-.105.386c-.022.143-.03.297-.026.463h.046a.935.935 0 0 1 .498-.475c.131-.054.287-.081.466-.081.213 0 .415.047.607.143.191.096.347.238.468.428.122.19.182.426.182.709 0 .295-.064.547-.193.756-.127.209-.301.369-.522.48a1.68 1.68 0 0 1-.756.164c-.168 0-.33-.021-.486-.064a1.352 1.352 0 0 1-.43-.208 1.327 1.327 0 0 1-.343-.381 1.917 1.917 0 0 1-.22-.554 3.263 3.263 0 0 1-.08-.764c0-.293.024-.557.071-.792.049-.234.12-.437.211-.609.094-.174.206-.317.337-.43.13-.116.28-.202.445-.258.166-.057.347-.085.542-.085.215 0 .405.029.569.088.164.058.301.136.413.234.113.098.199.205.258.322.06.117.094.235.102.352ZM4.983 7.707c0 .154.032.287.094.398a.66.66 0 0 0 .255.258.734.734 0 0 0 .357.088.771.771 0 0 0 .36-.082.607.607 0 0 0 .253-.249.817.817 0 0 0 .093-.41.745.745 0 0 0-.1-.398.63.63 0 0 0-.257-.244.74.74 0 0 0-.346-.082.72.72 0 0 0-.369.094.645.645 0 0 0-.252.255.765.765 0 0 0-.088.372ZM1.785 8.999h-.79V5h2.548v.653H1.785V6.77h1.606v.638H1.785V9Z" />
      </svg>
    </Key>
  )
}

function F7Key() {
  return (
    <Key name="F7">
      <svg viewBox="0 0 12 14">
        <path d="M5.02 10.273a.637.637 0 0 1-.223-.039 1.273 1.273 0 0 1-.223-.101L.801 7.91A.89.89 0 0 1 .5 7.648a.56.56 0 0 1-.004-.628.917.917 0 0 1 .305-.266L4.574 4.53a1.27 1.27 0 0 1 .223-.101.7.7 0 0 1 .223-.035c.153 0 .285.057.394.171.112.112.168.276.168.493v4.546c0 .217-.056.382-.168.497a.534.534 0 0 1-.394.171Zm-.192-.77a.099.099 0 0 0 .067-.023c.018-.018.027-.045.027-.082V5.266c0-.037-.01-.063-.027-.079a.09.09 0 0 0-.067-.027.129.129 0 0 0-.039.008 2.154 2.154 0 0 0-.047.02L1.29 7.233c-.047.029-.07.062-.07.098 0 .036.023.069.07.098l3.453 2.047.047.02a.129.129 0 0 0 .04.007Zm5.348.77a.626.626 0 0 1-.219-.039 1.164 1.164 0 0 1-.219-.101L5.961 7.91a.89.89 0 0 1-.3-.262.56.56 0 0 1-.005-.628.917.917 0 0 1 .305-.266L9.738 4.53c.078-.044.151-.078.219-.101a.7.7 0 0 1 .223-.035c.153 0 .286.057.398.171.112.112.168.276.168.493v4.546c0 .217-.056.382-.168.497a.543.543 0 0 1-.402.171Zm-.188-.77c.029 0 .051-.007.067-.023.018-.018.027-.045.027-.082V5.266c0-.037-.01-.063-.027-.079a.083.083 0 0 0-.067-.027.128.128 0 0 0-.039.008 2.152 2.152 0 0 0-.047.02L6.45 7.233c-.05.029-.074.062-.074.098 0 .036.025.069.074.098l3.453 2.047.047.02a.128.128 0 0 0 .04.007Z" />
      </svg>
      <svg viewBox="0 0 7 14">
        <path d="M3.781 5h2.754v.66L4.912 8.998H4.08l1.661-3.32v-.034h-1.96V5ZM1.473 8.999H.682V5H3.23v.653H1.473V6.77h1.605v.638H1.473V9Z" />
      </svg>
    </Key>
  )
}

function F8Key() {
  return (
    <Key name="F8">
      <svg viewBox="0 0 12 14">
        <path d="M7.996 10.418a.358.358 0 0 1-.258-.102.338.338 0 0 1-.105-.254V4.61c0-.101.035-.186.105-.254a.358.358 0 0 1 .258-.101c.1 0 .184.034.254.101.073.068.11.153.11.254v5.454a.332.332 0 0 1-.11.253.353.353 0 0 1-.254.102Zm2.508 0a.366.366 0 0 1-.258-.102.338.338 0 0 1-.105-.254V4.61c0-.101.035-.186.105-.254a.366.366 0 0 1 .258-.101.35.35 0 0 1 .258.101c.07.068.105.153.105.254v5.454a.338.338 0 0 1-.105.253.35.35 0 0 1-.258.102Zm-8.813-.145a.534.534 0 0 1-.394-.171c-.112-.112-.168-.276-.168-.493V5.063c0-.217.056-.38.168-.493a.534.534 0 0 1 .394-.172.7.7 0 0 1 .223.036c.07.023.145.058.223.105L5.91 6.758c.138.08.238.168.3.262.066.093.099.199.099.316a.566.566 0 0 1-.098.316.89.89 0 0 1-.3.262l-3.774 2.219a1.137 1.137 0 0 1-.223.105.7.7 0 0 1-.223.035Zm.192-.765c.01 0 .023-.003.039-.008a.197.197 0 0 0 .047-.023l3.453-2.043c.05-.029.074-.062.074-.098 0-.037-.025-.07-.074-.098L1.969 5.195a.286.286 0 0 0-.051-.023.09.09 0 0 0-.102.02c-.018.015-.027.04-.027.074v4.14c0 .034.01.06.027.078.019.016.04.024.067.024Z" />
      </svg>
      <svg viewBox="0 0 8 14">
        <path d="M5.644 8.506a.879.879 0 0 0 .386-.082.667.667 0 0 0 .273-.229.596.596 0 0 0 .102-.345.587.587 0 0 0-.102-.346.66.66 0 0 0-.273-.232.853.853 0 0 0-.386-.085.872.872 0 0 0-.39.085.66.66 0 0 0-.273.232.6.6 0 0 0-.1.346.609.609 0 0 0 .373.574.898.898 0 0 0 .39.082Zm0-1.858c.12 0 .229-.024.325-.073a.545.545 0 0 0 .304-.504.53.53 0 0 0-.082-.296.526.526 0 0 0-.222-.199.705.705 0 0 0-.325-.073.705.705 0 0 0-.326.073.53.53 0 0 0-.308.495.56.56 0 0 0 .083.302c.054.086.13.153.225.202a.705.705 0 0 0 .326.073Zm-.01 2.43a2.09 2.09 0 0 1-.814-.147c-.23-.1-.409-.238-.536-.413a1.031 1.031 0 0 1-.19-.616c0-.16.025-.298.076-.413a.931.931 0 0 1 .205-.296 1.174 1.174 0 0 1 .557-.287v-.035a1.23 1.23 0 0 1-.326-.161.874.874 0 0 1-.254-.29.885.885 0 0 1-.1-.437c0-.203.058-.383.173-.542.117-.16.28-.286.486-.378.21-.091.453-.137.733-.137.279 0 .522.046.729.137.209.092.371.218.486.378a.897.897 0 0 1 .173.542.845.845 0 0 1-.357.73c-.104.072-.211.125-.322.158v.035c.09.018.183.05.28.097.098.047.19.11.273.19.086.08.155.18.205.296.053.115.08.252.08.41 0 .233-.065.438-.194.616-.129.175-.31.313-.545.413-.232.1-.505.15-.817.15ZM1.762 9.001H.97V5.002H3.52v.653H1.762v1.116h1.605v.64H1.762V9Z" />
      </svg>
    </Key>
  )
}

function F9Key() {
  return (
    <Key name="F9">
      <svg viewBox="0 0 12 14">
        <path d="M1.816 10.273a.539.539 0 0 1-.398-.171c-.112-.115-.168-.28-.168-.497V5.06c0-.217.056-.38.168-.493a.538.538 0 0 1 .398-.171.7.7 0 0 1 .223.035c.07.023.145.057.223.101l3.773 2.223c.136.08.235.17.297.266a.56.56 0 0 1 0 .629.85.85 0 0 1-.297.261l-3.773 2.223a1.273 1.273 0 0 1-.223.101.637.637 0 0 1-.223.04Zm.188-.77a.104.104 0 0 0 .039-.007.356.356 0 0 0 .047-.02L5.543 7.43c.052-.029.078-.062.078-.098 0-.036-.026-.069-.078-.098L2.09 5.188a2.154 2.154 0 0 0-.047-.02.104.104 0 0 0-.04-.008.1.1 0 0 0-.066.027c-.018.016-.027.042-.027.079v4.132c0 .037.01.064.028.082.02.016.042.024.066.024Zm4.973.77a.539.539 0 0 1-.399-.171c-.112-.115-.168-.28-.168-.497V5.06c0-.217.056-.38.168-.493a.538.538 0 0 1 .399-.171.7.7 0 0 1 .222.035c.07.023.145.057.223.101l3.773 2.223c.136.08.235.17.297.266a.56.56 0 0 1 0 .629.85.85 0 0 1-.297.261l-3.773 2.223a1.274 1.274 0 0 1-.223.101.637.637 0 0 1-.222.04Zm.187-.77a.104.104 0 0 0 .04-.007.355.355 0 0 0 .046-.02l3.453-2.046c.052-.029.078-.062.078-.098 0-.036-.026-.069-.078-.098L7.25 5.188a2.132 2.132 0 0 0-.047-.02.104.104 0 0 0-.039-.008.1.1 0 0 0-.066.027c-.019.016-.028.042-.028.079v4.132c0 .037.01.064.028.082.02.016.043.024.066.024Z" />
      </svg>
      <svg viewBox="0 0 8 14">
        <path d="M4.217 8.115h.753a.403.403 0 0 0 .096.161c.051.051.118.093.2.126.082.034.176.05.284.05a.701.701 0 0 0 .395-.108.8.8 0 0 0 .27-.3c.068-.126.117-.273.146-.441.032-.168.044-.35.038-.542h-.044a.832.832 0 0 1-.178.272.902.902 0 0 1-.317.208c-.127.05-.277.076-.45.076a1.39 1.39 0 0 1-.637-.15 1.207 1.207 0 0 1-.474-.445A1.376 1.376 0 0 1 4.12 6.3c0-.287.065-.533.193-.738.131-.206.308-.363.53-.472.223-.111.475-.167.757-.167.205 0 .4.033.586.1.185.066.35.176.495.33.144.155.257.362.34.622.083.257.125.579.125.963 0 .47-.06.862-.181 1.178-.12.317-.298.556-.536.718-.237.16-.533.24-.89.24-.214 0-.401-.027-.563-.082a1.292 1.292 0 0 1-.405-.222 1.065 1.065 0 0 1-.252-.308.913.913 0 0 1-.102-.346Zm2.068-1.822a.853.853 0 0 0-.09-.407.617.617 0 0 0-.253-.258.717.717 0 0 0-.363-.09.693.693 0 0 0-.351.09.64.64 0 0 0-.252.258.829.829 0 0 0-.091.401c0 .16.031.295.094.404a.63.63 0 0 0 .251.25.77.77 0 0 0 .355.081.746.746 0 0 0 .349-.082.63.63 0 0 0 .254-.246.768.768 0 0 0 .097-.401ZM1.78 8.999H.987V5h2.55v.653H1.778V6.77h1.606v.638H1.779V9Z" />
      </svg>
    </Key>
  )
}

function F10Key() {
  return (
    <Key name="F10">
      <svg viewBox="0 0 9 14">
        <path d="M2.386 8.734c-.293 0-.514-.076-.663-.229-.147-.155-.22-.39-.22-.703v-1.59c0-.314.073-.547.22-.7.15-.155.37-.232.663-.232h1.187a.14.14 0 0 0 .096-.035l1.85-1.64c.112-.1.209-.174.29-.224a.523.523 0 0 1 .277-.074c.15 0 .273.05.37.153a.52.52 0 0 1 .145.37v6.363a.48.48 0 0 1-.145.36.497.497 0 0 1-.36.14.639.639 0 0 1-.29-.065 1.365 1.365 0 0 1-.286-.202L3.67 8.769a.14.14 0 0 0-.097-.035H2.386Zm.053-.677h1.318c.053 0 .101.007.145.022a.372.372 0 0 1 .141.088l1.714 1.551c.023.02.047.031.07.031.035 0 .053-.02.053-.062V4.314c0-.041-.02-.062-.057-.062a.058.058 0 0 0-.035.014.117.117 0 0 0-.031.022L4.043 5.842a.43.43 0 0 1-.14.093.456.456 0 0 1-.146.022H2.44c-.073 0-.127.017-.163.052-.035.035-.052.09-.052.163v1.67c0 .073.017.127.052.162.036.036.09.053.163.053Z" />
      </svg>
      <svg viewBox="0 0 10 14">
        <path d="M7.338 6.915v.17c0 .281.03.522.088.724.06.2.145.355.255.462.109.108.239.162.39.162a.53.53 0 0 0 .389-.162c.11-.109.193-.263.252-.462a2.61 2.61 0 0 0 .088-.724v-.17c0-.28-.03-.52-.088-.72a1.023 1.023 0 0 0-.252-.46.53.53 0 0 0-.39-.162c-.15 0-.28.054-.39.161-.109.106-.194.26-.254.46-.059.2-.088.44-.088.721Zm-.788.176v-.179c0-.4.06-.749.181-1.046.123-.297.3-.527.528-.691.228-.166.502-.25.82-.25.318 0 .59.083.815.247.226.162.399.391.518.688.12.297.179.648.179 1.052v.179c0 .299-.035.57-.106.814a1.869 1.869 0 0 1-.301.63 1.372 1.372 0 0 1-.48.402c-.188.093-.401.14-.64.14-.316 0-.587-.08-.814-.243a1.551 1.551 0 0 1-.521-.691 2.83 2.83 0 0 1-.18-1.052ZM5.463 9.001h-.785V5.764H4.64l-.302.214-.305.216-.302.214v-.726l.475-.34.472-.34h.785V9ZM1.338 9.001H.547V5.002h2.549v.653H1.338v1.116h1.605v.64H1.338V9Z" />
      </svg>
    </Key>
  )
}

function F11Key() {
  return (
    <Key name="F11">
      <svg viewBox="0 0 11 14">
        <path d="M2.622 8.734c-.293 0-.515-.076-.664-.229-.146-.155-.22-.39-.22-.703v-1.59c0-.314.074-.547.22-.7.15-.155.37-.232.664-.232h1.186a.14.14 0 0 0 .097-.035l1.85-1.64c.111-.1.208-.174.29-.224a.515.515 0 0 1 .272-.074c.15 0 .273.05.37.153a.52.52 0 0 1 .145.37v6.363a.488.488 0 0 1-.505.5.629.629 0 0 1-.286-.065 1.365 1.365 0 0 1-.286-.202l-1.85-1.657a.14.14 0 0 0-.097-.035H2.622Zm.052-.677h1.319c.052 0 .1.007.145.022a.372.372 0 0 1 .14.088l1.714 1.551c.024.02.047.031.07.031.036 0 .053-.02.053-.062V4.314c0-.041-.019-.062-.057-.062a.058.058 0 0 0-.035.014.117.117 0 0 0-.03.022L4.277 5.842a.43.43 0 0 1-.14.093.456.456 0 0 1-.145.022H2.674c-.073 0-.127.017-.162.052-.035.035-.053.09-.053.163v1.67c0 .073.018.127.053.162.035.036.09.053.162.053Zm5.568.756a.357.357 0 0 1-.162-.25.418.418 0 0 1 .087-.308c.118-.167.208-.359.273-.576.064-.217.097-.442.097-.677 0-.234-.033-.46-.097-.676a1.963 1.963 0 0 0-.273-.58.405.405 0 0 1-.087-.304.357.357 0 0 1 .162-.25.315.315 0 0 1 .26-.053c.093.018.17.068.228.15.167.222.296.483.387.782.09.296.136.606.136.931 0 .326-.045.638-.136.936-.091.3-.22.557-.387.774a.366.366 0 0 1-.229.158.329.329 0 0 1-.259-.057Z" />
      </svg>
      <svg viewBox="0 0 9 14">
        <path d="M8.273 8.999h-.785V5.762H7.45l-.302.214-.304.216-.302.214V5.68l.475-.34.471-.34h.785v3.999ZM5.42 8.999h-.785V5.762h-.038l-.302.214-.305.216-.302.214V5.68l.475-.34.472-.34h.785v3.999ZM1.295 8.999H.504V5h2.549v.653H1.295V6.77H2.9v.638H1.295V9Z" />
      </svg>
    </Key>
  )
}

function F12Key() {
  return (
    <Key name="F12">
      <svg viewBox="0 0 14 14">
        <path d="M2.18 8.734c-.294 0-.515-.076-.664-.229-.147-.155-.22-.39-.22-.703v-1.59c0-.314.073-.547.22-.7.149-.155.37-.232.663-.232h1.187a.14.14 0 0 0 .096-.035l1.85-1.64c.112-.1.209-.174.29-.224a.523.523 0 0 1 .277-.074c.15 0 .273.05.37.153a.52.52 0 0 1 .145.37v6.363a.48.48 0 0 1-.145.36.497.497 0 0 1-.36.14.639.639 0 0 1-.29-.065 1.364 1.364 0 0 1-.287-.202l-1.85-1.657a.14.14 0 0 0-.096-.035H2.179Zm.052-.677H3.55c.053 0 .101.007.145.022a.372.372 0 0 1 .14.088L5.55 9.718c.023.02.047.031.07.031.035 0 .053-.02.053-.062V4.313c0-.041-.02-.062-.057-.062a.058.058 0 0 0-.035.014.117.117 0 0 0-.031.022L3.836 5.842a.43.43 0 0 1-.14.093.456.456 0 0 1-.146.022H2.232c-.073 0-.128.017-.163.052-.035.035-.052.09-.052.163v1.67c0 .073.017.127.052.162.035.036.09.053.163.053Zm5.572.752a.357.357 0 0 1-.162-.25.418.418 0 0 1 .087-.309c.118-.167.208-.358.273-.575.064-.217.097-.443.097-.677 0-.234-.033-.46-.097-.677a1.963 1.963 0 0 0-.273-.58.405.405 0 0 1-.087-.303.357.357 0 0 1 .162-.25.315.315 0 0 1 .26-.053c.093.017.17.067.228.15.167.222.296.483.387.781.09.296.136.607.136.932 0 .325-.045.637-.136.936-.091.299-.22.557-.387.773a.366.366 0 0 1-.229.159.329.329 0 0 1-.259-.057Zm1.6 1.098a.317.317 0 0 1-.15-.233c-.011-.1.015-.199.08-.299.222-.322.395-.688.518-1.098a4.456 4.456 0 0 0 .004-2.558 3.737 3.737 0 0 0-.522-1.098.438.438 0 0 1-.08-.295.32.32 0 0 1 .15-.237.33.33 0 0 1 .268-.057c.097.02.174.074.233.162.272.37.483.8.633 1.292a5.147 5.147 0 0 1-.005 3.024c-.15.486-.359.917-.628 1.292a.363.363 0 0 1-.233.162.355.355 0 0 1-.268-.057Zm1.617 1.112a.32.32 0 0 1-.15-.237.43.43 0 0 1 .08-.295 6.144 6.144 0 0 0 .91-2.206c.084-.413.127-.84.127-1.283a6.294 6.294 0 0 0-.488-2.452 6.473 6.473 0 0 0-.55-1.037.43.43 0 0 1-.078-.295.32.32 0 0 1 .149-.237.348.348 0 0 1 .272-.053c.097.02.178.078.242.172A7.06 7.06 0 0 1 12.56 5.57c.094.46.14.936.14 1.428 0 .495-.046.973-.14 1.433-.094.46-.228.898-.404 1.314-.173.413-.38.8-.62 1.16a.392.392 0 0 1-.242.167.348.348 0 0 1-.272-.053Z" />
      </svg>
      <svg viewBox="0 0 10 14">
        <path d="M7.45 6.194H6.7v-.038a1.174 1.174 0 0 1 .337-.844c.112-.117.254-.21.424-.28.172-.071.374-.106.606-.106.276 0 .516.049.721.146.205.098.363.232.475.402.113.168.17.358.17.57 0 .185-.036.349-.106.493a1.825 1.825 0 0 1-.264.401c-.105.124-.216.25-.334.378l-.908 1.002v.038H9.5v.645H6.737v-.492L8.2 6.933c.072-.077.143-.156.211-.238.069-.084.125-.174.17-.27a.715.715 0 0 0 .067-.307.534.534 0 0 0-.29-.486.634.634 0 0 0-.307-.073.608.608 0 0 0-.322.082.562.562 0 0 0-.208.22.618.618 0 0 0-.07.29v.043ZM5.595 9.001H4.81V5.764H4.77a97.908 97.908 0 0 0-.606.43l-.302.214v-.726l.475-.34.472-.34h.785V9ZM1.47 9.001H.679V5.002h2.549v.653H1.47v1.116h1.605v.64H1.47V9Z" />
      </svg>
    </Key>
  )
}

function LockKey() {
  return (
    <Key name="Lock" className="rounded-tr-xl">
      <svg viewBox="0 0 12 14">
        <path d="M3.642 12.241c-.383 0-.671-.1-.865-.3-.193-.198-.29-.506-.29-.924V7.456c0-.416.097-.722.29-.919.194-.197.482-.295.865-.295h4.71c.39 0 .68.098.87.295.19.197.285.503.285.919v3.56c0 .42-.095.727-.284.924-.19.201-.48.301-.87.301h-4.71Zm.097-.838h4.528c.114 0 .198-.03.252-.091.057-.057.086-.154.086-.29V7.45c0-.136-.029-.23-.086-.284-.054-.058-.138-.086-.252-.086H3.739c-.115 0-.2.028-.258.086-.057.053-.086.148-.086.284v3.572c0 .136.029.233.086.29.057.06.143.091.258.091Zm-.35-4.742V5.049c0-.48.072-.898.215-1.257.147-.358.342-.655.586-.891.247-.236.526-.412.838-.526.311-.119.635-.178.972-.178.333 0 .655.06.967.178.311.114.59.29.838.526.247.236.442.533.585.891.147.359.22.777.22 1.257v1.612h-.897V4.947c0-.412-.078-.757-.236-1.036a1.6 1.6 0 0 0-.623-.64A1.729 1.729 0 0 0 6 3.057c-.308 0-.593.071-.854.214-.261.144-.47.357-.628.64-.158.279-.237.624-.237 1.036v1.714H3.39Z" />
      </svg>
    </Key>
  )
}

function BacktickKey() {
  return (
    <Key name="Backtick">
      <svg viewBox="0 0 8 14">
        <path d="M5.414 8.636c-.37 0-.681-.073-.936-.219a3.372 3.372 0 0 1-.677-.52 5.709 5.709 0 0 0-.568-.512.942.942 0 0 0-.615-.226.69.69 0 0 0-.465.164c-.123.11-.218.255-.287.438a1.847 1.847 0 0 0-.116.588H.677a2.858 2.858 0 0 1 .198-1.094c.137-.346.344-.627.622-.84.283-.22.643-.33 1.08-.33.378 0 .693.076.943.227.256.145.479.316.67.512.192.196.376.37.554.52a.96.96 0 0 0 .615.218.674.674 0 0 0 .465-.164c.128-.113.224-.26.287-.437.064-.182.1-.378.11-.588h1.073c.004.392-.064.761-.205 1.107a1.933 1.933 0 0 1-.622.841c-.278.21-.63.315-1.053.315Z" />
      </svg>
      <svg viewBox="0 0 7 14">
        <path d="M4.56 5.434h-.944L1.716 3h1.217L4.56 5.434Z" />
      </svg>
    </Key>
  )
}

function OneKey() {
  return (
    <Key name="One">
      <svg viewBox="0 0 4 14">
        <path d="M.768 10.355a.747.747 0 0 1 .763-.757c.229 0 .417.071.564.215.15.143.225.324.225.542a.705.705 0 0 1-.225.532.785.785 0 0 1-.564.21.757.757 0 0 1-.543-.21.713.713 0 0 1-.22-.532ZM1 3.668h1.096L1.97 8.389h-.854L1 3.668Z" />
      </svg>
      <svg viewBox="0 0 6 14">
        <path d="M4.263 12H3.108V4.587H3.05l-.673.483c-.22.16-.442.321-.667.482l-.666.489V4.866a161.932 161.932 0 0 1 2.063-1.53h1.155V12Z" />
      </svg>
    </Key>
  )
}

function TwoKey() {
  return (
    <Key name="Two">
      <svg viewBox="0 0 10 14">
        <path d="M7.208 10.285a1.66 1.66 0 0 1-.645-.113 1.166 1.166 0 0 1-.414-.279.802.802 0 0 1-.182-.354h-.054a.849.849 0 0 1-.21.343c-.103.115-.24.212-.408.29a1.344 1.344 0 0 1-.564.113c-.447 0-.794-.163-1.042-.489-.243-.325-.365-.75-.365-1.273V8.03c0-.512.126-.929.376-1.251.25-.326.61-.489 1.08-.489.204 0 .38.038.526.113a1.02 1.02 0 0 1 .537.575h.043v-.613h.779v2.627c0 .14.027.258.08.354a.572.572 0 0 0 .237.215c.1.047.211.07.333.07a.743.743 0 0 0 .494-.193c.15-.13.272-.33.365-.602.097-.272.145-.618.145-1.037V7.61c0-.522-.123-1.008-.37-1.455a2.78 2.78 0 0 0-1.074-1.09c-.47-.276-1.042-.414-1.72-.414h-.042c-.63 0-1.198.141-1.703.424-.501.28-.899.679-1.192 1.198-.294.52-.44 1.137-.44 1.853v.247c0 .57.087 1.067.263 1.493.175.426.415.779.72 1.058.304.28.655.49 1.052.629.397.14.82.21 1.268.21h.048c.401 0 .741-.026 1.02-.076a3.04 3.04 0 0 0 .71-.193v.693c-.151.068-.371.127-.661.177-.287.053-.647.08-1.08.08H5.07a4.865 4.865 0 0 1-1.58-.252 3.696 3.696 0 0 1-1.294-.757 3.487 3.487 0 0 1-.875-1.263c-.208-.505-.312-1.094-.312-1.767v-.311c0-.842.18-1.57.537-2.186a3.735 3.735 0 0 1 1.461-1.429c.62-.34 1.322-.51 2.106-.51h.043c.634 0 1.194.1 1.68.3a3.471 3.471 0 0 1 1.972 1.972c.172.43.258.877.258 1.343v.273c0 .491-.08.919-.242 1.284a2.022 2.022 0 0 1-.66.843c-.28.2-.598.301-.956.301ZM4.135 8.416c0 .415.08.714.242.897A.8.8 0 0 0 5 9.582a.85.85 0 0 0 .446-.113.769.769 0 0 0 .306-.36c.075-.165.113-.381.113-.65v-.43c0-.236-.04-.435-.118-.596a.828.828 0 0 0-.312-.365.827.827 0 0 0-.446-.124.79.79 0 0 0-.607.28c-.164.182-.247.469-.247.859v.333Z" />
      </svg>
      <svg viewBox="0 0 8 14">
        <path d="M2.407 5.748H1.27v-.044c0-.33.059-.648.177-.953a2.38 2.38 0 0 1 .527-.812c.233-.241.52-.432.863-.571.343-.14.739-.21 1.187-.21.55 0 1.029.104 1.435.311a2.252 2.252 0 0 1 1.276 2.063c0 .36-.068.69-.203.99a4.046 4.046 0 0 1-.533.87c-.22.28-.466.573-.737.882L2.96 10.915v.07h3.942V12H1.302v-.737l3.263-3.713c.157-.182.311-.37.463-.565.153-.199.278-.41.375-.635.101-.224.152-.47.152-.736 0-.296-.068-.555-.203-.775a1.351 1.351 0 0 0-.559-.507A1.724 1.724 0 0 0 4 4.155c-.334 0-.622.072-.863.216a1.474 1.474 0 0 0-.546.57 1.608 1.608 0 0 0-.184.763v.045Z" />
      </svg>
    </Key>
  )
}

function ThreeKey() {
  return (
    <Key name="Three">
      <svg viewBox="0 0 7 14">
        <path d="M6.476 6.434h-1.22l-.36 1.74H6.1l-.145.747H4.746L4.327 11h-.773l.424-2.08H2.302L1.878 11h-.774l.42-2.08H.378l.15-.746h1.156l.354-1.74H.9l.145-.747H2.19l.409-2.019h.778l-.413 2.02h1.676l.413-2.02h.768l-.408 2.02h1.214l-.15.746Zm-3.674-.027-.37 1.789h1.707l.365-1.789H2.802Z" />
      </svg>
      <svg viewBox="0 0 8 14">
        <path d="M2.965 8.033v-.996h.895c.305 0 .576-.064.813-.19.241-.128.43-.303.565-.528.135-.224.2-.478.197-.761 0-.42-.146-.76-.438-1.022-.292-.263-.671-.394-1.137-.394a1.83 1.83 0 0 0-.628.102 1.572 1.572 0 0 0-.482.266c-.14.11-.25.237-.33.381-.077.144-.121.296-.134.457h-1.11a2.105 2.105 0 0 1 .705-1.504c.227-.207.507-.373.837-.495.33-.123.713-.186 1.149-.19.55-.005 1.028.09 1.434.285.407.195.724.46.952.793.229.335.347.713.356 1.137.013.406-.07.757-.248 1.053a2.154 2.154 0 0 1-.64.705c-.255.17-.492.273-.712.311v.063c.246.026.512.115.8.267.292.152.544.379.755.68.216.296.328.676.337 1.142.008.376-.064.723-.216 1.04a2.37 2.37 0 0 1-.641.826c-.275.233-.601.413-.978.54-.376.122-.789.184-1.237.184-.576 0-1.075-.098-1.498-.292a2.476 2.476 0 0 1-.984-.8 2.14 2.14 0 0 1-.381-1.124h1.092c.03.229.122.436.279.622.156.182.364.328.622.438.262.106.56.161.895.165.38 0 .71-.067.99-.203.284-.14.502-.33.654-.57.152-.246.226-.524.222-.832a1.469 1.469 0 0 0-.876-1.359 2.09 2.09 0 0 0-.933-.197h-.946Z" />
      </svg>
    </Key>
  )
}

function FourKey() {
  return (
    <Key name="Four">
      <svg viewBox="0 0 7 14">
        <path d="M1.013 9.065h.946a1.268 1.268 0 0 0 .547.8c.147.097.315.17.505.22.194.051.405.076.634.076.326 0 .607-.048.843-.145.237-.1.417-.242.543-.424.125-.183.188-.398.188-.645a.948.948 0 0 0-.118-.483.996.996 0 0 0-.371-.36 2.372 2.372 0 0 0-.65-.247L2.86 7.54a2.748 2.748 0 0 1-.89-.392 1.81 1.81 0 0 1-.57-.623 1.773 1.773 0 0 1-.193-.833c0-.397.105-.746.316-1.047.212-.304.5-.542.865-.714.369-.176.79-.264 1.262-.264.462 0 .869.08 1.22.237.354.157.637.374.848.65.212.275.335.592.37.95h-.95a1.151 1.151 0 0 0-.263-.542 1.278 1.278 0 0 0-.516-.355 1.943 1.943 0 0 0-.72-.123c-.443 0-.801.11-1.074.328a1.023 1.023 0 0 0-.402.837c0 .247.082.455.247.623.168.169.413.294.736.376l1.197.317c.412.108.752.246 1.02.414.273.168.475.376.608.623.136.247.204.539.204.875 0 .427-.108.797-.322 1.112-.212.312-.514.555-.908.73-.394.176-.861.264-1.402.264-.369 0-.704-.047-1.004-.14-.301-.093-.56-.225-.78-.397a2.003 2.003 0 0 1-.52-.607 2.018 2.018 0 0 1-.226-.774Zm2.235 2.675V2.91h.639v8.83h-.64Z" />
      </svg>
      <svg viewBox="0 0 8 14">
        <path d="M4.432 3.336h1.726v5.872h1.124v1.015H6.158V12H5.041v-1.777H.833V9.201A52.895 52.895 0 0 1 2.578 6.27c.622-.978 1.24-1.956 1.854-2.933ZM1.988 9.208H5.04v-4.87h-.063A98.029 98.029 0 0 0 3.44 6.714c-.516.83-1 1.642-1.453 2.437v.058Z" />
      </svg>
    </Key>
  )
}

function FiveKey() {
  return (
    <Key name="Five">
      <svg viewBox="0 0 9 14">
        <path d="M6.643 10.398a.593.593 0 0 0 .516-.29c.132-.197.198-.458.198-.784 0-.33-.064-.589-.193-.779a.59.59 0 0 0-.52-.29.597.597 0 0 0-.522.29c-.129.194-.193.453-.193.78 0 .325.064.586.193.783.133.194.306.29.521.29Zm0 .704c-.465 0-.847-.165-1.144-.494-.294-.33-.44-.757-.44-1.284 0-.519.146-.943.44-1.273.297-.333.679-.5 1.144-.5.47 0 .85.165 1.144.495.297.33.446.755.446 1.278 0 .527-.149.954-.446 1.284-.294.33-.675.494-1.144.494ZM2.427 11H1.396l2.75-3.819L6.61 3.668h1.02L4.833 7.552 2.427 11Zm-.07-4.614a.593.593 0 0 0 .516-.29c.132-.197.198-.456.198-.779 0-.333-.064-.594-.193-.784a.597.597 0 0 0-.521-.29.61.61 0 0 0-.526.29c-.13.194-.194.455-.194.784 0 .326.065.586.194.78a.61.61 0 0 0 .526.29Zm0 .71c-.466 0-.849-.165-1.15-.495-.297-.333-.445-.76-.445-1.284 0-.522.148-.949.446-1.278.3-.33.683-.494 1.149-.494.469 0 .85.165 1.144.494.294.326.44.752.44 1.278 0 .523-.146.951-.44 1.284-.294.33-.675.494-1.144.494Z" />
      </svg>
      <svg viewBox="0 0 8 14">
        <path d="M1.163 9.982H2.28a1.334 1.334 0 0 0 .654.927c.165.097.342.169.533.215.194.043.391.064.59.064.296 0 .584-.074.863-.222.284-.152.515-.377.692-.673.182-.3.273-.664.273-1.092 0-.385-.087-.713-.26-.984a1.719 1.719 0 0 0-.68-.634 1.908 1.908 0 0 0-.907-.223 1.92 1.92 0 0 0-1.352.52 1.84 1.84 0 0 0-.35.433h-.99l.388-4.977h4.894v1.01H2.724L2.49 7.118h.064c.063-.093.169-.197.317-.31.152-.12.347-.221.584-.306.237-.084.516-.127.838-.127.504 0 .965.113 1.384.337.419.22.753.537 1.003.952.254.415.38.916.38 1.504 0 .593-.13 1.115-.393 1.568a2.737 2.737 0 0 1-1.08 1.06c-.456.254-.99.381-1.599.381-.397 0-.766-.055-1.104-.165a2.697 2.697 0 0 1-.876-.457 2.343 2.343 0 0 1-.597-.698 2.155 2.155 0 0 1-.247-.876Z" />
      </svg>
    </Key>
  )
}

function SixKey() {
  return (
    <Key name="Six">
      <svg viewBox="0 0 8 14">
        <path d="M3.46 3.68h.774L7 8.31H5.969L3.906 4.716h-.048L1.903 8.31H.883L3.46 3.68Z" />
      </svg>
      <svg viewBox="0 0 8 14">
        <path d="M6.907 5.131H5.74a1.06 1.06 0 0 0-.222-.425 1.44 1.44 0 0 0-.501-.38c-.216-.107-.487-.16-.813-.16a1.668 1.668 0 0 0-1.257.56c-.169.185-.317.42-.444.704-.127.279-.224.611-.292.996-.068.381-.102.813-.102 1.295h.064a2.105 2.105 0 0 1 1.124-1.098c.3-.135.66-.203 1.079-.203.452 0 .884.106 1.294.317.415.212.756.523 1.022.933.267.41.4.917.4 1.518 0 .634-.13 1.176-.393 1.624a2.568 2.568 0 0 1-1.067 1.016c-.448.233-.958.35-1.53.35-.313 0-.622-.041-.926-.121a2.684 2.684 0 0 1-.863-.407 2.752 2.752 0 0 1-.705-.78 4.092 4.092 0 0 1-.47-1.181c-.114-.47-.171-1.03-.171-1.682 0-.669.05-1.265.152-1.79.106-.525.254-.978.444-1.359.195-.385.426-.7.692-.945.271-.25.574-.436.908-.559a3.107 3.107 0 0 1 1.08-.184c.405 0 .77.057 1.091.171.326.115.6.267.825.457.229.187.406.396.533.629.127.228.2.463.216.704Zm-4.64 4.114c0 .41.085.76.254 1.047.17.288.394.508.673.66.28.152.584.229.914.229.338 0 .645-.068.92-.204.28-.14.502-.35.667-.634.165-.288.247-.65.247-1.086 0-.423-.088-.772-.266-1.047a1.644 1.644 0 0 0-.68-.61 1.97 1.97 0 0 0-.882-.203c-.355 0-.673.08-.952.242-.28.156-.5.374-.66.653a1.891 1.891 0 0 0-.235.953Z" />
      </svg>
    </Key>
  )
}

function SevenKey() {
  return (
    <Key name="Seven">
      <svg viewBox="0 0 8 14">
        <path d="M6.363 11a61.63 61.63 0 0 0-.16-.172 7.874 7.874 0 0 0-.167-.183 136.97 136.97 0 0 1-.328-.35 3.013 3.013 0 0 1-1.16.662 4.084 4.084 0 0 1-1.155.182c-.455 0-.87-.082-1.246-.247a2.102 2.102 0 0 1-.886-.709c-.219-.311-.328-.682-.328-1.112 0-.322.061-.605.183-.848.125-.247.295-.466.51-.656a3.996 3.996 0 0 1 .87-.58c.047-.025.093-.048.14-.07a7.04 7.04 0 0 1-.629-.778 1.577 1.577 0 0 1-.274-.908c0-.3.08-.573.237-.816.157-.248.377-.444.66-.591a2.08 2.08 0 0 1 .973-.22c.35 0 .666.07.945.209.283.136.507.328.671.575.169.243.253.53.253.86 0 .296-.072.558-.215.783-.14.222-.328.42-.564.591-.233.172-.49.333-.774.483.14.15.283.307.43.468.147.157.296.318.446.483l.446.489c.147.157.29.313.43.467.118-.222.202-.496.252-.822.05-.329.075-.705.075-1.127v-.237h.886v.247a6.95 6.95 0 0 1-.14 1.456c-.092.43-.243.802-.45 1.117l1.272 1.353H6.363ZM2.915 7.604a2.812 2.812 0 0 0-.747.58c-.186.215-.279.489-.279.822 0 .304.075.557.226.757.154.197.35.344.59.44.24.097.498.146.774.146.269 0 .559-.05.87-.15.315-.105.584-.273.806-.506L3.15 7.481a3.887 3.887 0 0 0-.123.065 3.576 3.576 0 0 0-.113.059Zm.494-1.031c.337-.172.614-.353.833-.543a.95.95 0 0 0 .333-.752c0-.283-.092-.508-.274-.676a.97.97 0 0 0-.693-.258c-.287 0-.52.09-.698.268a.91.91 0 0 0-.264.672c0 .193.054.385.162.574.11.187.311.425.601.715Z" />
      </svg>
      <svg viewBox="0 0 7 14">
        <path d="M.758 3.336h5.541V4.39L2.757 12H1.532L5.12 4.422v-.07H.758V3.337Z" />
      </svg>
    </Key>
  )
}

function EightKey() {
  return (
    <Key name="Eight">
      <svg viewBox="0 0 6 14">
        <path d="M3.365 7.891h-.752l.113-1.75-1.455.977-.371-.65 1.574-.779L.9 4.916l.37-.65 1.461.983-.118-1.757h.752L3.247 5.25l1.461-.983.37.65-1.578.773 1.579.779-.371.65-1.46-.978.117 1.751Z" />
      </svg>
      <svg viewBox="0 0 8 14">
        <path d="M4.006 11.22c.369 0 .699-.068.99-.203.293-.14.523-.333.692-.578.17-.245.254-.529.254-.85 0-.326-.084-.612-.254-.858-.169-.25-.4-.442-.691-.577a2.267 2.267 0 0 0-.99-.21c-.369 0-.701.07-.997.21a1.704 1.704 0 0 0-.698.577c-.17.246-.254.532-.254.857 0 .322.084.606.254.851.173.245.406.438.698.578.296.135.628.203.996.203Zm0-4.209c.313 0 .59-.061.832-.184a1.38 1.38 0 0 0 .78-1.27 1.349 1.349 0 0 0-.78-1.25 1.863 1.863 0 0 0-.832-.178c-.313 0-.592.06-.838.178a1.392 1.392 0 0 0-.577.502c-.14.211-.21.46-.21.749 0 .283.07.535.21.755.14.216.332.387.577.514.246.123.525.184.838.184Zm-.031 5.167c-.597 0-1.126-.106-1.587-.317a2.644 2.644 0 0 1-1.086-.895 2.308 2.308 0 0 1-.387-1.327c0-.343.055-.641.165-.895.114-.254.26-.467.438-.641a2.475 2.475 0 0 1 1.149-.647v-.058a2.48 2.48 0 0 1-.628-.336 2.041 2.041 0 0 1-.559-.635c-.148-.262-.222-.582-.222-.958 0-.449.118-.847.355-1.194s.561-.62.971-.818c.415-.2.89-.299 1.422-.299.534 0 1.005.1 1.416.299.415.198.74.471.977.818s.356.745.356 1.194c0 .385-.078.71-.235.977a2.038 2.038 0 0 1-.565.629c-.22.152-.427.26-.622.323v.058a2.475 2.475 0 0 1 1.149.647c.182.178.328.394.438.647.11.25.165.542.165.876 0 .504-.133.948-.4 1.334-.267.38-.635.679-1.104.894-.47.216-1.005.324-1.606.324Z" />
      </svg>
    </Key>
  )
}

function NineKey() {
  return (
    <Key name="Nine">
      <svg viewBox="0 0 4 14">
        <path d="M2.752 3.281h.843a7.74 7.74 0 0 0-.693 1.295 8.39 8.39 0 0 0-.66 3.298c0 .615.048 1.204.145 1.767.096.558.245 1.088.446 1.59.2.504.454.98.762 1.428h-.848a5.813 5.813 0 0 1-.79-1.289 7.802 7.802 0 0 1-.516-1.627 9.459 9.459 0 0 1-.177-1.859c0-.648.061-1.25.183-1.804a7.056 7.056 0 0 1 .515-1.531c.226-.47.49-.892.79-1.268Z" />
      </svg>
      <svg viewBox="0 0 8 14">
        <path d="M1.144 10.33h1.174c.038.136.116.269.235.4.122.127.285.233.489.317.203.08.446.121.73.121.397 0 .736-.091 1.015-.273.28-.186.502-.44.667-.762.169-.326.287-.7.355-1.123.072-.428.102-.885.089-1.371h-.064c-.088.237-.228.459-.418.666a1.985 1.985 0 0 1-.711.495 2.593 2.593 0 0 1-1.022.184c-.512 0-.978-.114-1.397-.342a2.596 2.596 0 0 1-1.003-1.003c-.245-.44-.368-.963-.368-1.568 0-.614.133-1.139.4-1.574.267-.436.626-.77 1.079-1.003.457-.233.97-.35 1.536-.35.381 0 .753.064 1.117.191.369.127.7.35.997.666.3.318.54.758.717 1.32.178.56.267 1.275.267 2.146 0 1.037-.119 1.905-.356 2.603-.237.694-.598 1.216-1.085 1.568-.483.35-1.096.527-1.84.527-.403 0-.76-.051-1.074-.153a2.458 2.458 0 0 1-.787-.4 2.111 2.111 0 0 1-.514-.59 1.88 1.88 0 0 1-.228-.692ZM5.7 6.122c0-.428-.082-.788-.247-1.08A1.644 1.644 0 0 0 4.8 4.37a1.865 1.865 0 0 0-.94-.235c-.321 0-.62.078-.895.235-.27.152-.488.374-.653.666-.161.288-.242.64-.242 1.054 0 .431.085.79.254 1.073.17.279.392.488.667.628.279.135.58.203.901.203.309 0 .601-.068.876-.203.28-.14.504-.347.673-.622.173-.28.26-.628.26-1.047Z" />
      </svg>
    </Key>
  )
}

function ZeroKey() {
  return (
    <Key name="Zero">
      <svg viewBox="0 0 4 14">
        <path d="M.405 3.281h.843c.304.376.568.799.79 1.268.222.465.393.976.515 1.53a8.42 8.42 0 0 1 .183 1.805 9.19 9.19 0 0 1-.183 1.859 7.592 7.592 0 0 1-.51 1.627 5.558 5.558 0 0 1-.79 1.29H.405c.308-.448.56-.925.757-1.43.2-.5.35-1.03.446-1.59.1-.562.15-1.15.15-1.766 0-.613-.06-1.196-.182-1.751a8.312 8.312 0 0 0-.484-1.547c-.2-.48-.43-.912-.687-1.295Z" />
      </svg>
      <svg viewBox="0 0 8 14">
        <path d="M2.064 7.519v.305c0 .694.078 1.29.235 1.79.16.5.385.884.673 1.155.292.267.634.4 1.028.4a1.47 1.47 0 0 0 1.035-.4c.287-.27.51-.656.666-1.155.157-.5.235-1.096.235-1.79v-.305c0-.694-.078-1.29-.235-1.79-.156-.504-.379-.889-.666-1.155A1.457 1.457 0 0 0 4 4.167c-.394 0-.736.136-1.028.407-.288.266-.512.651-.673 1.155-.157.5-.235 1.096-.235 1.79Zm-1.162.317v-.323c0-.872.123-1.634.369-2.285.245-.656.598-1.164 1.06-1.524.465-.364 1.026-.546 1.682-.546.656 0 1.212.182 1.67.546.46.36.812.865 1.053 1.517.241.648.362 1.411.362 2.292v.323c0 .656-.07 1.25-.21 1.784a4.224 4.224 0 0 1-.603 1.378c-.262.38-.586.672-.971.876A2.773 2.773 0 0 1 4 12.178c-.652 0-1.21-.18-1.676-.54-.461-.359-.814-.863-1.06-1.51-.241-.652-.362-1.416-.362-2.292Z" />
      </svg>
    </Key>
  )
}

function DashKey() {
  return (
    <Key name="Dash">
      <svg viewBox="0 0 6 14">
        <path d="M5.85 8.901H.144V8H5.85v.901Z" />
      </svg>
      <svg viewBox="0 0 6 14">
        <path d="M5 8.96H1V7.932h4V8.96Z" />
      </svg>
    </Key>
  )
}

function EqualsKey() {
  return (
    <Key name="Equals">
      <svg viewBox="0 0 8 14">
        <path d="M4.47 9.029v2.818h-.946V9.03H.94v-.952h2.584V5.404h.946v2.673h2.59v.952H4.47Z" />
      </svg>
      <svg viewBox="0 0 8 14">
        <path d="M6.679 10.177H1.315v-.92h5.364v.92Zm0-2.361H1.315v-.933h5.364v.933Z" />
      </svg>
    </Key>
  )
}

function DeleteKey() {
  return (
    <Key name="Delete" width={64}>
      <svg viewBox="0 0 24 14" className="mt-auto ml-auto">
        <path d="M21.871 10.066c-.367 0-.682-.073-.945-.218a1.5 1.5 0 0 1-.606-.63 2.092 2.092 0 0 1-.207-.96v-.625c0-.373.073-.693.219-.961.148-.271.352-.48.61-.625.257-.149.549-.223.874-.223.334 0 .628.074.883.223.258.148.46.358.606.629.146.27.218.59.218.957v.496h-2.707v.207c0 .318.09.586.27.805.18.218.44.328.781.328.159 0 .3-.022.422-.067a.862.862 0 0 0 .3-.18.537.537 0 0 0 .153-.238h.727a.984.984 0 0 1-.133.352c-.07.125-.17.243-.3.355-.131.11-.293.2-.485.27-.193.07-.42.105-.68.105Zm-1.055-2.449h2.004v-.066a1.33 1.33 0 0 0-.125-.594.937.937 0 0 0-.351-.398.96.96 0 0 0-.524-.141.98.98 0 0 0-.527.14.937.937 0 0 0-.352.399 1.33 1.33 0 0 0-.125.594v.066ZM17.527 5.902v-1.03h.7v1.03h.93v.575h-.93v2.367c0 .242.048.405.144.488.096.08.27.121.52.121.05 0 .108-.002.175-.008l.149-.011V10a4.384 4.384 0 0 1-.504.035c-.326 0-.574-.042-.746-.125a.684.684 0 0 1-.348-.39 2.046 2.046 0 0 1-.094-.672V6.477h-.628v-.575h.632ZM14.473 10.066c-.368 0-.683-.073-.946-.218a1.5 1.5 0 0 1-.605-.63 2.092 2.092 0 0 1-.207-.96v-.625c0-.373.073-.693.219-.961.148-.271.351-.48.609-.625.258-.149.55-.223.875-.223.333 0 .628.074.883.223.258.148.46.358.605.629.146.27.219.59.219.957v.496h-2.707v.207c0 .318.09.586.27.805.18.218.44.328.78.328.16 0 .3-.022.423-.067a.863.863 0 0 0 .3-.18.539.539 0 0 0 .153-.238h.726a.983.983 0 0 1-.133.352c-.07.125-.17.243-.3.355-.13.11-.292.2-.485.27-.192.07-.419.105-.68.105Zm-1.055-2.449h2.004v-.066a1.33 1.33 0 0 0-.125-.594.938.938 0 0 0-.352-.398.96.96 0 0 0-.523-.141.98.98 0 0 0-.527.14.937.937 0 0 0-.352.399c-.083.17-.125.367-.125.594v.066ZM10.797 10V4.387h.707V10h-.707ZM7.95 10.066c-.368 0-.683-.073-.946-.218a1.5 1.5 0 0 1-.606-.63 2.092 2.092 0 0 1-.207-.96v-.625c0-.373.073-.693.22-.961.148-.271.35-.48.609-.625.257-.149.549-.223.875-.223.333 0 .627.074.882.223.258.148.46.358.606.629.146.27.219.59.219.957v.496H6.895v.207c0 .318.09.586.27.805.179.218.44.328.78.328.16 0 .3-.022.422-.067a.863.863 0 0 0 .301-.18.538.538 0 0 0 .152-.238h.727a.984.984 0 0 1-.133.352c-.07.125-.17.243-.3.355-.13.11-.292.2-.485.27s-.42.105-.68.105ZM6.894 7.617h2.003v-.066a1.33 1.33 0 0 0-.125-.594.938.938 0 0 0-.351-.398.96.96 0 0 0-.524-.141.98.98 0 0 0-.527.14.937.937 0 0 0-.351.399 1.33 1.33 0 0 0-.125.594v.066ZM3.059 5.828c.195 0 .372.027.53.082.16.052.295.13.407.23.115.102.2.222.254.36h.035V4.387h.707V10h-.68v-.656h-.035a.745.745 0 0 1-.156.293c-.075.086-.17.16-.281.222a1.44 1.44 0 0 1-.367.145 1.645 1.645 0 0 1-.43.055c-.32 0-.599-.07-.836-.211a1.432 1.432 0 0 1-.547-.598 1.986 1.986 0 0 1-.195-.906v-.797c0-.35.066-.651.2-.906.132-.258.318-.457.558-.598.24-.143.518-.215.836-.215Zm.144.606a.976.976 0 0 0-.55.152.966.966 0 0 0-.352.43 1.657 1.657 0 0 0-.125.675v.5c0 .266.041.494.125.684.086.188.207.33.363.43.156.099.343.148.559.148.216 0 .402-.048.558-.144.16-.1.283-.237.371-.414.089-.18.133-.392.133-.637v-.711c0-.216-.047-.408-.14-.574a1.061 1.061 0 0 0-.941-.54Z" />
      </svg>
    </Key>
  )
}

function TabKey() {
  return (
    <Key name="Tab" width={64}>
      <svg viewBox="0 0 13 14" className="mt-auto mr-auto">
        <path d="M10.016 10.063c-.193 0-.37-.027-.532-.079a1.235 1.235 0 0 1-.418-.234 1.037 1.037 0 0 1-.27-.36h-.034V10h-.68V4.387h.707v2.156h.035a.83.83 0 0 1 .16-.29.998.998 0 0 1 .27-.226c.107-.065.226-.114.36-.148.135-.034.275-.05.421-.05.323 0 .602.07.836.214.237.143.42.346.547.61.128.262.191.575.191.937v.707c0 .364-.065.678-.195.941a1.407 1.407 0 0 1-.55.61c-.238.143-.52.214-.848.214Zm-.137-.606c.216 0 .4-.05.55-.148a.932.932 0 0 0 .348-.438c.081-.193.121-.424.121-.695v-.438c0-.276-.041-.51-.125-.703a.95.95 0 0 0-.351-.445.98.98 0 0 0-.563-.156c-.218 0-.408.05-.57.148A.97.97 0 0 0 8.918 7a1.508 1.508 0 0 0-.129.648v.676c0 .219.047.414.14.586.094.172.223.306.387.402.164.097.352.145.563.145ZM4.82 10.066c-.398 0-.722-.114-.972-.343-.248-.232-.371-.538-.371-.918 0-.36.122-.65.367-.871.247-.224.607-.336 1.078-.336h1.074V7.19c0-.263-.078-.457-.234-.582-.154-.125-.362-.187-.625-.187-.175 0-.318.02-.43.062a.623.623 0 0 0-.258.172.717.717 0 0 0-.133.254h-.683c.015-.174.06-.328.133-.46.075-.136.178-.25.308-.34.13-.094.287-.165.469-.212.182-.049.388-.074.617-.074.284 0 .542.044.774.133.234.089.421.233.562.434.14.2.211.468.211.804V10h-.68v-.547h-.03c-.069.11-.156.211-.263.305-.104.094-.231.17-.382.226a1.559 1.559 0 0 1-.532.082Zm.196-.593c.187 0 .354-.038.5-.114a.94.94 0 0 0 .347-.312.798.798 0 0 0 .133-.453v-.457H4.98c-.278 0-.479.06-.601.183a.632.632 0 0 0-.18.461c0 .237.082.412.246.524a.999.999 0 0 0 .57.168ZM.926 5.902v-1.03h.699v1.03h.93v.575h-.93v2.367c0 .242.048.405.145.488.096.08.27.121.52.121.049 0 .107-.002.175-.008l.148-.011V10a4.385 4.385 0 0 1-.504.035c-.325 0-.574-.042-.746-.125a.684.684 0 0 1-.347-.39 2.046 2.046 0 0 1-.094-.672V6.477H.293v-.575h.633Z" />
      </svg>
    </Key>
  )
}

function QKey() {
  return (
    <Key name="Q">
      <svg viewBox="0 0 10 14">
        <path d="M8.51 13.038H7.124l-.84-1.155c-.238.105-.49.182-.76.232-.264.05-.544.075-.84.075-.775 0-1.445-.168-2.01-.505a3.36 3.36 0 0 1-1.299-1.443c-.3-.629-.451-1.374-.451-2.235V6.66c0-.861.15-1.604.451-2.228.305-.63.738-1.112 1.299-1.45.565-.337 1.235-.505 2.01-.505.779 0 1.449.168 2.01.505.564.338.997.82 1.298 1.45.301.624.451 1.367.451 2.228v1.347c0 .697-.102 1.324-.307 1.88a3.41 3.41 0 0 1-.896 1.38l1.272 1.771ZM2.175 8.014c0 .642.1 1.194.3 1.654.206.46.495.813.869 1.06.373.246.818.371 1.333.376.168 0 .33-.014.485-.041.155-.032.3-.076.438-.13L4.354 9.21h1.381l.383.533c.132.178.262.356.39.533.223-.268.392-.592.506-.97.118-.383.177-.814.177-1.292V6.66c0-.638-.102-1.187-.307-1.647-.2-.46-.488-.811-.862-1.053-.373-.246-.82-.37-1.34-.37-.514 0-.961.124-1.34.37-.373.242-.662.592-.867 1.053-.2.46-.301 1.01-.301 1.647v1.354Z" />
      </svg>
    </Key>
  )
}

function WKey() {
  return (
    <Key name="W">
      <svg viewBox="0 0 12 14">
        <path d="M3.958 11.999H2.837L.273 2.668h1.45l1.695 7.424h.068l1.826-7.424h1.23l1.832 7.424h.068l1.696-7.424h1.442l-2.557 9.331H7.896l-1.942-7.3H5.9l-1.941 7.3Z" />
      </svg>
    </Key>
  )
}

function EKey() {
  return (
    <Key name="E">
      <svg viewBox="0 0 8 14">
        <path d="M6.932 10.892v1.107h-5.64V2.668h5.64v1.107H2.543v2.926h4.143v1.08H2.543v3.11h4.389Z" />
      </svg>
    </Key>
  )
}

function RKey() {
  return (
    <Key name="R">
      <svg viewBox="0 0 9 14">
        <path d="M4.696 2.668c.625 0 1.162.116 1.614.349.45.227.8.551 1.045.97.247.42.37.912.37 1.477 0 .428-.073.816-.22 1.162a2.61 2.61 0 0 1-1.517 1.442L7.888 12H6.46L4.744 8.308H2.543v3.691H1.292V2.668h3.404Zm-2.153 4.56h1.941c.63 0 1.11-.144 1.443-.431.332-.287.499-.709.499-1.265 0-.565-.169-1-.506-1.305-.333-.31-.795-.465-1.388-.465H2.543v3.466Z" />
      </svg>
    </Key>
  )
}

function TKey() {
  return (
    <Key name="T">
      <svg viewBox="0 0 8 14">
        <path d="M4.396 11.999H3.138V3.782H.32V2.668h6.884v1.114h-2.81V12Z" />
      </svg>
    </Key>
  )
}

function YKey() {
  return (
    <Key name="Y">
      <svg viewBox="0 0 8 14">
        <path d="M4.553 11.999H3.295V8.027L.178 2.668h1.428l2.29 4.122h.062l2.297-4.122H7.67L4.553 8.021v3.978Z" />
      </svg>
    </Key>
  )
}

function UKey() {
  return (
    <Key name="U">
      <svg viewBox="0 0 10 14">
        <path d="M4.744 11.07c.702 0 1.26-.208 1.675-.623.415-.419.622-.989.622-1.709v-6.07h1.265v6.111c0 .68-.148 1.276-.445 1.791-.296.515-.71.914-1.244 1.197-.533.282-1.157.423-1.873.423-.715 0-1.34-.14-1.873-.423a3.055 3.055 0 0 1-1.244-1.197c-.296-.515-.444-1.112-.444-1.79V2.667H2.44v6.07c0 .72.208 1.29.623 1.71.419.414.98.621 1.681.621Z" />
      </svg>
    </Key>
  )
}

function IKey() {
  return (
    <Key name="I">
      <svg viewBox="0 0 4 14">
        <path d="M2.55 11.999H1.292V2.668H2.55v9.331Z" />
      </svg>
    </Key>
  )
}

function OKey() {
  return (
    <Key name="O">
      <svg viewBox="0 0 10 14">
        <path d="M7.178 6.653c0-.688-.112-1.257-.335-1.709-.219-.45-.515-.788-.889-1.011a2.393 2.393 0 0 0-1.271-.342c-.474 0-.9.114-1.279.342-.373.223-.67.56-.888 1.011-.22.452-.329 1.021-.329 1.71v1.36c0 .683.11 1.253.329 1.709.218.455.515.795.888 1.018.379.223.805.335 1.279.335a2.43 2.43 0 0 0 1.271-.335c.374-.223.67-.563.889-1.018.223-.456.335-1.026.335-1.71v-1.36Zm1.264 1.36c0 .88-.155 1.632-.464 2.257-.31.624-.748 1.1-1.313 1.428-.56.328-1.221.492-1.982.492-.766 0-1.431-.164-1.996-.492-.561-.328-.996-.804-1.306-1.428-.305-.625-.458-1.377-.458-2.256V6.66c0-.884.153-1.638.458-2.263.31-.624.745-1.1 1.306-1.428.565-.328 1.23-.492 1.996-.492.76 0 1.421.164 1.982.492a3.303 3.303 0 0 1 1.313 1.428c.31.625.464 1.379.464 2.263v1.354Z" />
      </svg>
    </Key>
  )
}

function PKey() {
  return (
    <Key name="P">
      <svg viewBox="0 0 9 14">
        <path d="M4.724 2.668c.606 0 1.132.125 1.579.376.446.246.793.588 1.039 1.025.246.438.369.946.369 1.525 0 .574-.123 1.082-.37 1.524-.245.438-.594.78-1.045 1.026-.447.246-.973.369-1.58.369H2.55v3.486H1.292V2.668h3.432Zm1.695 2.926c0-.584-.16-1.032-.479-1.347-.314-.319-.758-.478-1.333-.478H2.55v3.65h2.057c.388 0 .714-.07.978-.212.269-.146.474-.355.615-.629.146-.273.219-.601.219-.984Z" />
      </svg>
    </Key>
  )
}

function LeftSquareBracketKey() {
  return (
    <Key name="LeftSquareBracket">
      <svg viewBox="0 0 5 14">
        <path d="M1.524 7.692v.158c.388.032.68.175.875.43.2.255.301.622.301 1.1v2.086c0 .328.046.597.137.807a.916.916 0 0 0 .472.471c.218.105.515.157.888.157h.267v.964H4.06c-.534 0-.987-.077-1.36-.232a1.755 1.755 0 0 1-.862-.738c-.196-.338-.294-.782-.294-1.333V9.784c0-.474-.075-.82-.226-1.039-.145-.223-.45-.335-.916-.335V7.132c.465 0 .77-.112.916-.335.15-.223.226-.57.226-1.04V4.173c0-.552.098-.99.294-1.313.2-.328.487-.565.861-.71.374-.146.825-.22 1.354-.22h.41v.965h-.267c-.373 0-.67.05-.888.15a.92.92 0 0 0-.472.465c-.091.21-.137.48-.137.813v1.853c0 .483-.1.847-.3 1.094-.197.241-.488.382-.876.423Z" />
      </svg>
      <svg viewBox="0 0 5 14">
        <path d="M1.36 1.93h3.104v.93H2.543v10.075h1.92v.93H1.36V1.93Z" />
      </svg>
    </Key>
  )
}

function RightSquareBracketKey() {
  return (
    <Key name="RightSquareBracket">
      <svg viewBox="0 0 5 14">
        <path d="M3.343 7.85v-.158c-.378-.04-.67-.182-.875-.423-.2-.247-.301-.611-.301-1.094V4.322c0-.332-.048-.604-.144-.813a.89.89 0 0 0-.464-.465c-.22-.1-.515-.15-.89-.15H.404V1.93h.41c.53 0 .98.073 1.354.218.374.146.659.383.854.711.201.324.301.761.301 1.313v1.586c0 .47.073.816.219 1.039.15.223.458.335.923.335V8.41c-.465 0-.773.112-.923.335-.146.219-.219.565-.219 1.04v1.777c0 .551-.1.995-.3 1.333a1.72 1.72 0 0 1-.855.738c-.374.155-.825.232-1.354.232h-.41v-.964H.67c.374 0 .67-.052.889-.157a.887.887 0 0 0 .464-.471c.096-.21.144-.48.144-.807V9.38c0-.479.1-.846.3-1.1.206-.256.498-.4.876-.431Z" />
      </svg>
      <svg viewBox="0 0 5 14">
        <path d="M.403 1.93h3.104v11.935H.403v-.93h1.928V2.86H.403v-.93Z" />
      </svg>
    </Key>
  )
}

function BackSlashKey() {
  return (
    <Key name="BackSlash">
      <svg viewBox="0 0 5 14">
        <path d="M2.974 13.865H1.887V1.93h1.087v11.935Z" />
      </svg>
      <svg viewBox="0 0 5 14">
        <path d="M4.69 13.865H3.54L.178 1.93h1.141l3.37 11.935Z" />
      </svg>
    </Key>
  )
}

function CapsLockKey() {
  return (
    <Key name="CapsLock" width={72}>
      <svg viewBox="0 0 3 3" className="mt-1 mr-auto h-[3px] fill-gray-500">
        <path d="M3 1.5a1.5 1.5 0 1 1-3 0 1.5 1.5 0 0 1 3 0Z" />
      </svg>
      <svg viewBox="0 0 35 14" className="mt-auto mr-auto">
        <path d="M31.547 8.488V10h-.707V4.387h.707v3.351h.027l1.656-1.847h.817l-1.61 1.785L34.153 10h-.843l-1.391-1.89-.371.378ZM26.387 8.273v-.671c0-.36.069-.672.207-.938.14-.268.338-.475.593-.621.258-.146.563-.219.915-.219.343 0 .631.063.863.188.232.122.407.28.527.472.12.19.182.388.188.594h-.688a.887.887 0 0 0-.11-.27.668.668 0 0 0-.265-.257c-.122-.073-.29-.11-.504-.11-.333 0-.587.107-.761.32-.172.214-.258.502-.258.864v.656c0 .354.09.638.27.852.182.21.432.316.75.316.202 0 .365-.027.488-.082a.683.683 0 0 0 .285-.215.803.803 0 0 0 .133-.293h.687c-.008.2-.07.395-.188.582a1.283 1.283 0 0 1-.527.454c-.232.114-.529.171-.89.171-.352 0-.657-.073-.915-.218a1.501 1.501 0 0 1-.593-.621 2.057 2.057 0 0 1-.207-.954ZM23.625 5.824c.341 0 .645.072.91.215.268.14.48.348.633.621.154.271.23.597.23.977v.61c0 .385-.076.714-.23.987-.154.271-.365.478-.633.621-.265.141-.569.211-.91.211-.346 0-.654-.07-.922-.21a1.543 1.543 0 0 1-.629-.618c-.15-.273-.226-.604-.226-.992v-.605c0-.373.076-.694.23-.965.154-.271.365-.48.633-.63.268-.148.573-.222.914-.222Zm.004.602a1 1 0 0 0-.574.164 1.058 1.058 0 0 0-.371.45c-.086.19-.13.41-.13.66v.495c0 .256.042.478.126.668.083.19.204.339.363.446.161.106.357.16.586.16a.993.993 0 0 0 .574-.16.985.985 0 0 0 .36-.446c.083-.19.125-.412.125-.668V7.7a1.6 1.6 0 0 0-.13-.664.974.974 0 0 0-.93-.61ZM19.93 10V4.387h.707V10h-.707ZM13.828 7.07c0-.291.072-.528.215-.71.146-.186.335-.32.566-.407.235-.086.485-.129.75-.129.297 0 .555.048.774.145.221.096.393.23.515.402a1 1 0 0 1 .184.602h-.656a.576.576 0 0 0-.106-.282.67.67 0 0 0-.277-.222 1.02 1.02 0 0 0-.457-.09c-.24 0-.438.053-.594.16a.527.527 0 0 0-.23.461.45.45 0 0 0 .093.297.677.677 0 0 0 .243.176c.101.041.212.074.332.097.12.024.237.046.351.067.258.044.492.105.703.183.214.076.384.19.512.34.128.149.192.357.192.625 0 .274-.07.507-.207.7-.136.19-.327.334-.575.433a2.284 2.284 0 0 1-.86.148c-.304 0-.572-.044-.804-.132a1.21 1.21 0 0 1-.543-.387 1.005 1.005 0 0 1-.195-.625h.691c.018.122.06.228.125.316a.63.63 0 0 0 .285.207c.125.047.283.07.473.07.216 0 .39-.03.524-.09a.657.657 0 0 0 .296-.234.583.583 0 0 0 .094-.324c0-.153-.045-.27-.137-.351a.902.902 0 0 0-.37-.184 6.202 6.202 0 0 0-.52-.121 5.533 5.533 0 0 1-.7-.188 1.165 1.165 0 0 1-.5-.332c-.124-.145-.187-.352-.187-.62ZM11.23 10.063c-.187 0-.359-.027-.515-.079a1.153 1.153 0 0 1-.399-.226 1.108 1.108 0 0 1-.265-.367h-.035v1.972h-.707V5.887h.68v.656h.034a1.007 1.007 0 0 1 .454-.523c.109-.063.229-.11.359-.141.13-.034.27-.05.418-.05.323 0 .601.07.836.214.234.143.415.346.543.61.127.262.191.575.191.937v.707c0 .364-.065.678-.195.941a1.407 1.407 0 0 1-.55.61 1.62 1.62 0 0 1-.849.214Zm-.144-.606c.219 0 .404-.05.555-.152a.955.955 0 0 0 .351-.438c.08-.193.121-.426.121-.699V7.73c0-.276-.04-.51-.12-.703a.944.944 0 0 0-.356-.441 1.024 1.024 0 0 0-.57-.152c-.217 0-.404.048-.563.144a.959.959 0 0 0-.363.414c-.084.177-.125.39-.125.637v.715c0 .219.044.413.132.582a.995.995 0 0 0 .38.39c.16.094.347.141.558.141ZM6.047 10.066c-.399 0-.723-.114-.973-.343-.247-.232-.37-.538-.37-.918 0-.36.122-.65.366-.871.248-.224.607-.336 1.078-.336h1.075V7.19c0-.263-.078-.457-.235-.582-.153-.125-.362-.187-.625-.187-.174 0-.317.02-.43.062a.623.623 0 0 0-.257.172.717.717 0 0 0-.133.254h-.684c.016-.174.06-.328.133-.46.076-.136.179-.25.309-.34.13-.094.286-.165.469-.212.182-.049.388-.074.617-.074.284 0 .541.044.773.133.235.089.422.233.563.434.14.2.21.468.21.804V10h-.68v-.547h-.03c-.068.11-.155.211-.262.305-.104.094-.232.17-.383.226a1.559 1.559 0 0 1-.531.082Zm.195-.593c.188 0 .354-.038.5-.114a.94.94 0 0 0 .348-.312.798.798 0 0 0 .133-.453v-.457H6.207c-.279 0-.48.06-.602.183a.632.632 0 0 0-.18.461c0 .237.083.412.247.524a.999.999 0 0 0 .57.168ZM.496 8.273v-.671c0-.36.07-.672.207-.938.14-.268.339-.475.594-.621.258-.146.562-.219.914-.219.344 0 .631.063.863.188.232.122.408.28.528.472.12.19.182.388.187.594h-.687a.888.888 0 0 0-.11-.27.668.668 0 0 0-.265-.257c-.123-.073-.29-.11-.504-.11-.334 0-.588.107-.762.32-.172.214-.258.502-.258.864v.656c0 .354.09.638.27.852.182.21.432.316.75.316.203 0 .366-.027.488-.082a.683.683 0 0 0 .285-.215.804.804 0 0 0 .133-.293h.687c-.007.2-.07.395-.187.582a1.283 1.283 0 0 1-.527.454c-.232.114-.53.171-.891.171-.352 0-.656-.073-.914-.218a1.501 1.501 0 0 1-.594-.621 2.058 2.058 0 0 1-.207-.954Z" />
      </svg>
    </Key>
  )
}

function AKey() {
  return (
    <Key name="A">
      <svg viewBox="0 0 9 14">
        <path d="M1.654 11.999H.35L3.65 2.668h1.306l3.295 9.331H6.925l-.916-2.762H2.57L1.654 12Zm2.66-7.889h-.048L2.898 8.232h2.79L4.313 4.11Z" />
      </svg>
    </Key>
  )
}

function SKey() {
  return (
    <Key name="S">
      <svg viewBox="0 0 9 14">
        <path d="M.608 9.634H1.88c.045.223.13.424.253.601.123.178.28.33.471.458.196.128.42.228.67.301.256.069.536.103.841.103.438 0 .811-.066 1.121-.199.315-.132.556-.319.725-.56.169-.242.253-.526.253-.855 0-.373-.123-.674-.37-.902-.245-.232-.628-.415-1.148-.547l-1.62-.424a3.641 3.641 0 0 1-1.196-.52 2.36 2.36 0 0 1-.752-.826 2.35 2.35 0 0 1-.26-1.115c0-.524.141-.989.424-1.394.283-.41.668-.73 1.155-.957.492-.233 1.053-.349 1.682-.349.615 0 1.157.105 1.627.315.47.21.843.499 1.12.868.284.369.448.79.493 1.264H6.104a1.495 1.495 0 0 0-.348-.717 1.752 1.752 0 0 0-.684-.465c-.278-.114-.599-.171-.964-.171-.587 0-1.061.144-1.421.43-.36.288-.54.661-.54 1.122 0 .328.111.604.334.827.228.223.554.39.978.499l1.593.417c.547.146 1 .33 1.36.554.36.223.629.499.807.827.177.328.266.718.266 1.169 0 .565-.143 1.06-.43 1.483-.283.42-.686.745-1.21.978-.52.227-1.14.341-1.86.341-.492 0-.938-.061-1.34-.184a3.188 3.188 0 0 1-1.039-.527 2.673 2.673 0 0 1-.697-.806 2.644 2.644 0 0 1-.3-1.04Z" />
      </svg>
    </Key>
  )
}

function DKey() {
  return (
    <Key name="D">
      <svg viewBox="0 0 10 14">
        <path d="M1.292 2.668h3.035c.907 0 1.664.178 2.27.533.61.351 1.068.873 1.374 1.566.31.692.465 1.547.465 2.563 0 1.012-.155 1.866-.465 2.564-.306.692-.764 1.216-1.374 1.572-.606.355-1.363.533-2.27.533H1.292V2.668Zm1.258 1.1v7.124h1.688c.483 0 .907-.073 1.272-.22.364-.15.665-.373.902-.67.237-.295.415-.664.533-1.106.123-.443.185-.962.185-1.56 0-.792-.107-1.453-.321-1.982-.21-.528-.529-.925-.957-1.189-.429-.264-.967-.396-1.614-.396H2.55Z" />
      </svg>
    </Key>
  )
}

function FKey() {
  return (
    <Key name="F">
      <svg viewBox="0 0 8 14">
        <path d="M2.55 11.999H1.292V2.668H6.85v1.114h-4.3V6.9h3.937V8H2.55v3.999Z" />
      </svg>
    </Key>
  )
}

function GKey() {
  return (
    <Key name="G">
      <svg viewBox="0 0 10 14">
        <path d="M8.107 5.218H6.85a2.214 2.214 0 0 0-.287-.677 1.948 1.948 0 0 0-.472-.513 1.948 1.948 0 0 0-.643-.328 2.734 2.734 0 0 0-.793-.11c-.774 0-1.38.274-1.818.821-.438.542-.656 1.297-.656 2.263v1.271c0 .657.098 1.22.294 1.689.2.465.487.822.86 1.073.375.246.828.37 1.361.37.497 0 .912-.087 1.244-.26.338-.174.588-.41.752-.712.169-.305.253-.651.253-1.039v-.875H4.587V7.118h3.582V9.04c0 .451-.078.868-.232 1.251a2.82 2.82 0 0 1-.67 1.005 3.065 3.065 0 0 1-1.08.663c-.429.155-.923.232-1.484.232-.601 0-1.137-.095-1.606-.287a3.16 3.16 0 0 1-1.19-.834 3.672 3.672 0 0 1-.738-1.333c-.164-.524-.246-1.12-.246-1.79V6.673c0-.857.15-1.6.451-2.229.3-.629.731-1.114 1.292-1.456.565-.342 1.237-.512 2.017-.512.469 0 .9.066 1.292.198a3.27 3.27 0 0 1 1.032.56c.3.242.544.531.731.869.187.332.31.704.37 1.114Z" />
      </svg>
    </Key>
  )
}

function HKey() {
  return (
    <Key name="H">
      <svg viewBox="0 0 10 14">
        <path d="M8.429 11.999H7.17V7.727H2.55v4.272H1.292V2.668H2.55v3.951h4.62V2.668H8.43v9.331Z" />
      </svg>
    </Key>
  )
}

function JKey() {
  return (
    <Key name="J">
      <svg viewBox="0 0 8 14">
        <path d="M.349 9.805h1.23c.01.246.075.462.198.65.123.186.287.334.493.443.21.105.446.158.71.158.502 0 .882-.144 1.142-.431.26-.292.39-.71.39-1.258V2.668h1.264v6.597c0 .934-.246 1.656-.738 2.167-.492.505-1.178.758-2.058.758-.4 0-.765-.059-1.093-.177a2.31 2.31 0 0 1-.827-.492 2.206 2.206 0 0 1-.527-.76 2.437 2.437 0 0 1-.184-.956Z" />
      </svg>
    </Key>
  )
}

function KKey() {
  return (
    <Key name="K">
      <svg viewBox="0 0 9 14">
        <path d="M8.08 2.668 4.614 6.646 8.258 12H6.672L3.678 7.508 2.55 8.745V12H1.292V2.668H2.55v4.478h.054L6.61 2.668h1.47Z" />
      </svg>
    </Key>
  )
}

function LKey() {
  return (
    <Key name="L">
      <svg viewBox="0 0 8 14">
        <path d="M6.815 10.878v1.121H1.292V2.668H2.55v8.21h4.265Z" />
      </svg>
    </Key>
  )
}

function SemicolonKey() {
  return (
    <Key name="Semicolon">
      <svg viewBox="0 0 5 14">
        <path d="M1.176 4.991c0-.287.095-.524.287-.71A.953.953 0 0 1 2.153 4c.283 0 .52.093.711.28a.94.94 0 0 1 .294.711.903.903 0 0 1-.294.69.991.991 0 0 1-.71.274.965.965 0 0 1-.691-.273.913.913 0 0 1-.287-.69Zm0 4.635c0-.292.095-.529.287-.711a.953.953 0 0 1 .69-.28c.283 0 .52.093.711.28a.94.94 0 0 1 .294.711.916.916 0 0 1-.294.69.991.991 0 0 1-.71.274.965.965 0 0 1-.691-.274.927.927 0 0 1-.287-.69Z" />
      </svg>
      <svg viewBox="0 0 5 14">
        <path d="M1.183 3.991c0-.287.095-.524.287-.71A.953.953 0 0 1 2.16 3c.283 0 .52.093.711.28a.951.951 0 0 1 .287.711.913.913 0 0 1-.287.69.991.991 0 0 1-.71.274.965.965 0 0 1-.691-.273.913.913 0 0 1-.287-.69Zm.758 7.916H.827l.615-3.965h1.45l-.95 3.965Z" />
      </svg>
    </Key>
  )
}

function SingleQuoteKey() {
  return (
    <Key name="SingleQuote">
      <svg viewBox="0 0 7 14">
        <path d="M1.955 4H3.07l-.615 3.965h-1.45L1.956 4ZM4.86 4h1.115l-.616 3.965H3.91L4.86 4Z" />
      </svg>
      <svg viewBox="0 0 5 14">
        <path d="M2.105 2.668H3.22l-.616 3.965H1.155l.95-3.965Z" />
      </svg>
    </Key>
  )
}

function ReturnKey() {
  return (
    <Key name="Return" width={72}>
      <svg viewBox="0 0 24 14" className="mt-auto ml-auto">
        <path d="M19.945 10V5.89h.7v.747h.039a1.32 1.32 0 0 1 .18-.352c.085-.125.213-.233.382-.324.17-.091.397-.137.684-.137.414 0 .75.114 1.008.34.257.227.386.57.386 1.031V10h-.707V7.418c0-.216-.036-.396-.11-.54a.718.718 0 0 0-.304-.327.976.976 0 0 0-.476-.11c-.24 0-.44.056-.602.168a1.008 1.008 0 0 0-.355.457 1.702 1.702 0 0 0-.118.645V10h-.707ZM16.875 10V5.89h.707v.633h.035c.037-.09.096-.19.18-.296.083-.107.202-.2.355-.278.154-.08.35-.12.59-.12.044 0 .087.002.13.007.043.003.086.006.128.012v.644a.838.838 0 0 0-.145-.015 2.165 2.165 0 0 0-.187-.008c-.24 0-.44.044-.602.133a.892.892 0 0 0-.359.355c-.08.148-.121.316-.121.504V10h-.711ZM15.422 5.89V10h-.7v-.684h-.038c-.04.112-.11.226-.211.34a1.21 1.21 0 0 1-.403.293 1.43 1.43 0 0 1-.61.117c-.273 0-.516-.044-.73-.132a1.047 1.047 0 0 1-.496-.422c-.12-.196-.18-.452-.18-.77V5.891h.708v2.66c0 .226.04.406.12.539.084.13.192.223.325.277.135.055.285.082.45.082.202 0 .383-.043.542-.129.16-.088.284-.207.375-.355a.89.89 0 0 0 .14-.485V5.89h.708ZM9.3 5.902v-1.03h.7v1.03h.93v.575H10v2.367c0 .242.048.405.145.488.096.08.27.121.52.121.049 0 .107-.002.175-.008l.148-.011V10a4.382 4.382 0 0 1-.504.035c-.325 0-.574-.042-.746-.125a.684.684 0 0 1-.347-.39 2.046 2.046 0 0 1-.094-.672V6.477h-.629v-.575h.633ZM6.246 10.066c-.367 0-.682-.073-.945-.218a1.5 1.5 0 0 1-.606-.63 2.092 2.092 0 0 1-.207-.96v-.625c0-.373.073-.693.219-.961.148-.271.352-.48.61-.625.257-.149.549-.223.874-.223.334 0 .628.074.883.223.258.148.46.358.606.629.146.27.218.59.218.957v.496H5.191v.207c0 .318.09.586.27.805.18.218.44.328.781.328.159 0 .3-.022.422-.067a.863.863 0 0 0 .3-.18.538.538 0 0 0 .153-.238h.727a.984.984 0 0 1-.133.352c-.07.125-.17.243-.3.355-.131.11-.293.2-.485.27-.193.07-.42.105-.68.105ZM5.191 7.617h2.004v-.066a1.33 1.33 0 0 0-.125-.594.937.937 0 0 0-.351-.398.96.96 0 0 0-.524-.141.98.98 0 0 0-.527.14.937.937 0 0 0-.352.399 1.33 1.33 0 0 0-.125.594v.066ZM1.625 10V5.89h.707v.633h.035c.037-.09.097-.19.18-.296.083-.107.202-.2.355-.278.154-.08.35-.12.59-.12.044 0 .087.002.13.007.043.003.086.006.128.012v.644a.838.838 0 0 0-.145-.015 2.165 2.165 0 0 0-.187-.008c-.24 0-.44.044-.602.133a.891.891 0 0 0-.359.355c-.08.148-.121.316-.121.504V10h-.711Z" />
      </svg>
    </Key>
  )
}

function ShiftKey({ position }: { position: 'Left' | 'Right' }) {
  return (
    <Key name={`${position}Shift`} width={94}>
      <svg
        viewBox="0 0 18 14"
        className={clsx('mt-auto', position === 'Left' ? 'mr-auto' : 'ml-auto')}
      >
        <path d="M15.91 5.902v-1.03h.7v1.03h.93v.575h-.93v2.367c0 .242.048.405.144.488.096.08.27.121.52.121.049 0 .107-.002.175-.008l.149-.011V10a4.384 4.384 0 0 1-.504.035c-.326 0-.574-.042-.746-.125A.683.683 0 0 1 16 9.52a2.046 2.046 0 0 1-.094-.672V6.477h-.629v-.575h.633ZM12.898 10V6.477h-.691v-.575h.691v-.41c0-.396.1-.686.301-.87.2-.188.53-.282.989-.282a4.79 4.79 0 0 1 .468.023v.57a1.49 1.49 0 0 0-.168-.011 6.391 6.391 0 0 0-.199-.004c-.242 0-.418.044-.527.133-.11.086-.164.251-.164.496v.355h.953v.575h-.946V10h-.707ZM10.348 10V5.89h.707V10h-.707Zm-.149-5.375c0-.143.047-.26.14-.352a.498.498 0 0 1 .356-.136c.149 0 .271.045.367.136a.462.462 0 0 1 .145.352c0 .14-.048.257-.145.348a.513.513 0 0 1-.367.136.498.498 0 0 1-.355-.136.464.464 0 0 1-.14-.348ZM5.488 10V4.387h.707v2.25h.035c.03-.094.088-.205.176-.332.091-.13.226-.242.403-.336.177-.097.407-.145.691-.145.398 0 .728.115.988.344.263.23.395.574.395 1.035V10h-.707V7.426c0-.216-.038-.397-.114-.543a.725.725 0 0 0-.316-.332 1 1 0 0 0-.48-.11.998.998 0 0 0-.59.168 1.033 1.033 0 0 0-.36.457c-.08.193-.12.408-.12.645V10h-.708ZM1.21 7.07c0-.291.073-.528.216-.71.146-.186.334-.32.566-.407.235-.086.485-.129.75-.129.297 0 .555.048.774.145.221.096.393.23.515.402a1 1 0 0 1 .184.602h-.656a.576.576 0 0 0-.106-.282.67.67 0 0 0-.277-.222 1.02 1.02 0 0 0-.457-.09c-.24 0-.438.053-.594.16a.527.527 0 0 0-.23.461.45.45 0 0 0 .093.297.677.677 0 0 0 .242.176c.102.041.213.074.333.097.12.024.236.046.351.067.258.044.492.105.703.183.214.076.384.19.512.34.128.149.191.357.191.625 0 .274-.069.507-.207.7-.135.19-.327.334-.574.433a2.284 2.284 0 0 1-.86.148c-.304 0-.572-.044-.804-.132a1.21 1.21 0 0 1-.543-.387 1.005 1.005 0 0 1-.195-.625h.691c.018.122.06.228.125.316a.63.63 0 0 0 .285.207c.125.047.283.07.473.07.216 0 .39-.03.523-.09a.657.657 0 0 0 .297-.234.583.583 0 0 0 .094-.324c0-.153-.046-.27-.137-.351a.902.902 0 0 0-.37-.184 6.195 6.195 0 0 0-.52-.121 5.53 5.53 0 0 1-.7-.188 1.165 1.165 0 0 1-.5-.332c-.125-.145-.187-.352-.187-.62Z" />
      </svg>
    </Key>
  )
}

function ZKey() {
  return (
    <Key name="Z">
      <svg viewBox="0 0 9 14">
        <path d="M.752 11.999v-.868l4.942-7.287v-.069H.923V2.668h6.33v.868l-4.915 7.28v.076h5.031v1.107H.752Z" />
      </svg>
    </Key>
  )
}

function XKey() {
  return (
    <Key name="X">
      <svg viewBox="0 0 9 14">
        <path d="m7.882 2.668-3.083 4.71 3.021 4.621H6.31l-2.29-3.65H3.95l-2.31 3.65H.226L3.26 7.296.267 2.668h1.51l2.304 3.664h.068l2.29-3.664h1.443Z" />
      </svg>
    </Key>
  )
}

function CKey() {
  return (
    <Key name="C">
      <svg viewBox="0 0 9 14">
        <path d="M2.188 6.701v1.32c0 .647.095 1.198.287 1.654.191.455.469.804.834 1.046.364.237.806.355 1.326.355.4 0 .756-.07 1.066-.212.31-.141.56-.337.752-.588a1.77 1.77 0 0 0 .356-.875h1.244v.041c-.046.529-.224 1-.533 1.415-.306.415-.71.741-1.21.978-.497.237-1.058.355-1.682.355-.788 0-1.46-.161-2.017-.485a3.143 3.143 0 0 1-1.257-1.422c-.288-.62-.431-1.37-.431-2.249V6.687c0-.88.146-1.633.437-2.262.292-.63.716-1.11 1.272-1.443.556-.337 1.221-.505 1.996-.505.47 0 .905.07 1.306.211a3.32 3.32 0 0 1 1.06.602c.304.26.548.565.73.916.187.351.297.738.329 1.162v.041H6.809a1.937 1.937 0 0 0-.356-.936 2.026 2.026 0 0 0-.766-.643c-.31-.16-.66-.24-1.052-.24-.51 0-.948.126-1.313.377-.364.246-.645.601-.84 1.066-.197.46-.295 1.016-.295 1.668Z" />
      </svg>
    </Key>
  )
}

function VKey() {
  return (
    <Key name="V">
      <svg viewBox="0 0 9 14">
        <path d="M3.637 11.999.294 2.668h1.435l2.543 7.855h.069l2.536-7.855h1.422l-3.343 9.331h-1.32Z" />
      </svg>
    </Key>
  )
}

function BKey() {
  return (
    <Key name="B">
      <svg viewBox="0 0 9 14">
        <path d="M1.292 11.999V2.668h3.541c.798 0 1.431.205 1.9.615.47.41.705.97.705 1.682 0 .35-.06.665-.178.943-.119.274-.29.504-.513.69a2.263 2.263 0 0 1-.813.424v.055c.414.055.772.18 1.073.376.305.196.54.458.704.786.169.324.253.704.253 1.142 0 .547-.132 1.016-.397 1.408-.264.387-.64.686-1.128.896-.487.21-1.064.314-1.729.314H1.292Zm1.251-8.25v2.925h1.408c.515 0 .934-.053 1.258-.157.328-.11.57-.274.725-.493.16-.223.239-.505.239-.847 0-.451-.137-.802-.41-1.053-.274-.25-.657-.376-1.149-.376H2.543Zm0 3.978v3.192h1.6c.902 0 1.55-.12 1.941-.362.396-.247.595-.638.595-1.176 0-.351-.075-.65-.226-.896a1.416 1.416 0 0 0-.663-.56c-.287-.132-.636-.198-1.046-.198H2.543Z" />
      </svg>
    </Key>
  )
}

function NKey() {
  return (
    <Key name="N">
      <svg viewBox="0 0 10 14">
        <path d="M2.53 11.999H1.291V2.668h1.19l4.614 7.178h.061V2.668h1.238v9.331h-1.19L2.577 4.814H2.53V12Z" />
      </svg>
    </Key>
  )
}

function MKey() {
  return (
    <Key name="M">
      <svg viewBox="0 0 12 14">
        <path d="M2.46 11.999H1.32V2.668h1.237l3.042 7.15h.04l3.036-7.15h1.23v9.331H8.764V5.218h-.069l-2.57 5.96H5.1l-2.578-5.96h-.061v6.781Z" />
      </svg>
    </Key>
  )
}

function CommaKey() {
  return (
    <Key name="Comma">
      <svg viewBox="0 0 8 14">
        <path d="M.978 8.993V8.01l5.892-3.52V5.8L2.297 8.426v.061l4.573 2.489v1.298L.978 8.994Z" />
      </svg>
      <svg viewBox="0 0 4 14">
        <path d="M1.825 10.965H.711L1.326 7h1.45l-.95 3.965Z" />
      </svg>
    </Key>
  )
}

function PeriodKey() {
  return (
    <Key name="Period">
      <svg viewBox="0 0 8 14">
        <path d="M6.87 8.009v.984L.984 12.274v-1.298l4.574-2.489v-.061L.984 5.8V4.488l5.886 3.52Z" />
      </svg>
      <svg viewBox="0 0 5 14">
        <path d="M1.06 8.991c0-.291.095-.528.287-.71A.953.953 0 0 1 2.037 8c.283 0 .52.093.711.28a.951.951 0 0 1 .287.711.927.927 0 0 1-.287.69.991.991 0 0 1-.71.274.965.965 0 0 1-.691-.273.927.927 0 0 1-.287-.69Z" />
      </svg>
    </Key>
  )
}

function ForwardSlashKey() {
  return (
    <Key name="ForwardSlash">
      <svg viewBox="0 0 7 14">
        <path d="M2.317 11.179c0-.278.094-.508.28-.69a.95.95 0 0 1 .691-.274c.292 0 .531.091.718.273a.913.913 0 0 1 .287.69.897.897 0 0 1-.287.677c-.187.178-.426.267-.718.267a.964.964 0 0 1-.69-.267.908.908 0 0 1-.28-.676ZM.622 5.129C.618 4.619.7 4.192.868 3.85c.169-.347.394-.62.677-.82.283-.206.59-.354.923-.445a3.838 3.838 0 0 1 1.005-.137c.47 0 .913.087 1.333.26.424.169.768.426 1.032.772.264.347.396.787.396 1.32 0 .378-.066.702-.198.97a2.552 2.552 0 0 1-.499.69c-.2.197-.41.381-.629.555-.26.21-.47.39-.629.54-.155.146-.269.3-.341.465-.069.164-.103.378-.103.642v.41H2.639L2.632 8.5a1.844 1.844 0 0 1 .178-.861 2.46 2.46 0 0 1 .485-.663c.2-.196.412-.383.636-.56.319-.256.572-.489.759-.698.186-.21.28-.492.28-.848 0-.328-.075-.592-.226-.793a1.248 1.248 0 0 0-.574-.437 1.956 1.956 0 0 0-.752-.144c-.273 0-.513.043-.718.13-.2.082-.367.198-.499.349a1.47 1.47 0 0 0-.3.52 1.917 1.917 0 0 0-.103.635H.622Z" />
      </svg>
      <svg viewBox="0 0 5 14">
        <path d="M1.32 13.865H.177L3.54 1.93h1.148L1.32 13.865Z" />
      </svg>
    </Key>
  )
}

function FunctionKey() {
  return (
    <Key name="Function" className="rounded-bl-xl">
      <svg viewBox="0 0 8 14" className="ml-auto">
        <path d="M3.945 10V5.89h.7v.747h.039c.033-.11.093-.227.18-.352.085-.125.213-.233.382-.324.17-.091.397-.137.684-.137.414 0 .75.114 1.008.34.257.227.386.57.386 1.031V10h-.707V7.418c0-.216-.036-.396-.11-.54a.718.718 0 0 0-.304-.327.976.976 0 0 0-.476-.11c-.24 0-.44.056-.602.168a1.007 1.007 0 0 0-.355.457 1.701 1.701 0 0 0-.118.645V10h-.707ZM1.164 10V6.477H.473v-.575h.691v-.41c0-.396.1-.686.3-.87.201-.188.53-.282.99-.282a4.79 4.79 0 0 1 .468.023v.57a1.492 1.492 0 0 0-.168-.011 6.39 6.39 0 0 0-.2-.004c-.241 0-.417.044-.527.133-.109.086-.164.251-.164.496v.355h.953v.575h-.945V10h-.707Z" />
      </svg>
      <svg viewBox="0 0 11 14" className="mt-auto mr-auto mb-0.5 ml-0.5">
        <path d="M5.014 10.979c-.29 0-.564-.1-.822-.299-.255-.199-.479-.477-.672-.835a5.007 5.007 0 0 1-.457-1.261A7.208 7.208 0 0 1 2.9 7.006c0-.568.055-1.095.163-1.582A5 5 0 0 1 3.52 4.16c.193-.36.417-.64.672-.84.255-.199.53-.298.822-.298.29 0 .563.1.818.298.254.2.479.48.672.84.196.357.349.779.457 1.265.111.487.167 1.014.167 1.582 0 .569-.056 1.095-.167 1.578a5.007 5.007 0 0 1-.457 1.261c-.193.358-.418.636-.672.835-.255.2-.528.299-.818.299Zm0-7.431c-.202 0-.395.092-.58.277a2.508 2.508 0 0 0-.492.76c-.144.32-.258.687-.343 1.103a6.806 6.806 0 0 0-.123 1.318c0 .454.041.89.123 1.306.085.413.2.78.343 1.103.143.322.308.575.492.76.185.184.378.277.58.277.202 0 .394-.093.576-.277.184-.185.348-.438.492-.76.144-.323.256-.69.338-1.103a6.51 6.51 0 0 0 .128-1.306c0-.463-.043-.902-.128-1.318a5.094 5.094 0 0 0-.338-1.103 2.508 2.508 0 0 0-.492-.76c-.182-.185-.374-.277-.576-.277Zm-.285-.47h.566V10.9H4.73V3.078ZM5.014 8.5c.43 0 .838.037 1.222.11.387.07.734.172 1.041.304.308.128.557.282.747.461l-.43.36c-.273-.214-.63-.378-1.073-.492a5.867 5.867 0 0 0-1.507-.176c-.565 0-1.07.059-1.512.176-.442.114-.8.278-1.072.492L2 9.375c.193-.179.442-.333.747-.461a4.839 4.839 0 0 1 1.041-.304c.387-.073.796-.11 1.226-.11Zm3.819-1.784v.563H1.191v-.563h7.642ZM5.014 5.512c-.43 0-.84-.035-1.226-.105a4.839 4.839 0 0 1-1.041-.303A2.628 2.628 0 0 1 2 4.638l.43-.356c.273.214.63.378 1.072.492.443.114.947.171 1.512.171a6.01 6.01 0 0 0 1.507-.171c.443-.114.8-.278 1.073-.492l.43.356c-.19.178-.439.334-.747.466a4.839 4.839 0 0 1-1.041.303c-.384.07-.791.105-1.222.105Zm0 5.757a4.078 4.078 0 0 1-1.652-.338 4.42 4.42 0 0 1-2.285-2.28 4.123 4.123 0 0 1-.334-1.653c0-.583.111-1.133.334-1.648a4.343 4.343 0 0 1 2.285-2.286A4.112 4.112 0 0 1 5.01 2.73c.583 0 1.132.112 1.648.334A4.343 4.343 0 0 1 8.943 5.35c.225.515.338 1.065.338 1.648 0 .585-.111 1.136-.334 1.652a4.358 4.358 0 0 1-2.285 2.28 4.068 4.068 0 0 1-1.648.339Zm0-.624c.498 0 .967-.095 1.406-.286a3.728 3.728 0 0 0 1.956-1.951 3.53 3.53 0 0 0 .286-1.41c0-.498-.096-.967-.286-1.407a3.662 3.662 0 0 0-.791-1.164 3.728 3.728 0 0 0-1.165-.791 3.51 3.51 0 0 0-1.41-.286c-.498 0-.967.095-1.406.286-.44.19-.828.454-1.165.79-.337.335-.6.723-.791 1.165-.19.44-.286.909-.286 1.407s.096.968.286 1.41a3.728 3.728 0 0 0 1.956 1.951c.442.19.912.286 1.41.286Z" />
      </svg>
    </Key>
  )
}

function ControlKey() {
  return (
    <Key name="Control">
      <svg viewBox="0 0 13 14" className="mt-0.5 ml-auto">
        <path d="m1.582 8.568 4.254-4.347a.604.604 0 0 1 .445-.2.54.54 0 0 1 .235.053.55.55 0 0 1 .199.147l4.26 4.347c.113.114.17.252.17.416a.56.56 0 0 1-.287.498.565.565 0 0 1-.293.077.573.573 0 0 1-.422-.17L5.994 5.135h.569L2.407 9.389a.554.554 0 0 1-.416.17.597.597 0 0 1-.299-.077.573.573 0 0 1-.234-.726.678.678 0 0 1 .123-.188Z" />
      </svg>
      <svg viewBox="0 0 28 14" className="mt-auto ml-auto">
        <path d="M25.746 10V4.387h.707V10h-.707ZM22.762 5.824c.34 0 .644.072.91.215.268.14.479.348.633.621.153.271.23.597.23.977v.61c0 .385-.077.714-.23.987-.154.271-.365.478-.633.621-.266.141-.57.211-.91.211-.347 0-.654-.07-.922-.21a1.543 1.543 0 0 1-.63-.618c-.15-.273-.226-.604-.226-.992v-.605c0-.373.077-.694.23-.965.155-.271.365-.48.634-.63.268-.148.573-.222.914-.222Zm.004.602a1 1 0 0 0-.575.164 1.06 1.06 0 0 0-.37.45c-.087.19-.13.41-.13.66v.495c0 .256.042.478.125.668.084.19.205.339.364.446.161.106.357.16.586.16a.993.993 0 0 0 .574-.16.985.985 0 0 0 .36-.446c.083-.19.124-.412.124-.668V7.7a1.6 1.6 0 0 0-.129-.664.974.974 0 0 0-.93-.61ZM18.121 10V5.89h.707v.633h.035c.037-.09.097-.19.18-.296.083-.107.202-.2.355-.278.154-.08.35-.12.59-.12.045 0 .088.002.13.007.044.003.086.006.128.012v.644a.838.838 0 0 0-.144-.015 2.165 2.165 0 0 0-.188-.008c-.24 0-.44.044-.602.133a.891.891 0 0 0-.359.355c-.08.148-.121.316-.121.504V10h-.71ZM15.273 5.902v-1.03h.7v1.03h.93v.575h-.93v2.367c0 .242.048.405.144.488.096.08.27.121.52.121.05 0 .108-.002.175-.008l.149-.011V10a4.384 4.384 0 0 1-.504.035c-.325 0-.574-.042-.746-.125a.684.684 0 0 1-.348-.39 2.046 2.046 0 0 1-.093-.672V6.477h-.63v-.575h.633ZM10.3 10V5.89h.7v.747h.04a1.32 1.32 0 0 1 .179-.352c.086-.125.213-.233.383-.324.169-.091.397-.137.683-.137.414 0 .75.114 1.008.34.258.227.387.57.387 1.031V10h-.707V7.418c0-.216-.037-.396-.11-.54a.718.718 0 0 0-.304-.327.976.976 0 0 0-.477-.11c-.24 0-.44.056-.602.168a1.007 1.007 0 0 0-.355.457 1.702 1.702 0 0 0-.117.645V10H10.3ZM7.34 5.824c.341 0 .644.072.91.215.268.14.48.348.633.621.153.271.23.597.23.977v.61c0 .385-.077.714-.23.987-.154.271-.365.478-.633.621-.266.141-.569.211-.91.211-.347 0-.654-.07-.922-.21a1.542 1.542 0 0 1-.629-.618c-.151-.273-.226-.604-.226-.992v-.605c0-.373.076-.694.23-.965.154-.271.365-.48.633-.63.268-.148.573-.222.914-.222Zm.004.602a1 1 0 0 0-.574.164 1.058 1.058 0 0 0-.372.45c-.085.19-.128.41-.128.66v.495c0 .256.041.478.125.668.083.19.204.339.363.446.161.106.357.16.586.16a.994.994 0 0 0 .574-.16.986.986 0 0 0 .36-.446c.083-.19.124-.412.124-.668V7.7c0-.252-.043-.474-.129-.664a.974.974 0 0 0-.93-.61ZM1.32 8.273v-.671c0-.36.07-.672.207-.938.141-.268.339-.475.594-.621.258-.146.563-.219.914-.219.344 0 .632.063.863.188.232.122.408.28.528.472.12.19.182.388.187.594h-.687a.888.888 0 0 0-.11-.27.668.668 0 0 0-.265-.257c-.123-.073-.29-.11-.504-.11-.333 0-.587.107-.762.32-.172.214-.258.502-.258.864v.656c0 .354.09.638.27.852.182.21.432.316.75.316.203 0 .366-.027.488-.082a.683.683 0 0 0 .285-.215.804.804 0 0 0 .133-.293h.688c-.008.2-.07.395-.188.582a1.283 1.283 0 0 1-.527.454c-.232.114-.529.171-.89.171-.352 0-.657-.073-.915-.218a1.502 1.502 0 0 1-.594-.621 2.058 2.058 0 0 1-.207-.954Z" />
      </svg>
    </Key>
  )
}

function OptionKey({ position }: { position: 'Left' | 'Right' }) {
  return (
    <Key name={`${position}Option`}>
      <svg
        viewBox="0 0 12 14"
        className={clsx('mt-0.5', position === 'Left' ? 'ml-auto' : 'mr-auto')}
      >
        <path d="M4.559 3.025a.979.979 0 0 1 .914.592l2.765 6.188c.063.129.172.193.328.193h1.811c.14 0 .262.047.363.14a.452.452 0 0 1 .153.346.47.47 0 0 1-.153.352.517.517 0 0 1-.363.14H8.309c-.22 0-.403-.046-.551-.14a.967.967 0 0 1-.358-.451L4.623 4.197a.31.31 0 0 0-.305-.193H2.502a.525.525 0 0 1-.37-.14.46.46 0 0 1-.146-.346c0-.141.05-.258.147-.352a.525.525 0 0 1 .369-.14h2.057Zm5.818 0c.144 0 .266.047.363.141.098.09.147.205.147.346 0 .14-.05.258-.147.351a.516.516 0 0 1-.363.135h-2.66a.516.516 0 0 1-.363-.135.477.477 0 0 1-.141-.351c0-.14.047-.256.14-.346a.504.504 0 0 1 .364-.14h2.66Z" />
      </svg>
      <svg
        viewBox="0 0 28 14"
        className={clsx('mt-auto', position === 'Left' ? 'ml-auto' : 'mr-auto')}
      >
        <path d="M21.871 10V5.89h.7v.747h.038a1.32 1.32 0 0 1 .18-.352c.086-.125.214-.233.383-.324.17-.091.397-.137.683-.137.415 0 .75.114 1.008.34.258.227.387.57.387 1.031V10h-.707V7.418c0-.216-.037-.396-.11-.54a.718.718 0 0 0-.304-.327.976.976 0 0 0-.477-.11c-.24 0-.44.056-.601.168a1.007 1.007 0 0 0-.356.457 1.702 1.702 0 0 0-.117.645V10h-.707ZM18.91 5.824c.341 0 .645.072.91.215.268.14.48.348.633.621.154.271.23.597.23.977v.61c0 .385-.076.714-.23.987-.154.271-.364.478-.633.621-.265.141-.569.211-.91.211-.346 0-.654-.07-.922-.21a1.542 1.542 0 0 1-.629-.618c-.15-.273-.226-.604-.226-.992v-.605c0-.373.077-.694.23-.965.154-.271.365-.48.633-.63.268-.148.573-.222.914-.222Zm.004.602a1 1 0 0 0-.574.164 1.058 1.058 0 0 0-.371.45c-.086.19-.13.41-.13.66v.495c0 .256.043.478.126.668.083.19.204.339.363.446.162.106.357.16.586.16a.994.994 0 0 0 .574-.16.986.986 0 0 0 .36-.446c.083-.19.125-.412.125-.668V7.7a1.6 1.6 0 0 0-.13-.664.974.974 0 0 0-.93-.61ZM15.133 10V5.89h.707V10h-.707Zm-.149-5.375c0-.143.047-.26.141-.352a.498.498 0 0 1 .355-.136c.149 0 .271.045.368.136a.463.463 0 0 1 .144.352c0 .14-.048.257-.144.348a.513.513 0 0 1-.368.136.498.498 0 0 1-.355-.136.464.464 0 0 1-.14-.348ZM12.25 5.902v-1.03h.7v1.03h.929v.575h-.93v2.367c0 .242.048.405.145.488.096.08.27.121.52.121.049 0 .107-.002.175-.008l.149-.011V10a4.387 4.387 0 0 1-.504.035c-.326 0-.575-.042-.747-.125a.683.683 0 0 1-.347-.39 2.046 2.046 0 0 1-.094-.672V6.477h-.629v-.575h.633ZM9.234 10.063c-.187 0-.359-.027-.515-.079a1.153 1.153 0 0 1-.399-.226 1.109 1.109 0 0 1-.265-.367H8.02v1.972h-.707V5.887h.68v.656h.034a1.006 1.006 0 0 1 .453-.523c.11-.063.23-.11.36-.141.13-.034.27-.05.418-.05.323 0 .601.07.836.214.234.143.415.346.543.61.127.262.191.575.191.937v.707c0 .364-.065.678-.195.941a1.407 1.407 0 0 1-.551.61 1.62 1.62 0 0 1-.848.214Zm-.144-.606c.219 0 .403-.05.555-.152a.955.955 0 0 0 .351-.438c.08-.193.121-.426.121-.699V7.73c0-.276-.04-.51-.12-.703a.944.944 0 0 0-.356-.441 1.024 1.024 0 0 0-.57-.152c-.217 0-.404.048-.563.144a.958.958 0 0 0-.363.414c-.084.177-.125.39-.125.637v.715c0 .219.044.413.132.582a.994.994 0 0 0 .38.39c.16.094.347.141.558.141ZM4.348 5.824c.34 0 .644.072.91.215.268.14.479.348.633.621.153.271.23.597.23.977v.61c0 .385-.077.714-.23.987-.154.271-.365.478-.633.621-.266.141-.57.211-.91.211-.347 0-.654-.07-.922-.21a1.542 1.542 0 0 1-.63-.618c-.15-.273-.226-.604-.226-.992v-.605c0-.373.077-.694.23-.965.154-.271.365-.48.634-.63.268-.148.573-.222.914-.222Zm.004.602a1 1 0 0 0-.575.164 1.058 1.058 0 0 0-.37.45c-.087.19-.13.41-.13.66v.495c0 .256.042.478.125.668.084.19.205.339.364.446.161.106.356.16.586.16a.994.994 0 0 0 .574-.16.986.986 0 0 0 .36-.446c.082-.19.124-.412.124-.668V7.7c0-.252-.043-.474-.129-.664a.974.974 0 0 0-.93-.61Z" />
      </svg>
    </Key>
  )
}

function CommandKey({ position }: { position: 'Left' | 'Right' }) {
  return (
    <Key name={`${position}Command`} width={50}>
      <svg
        viewBox="0 0 12 14"
        className={clsx('mt-0.5', position === 'Left' ? 'ml-auto' : 'mr-auto')}
      >
        <path d="M3.796 6.68h-.844a1.747 1.747 0 0 1-1.517-.85 1.677 1.677 0 0 1-.235-.876c0-.322.078-.616.235-.883a1.74 1.74 0 0 1 1.517-.876 1.74 1.74 0 0 1 1.51.876c.157.267.235.561.235.883v.844h1.949v-.844c0-.322.078-.616.235-.883.156-.27.366-.484.628-.64.267-.157.563-.236.889-.236a1.74 1.74 0 0 1 1.504.876c.161.267.242.561.242.883 0 .321-.08.613-.242.876-.156.262-.368.47-.634.622a1.7 1.7 0 0 1-.87.228h-.85v1.974h.85c.317 0 .607.079.87.235.266.152.478.36.634.622.161.258.242.548.242.87 0 .326-.08.622-.242.889a1.823 1.823 0 0 1-.634.634 1.665 1.665 0 0 1-.87.235c-.326 0-.622-.078-.889-.235a1.84 1.84 0 0 1-.628-.634 1.722 1.722 0 0 1-.235-.89v-.843H4.697v.844c0 .326-.078.622-.235.889a1.823 1.823 0 0 1-.634.634 1.677 1.677 0 0 1-.876.235 1.71 1.71 0 0 1-.883-.235 1.823 1.823 0 0 1-.634-.634 1.722 1.722 0 0 1-.235-.89c0-.32.078-.61.235-.869.156-.262.368-.47.634-.622a1.71 1.71 0 0 1 .883-.235h.844V6.68Zm-.838-.876h.838V4.96a.831.831 0 0 0-.254-.61.796.796 0 0 0-.59-.253.824.824 0 0 0-.603.254.82.82 0 0 0-.248.603c0 .232.085.433.254.603.17.165.37.247.603.247Zm5.427 0a.82.82 0 0 0 .603-.247.834.834 0 0 0 .248-.603.82.82 0 0 0-.248-.603.796.796 0 0 0-.59-.254.824.824 0 0 0-.603.254.842.842 0 0 0-.248.609v.844h.838ZM4.697 8.667H6.64V6.674H4.697v1.993Zm-1.739.857a.824.824 0 0 0-.603.254.803.803 0 0 0-.254.596.842.842 0 0 0 .85.857.806.806 0 0 0 .591-.247.838.838 0 0 0 .254-.616v-.844h-.838Zm5.427 0h-.838v.844a.85.85 0 0 0 .851.863.806.806 0 0 0 .59-.247.842.842 0 0 0 .248-.61.813.813 0 0 0-.248-.596.81.81 0 0 0-.603-.254Z" />
      </svg>
      <svg
        viewBox="0 0 40 14"
        className={clsx('mt-auto', position === 'Left' ? 'ml-auto' : 'mr-auto')}
      >
        <path d="M35.672 5.828c.195 0 .372.027.531.082.159.052.294.13.406.23.115.102.2.222.254.36h.035V4.387h.707V10h-.68v-.656h-.034a.745.745 0 0 1-.157.293c-.075.086-.169.16-.28.222-.11.063-.233.111-.368.145a1.644 1.644 0 0 1-.43.055c-.32 0-.599-.07-.836-.211a1.431 1.431 0 0 1-.547-.598 1.986 1.986 0 0 1-.195-.906v-.797c0-.35.066-.651.2-.906.132-.258.318-.457.558-.598.24-.143.518-.215.836-.215Zm.144.606a.976.976 0 0 0-.55.152.966.966 0 0 0-.352.43 1.656 1.656 0 0 0-.125.675v.5c0 .266.042.494.125.684.086.188.207.33.363.43.157.099.343.148.559.148.216 0 .402-.048.559-.144.158-.1.282-.237.37-.414.09-.18.133-.392.133-.637v-.711a1.15 1.15 0 0 0-.14-.574 1.062 1.062 0 0 0-.941-.54ZM29.527 10V5.89h.7v.747h.039c.034-.11.093-.227.18-.352.085-.125.213-.233.382-.324.17-.091.397-.137.684-.137.414 0 .75.114 1.008.34.257.227.386.57.386 1.031V10H32.2V7.418c0-.216-.036-.396-.11-.54a.718.718 0 0 0-.304-.327.977.977 0 0 0-.476-.11c-.24 0-.44.056-.602.168a1.007 1.007 0 0 0-.355.457 1.7 1.7 0 0 0-.118.645V10h-.707ZM26.27 10.066c-.399 0-.723-.114-.973-.343-.248-.232-.371-.538-.371-.918 0-.36.122-.65.367-.871.247-.224.607-.336 1.078-.336h1.074V7.19c0-.263-.078-.457-.234-.582-.154-.125-.362-.187-.625-.187-.174 0-.318.02-.43.062a.624.624 0 0 0-.258.172.718.718 0 0 0-.132.254h-.684c.016-.174.06-.328.133-.46.075-.136.178-.25.308-.34.13-.094.287-.165.47-.212.181-.049.387-.074.616-.074.284 0 .542.044.774.133.234.089.422.233.562.434.14.2.211.468.211.804V10h-.68v-.547h-.03c-.068.11-.156.211-.262.305-.105.094-.232.17-.383.226a1.56 1.56 0 0 1-.531.082Zm.195-.593c.187 0 .354-.038.5-.114a.939.939 0 0 0 .348-.312.799.799 0 0 0 .132-.453v-.457H26.43c-.279 0-.48.06-.602.183a.632.632 0 0 0-.18.461c0 .237.082.412.247.524a.999.999 0 0 0 .57.168ZM18.16 10V5.89h.7v.657h.038c.032-.1.086-.203.165-.313a.96.96 0 0 1 .343-.289c.151-.08.353-.12.606-.12.208 0 .389.033.543.1.156.068.284.16.383.278.101.117.173.249.214.395h.032a1.28 1.28 0 0 1 .242-.375c.11-.12.247-.216.414-.29.166-.072.366-.109.598-.109.421 0 .752.112.992.336.24.222.36.538.36.95V10h-.708V7.309a.995.995 0 0 0-.098-.457.686.686 0 0 0-.285-.305.931.931 0 0 0-.469-.11.893.893 0 0 0-.785.46.993.993 0 0 0-.12.498V10h-.7V7.336c0-.18-.035-.337-.105-.473a.741.741 0 0 0-.301-.312.898.898 0 0 0-.465-.114.882.882 0 0 0-.461.122.819.819 0 0 0-.313.332 1.047 1.047 0 0 0-.113.496V10h-.707ZM11.16 10V5.89h.7v.657h.038c.032-.1.086-.203.165-.313a.96.96 0 0 1 .343-.289c.151-.08.353-.12.606-.12.208 0 .389.033.543.1.156.068.284.16.383.278.101.117.173.249.214.395h.032a1.28 1.28 0 0 1 .242-.375c.11-.12.247-.216.414-.29.167-.072.366-.109.598-.109.421 0 .752.112.992.336.24.222.36.538.36.95V10h-.708V7.309a.995.995 0 0 0-.098-.457.686.686 0 0 0-.285-.305.931.931 0 0 0-.469-.11.893.893 0 0 0-.785.46.993.993 0 0 0-.12.498V10h-.7V7.336c0-.18-.035-.337-.105-.473a.741.741 0 0 0-.301-.312.898.898 0 0 0-.465-.114.882.882 0 0 0-.461.122.819.819 0 0 0-.313.332 1.047 1.047 0 0 0-.113.496V10h-.707ZM8.2 5.824c.34 0 .644.072.91.215.268.14.479.348.632.621.154.271.23.597.23.977v.61c0 .385-.076.714-.23.987-.153.271-.364.478-.633.621-.265.141-.569.211-.91.211-.346 0-.653-.07-.922-.21a1.542 1.542 0 0 1-.629-.618c-.15-.273-.226-.604-.226-.992v-.605c0-.373.077-.694.23-.965.154-.271.365-.48.633-.63.268-.148.573-.222.914-.222Zm.003.602a1 1 0 0 0-.574.164 1.058 1.058 0 0 0-.371.45c-.086.19-.13.41-.13.66v.495c0 .256.043.478.126.668.083.19.204.339.363.446.162.106.357.16.586.16a.994.994 0 0 0 .574-.16.986.986 0 0 0 .36-.446c.083-.19.125-.412.125-.668V7.7c0-.252-.043-.474-.13-.664a.974.974 0 0 0-.93-.61ZM2.18 8.273v-.671c0-.36.069-.672.207-.938.14-.268.338-.475.593-.621.258-.146.563-.219.915-.219.343 0 .631.063.863.188.232.122.407.28.527.472.12.19.182.388.188.594h-.688a.888.888 0 0 0-.11-.27.668.668 0 0 0-.265-.257c-.122-.073-.29-.11-.504-.11-.333 0-.587.107-.761.32-.172.214-.258.502-.258.864v.656c0 .354.09.638.27.852.182.21.432.316.75.316.202 0 .365-.027.488-.082a.683.683 0 0 0 .285-.215.804.804 0 0 0 .133-.293H5.5c-.008.2-.07.395-.188.582a1.283 1.283 0 0 1-.527.454c-.232.114-.528.171-.89.171-.352 0-.657-.073-.915-.218a1.501 1.501 0 0 1-.593-.621 2.058 2.058 0 0 1-.207-.954Z" />
      </svg>
    </Key>
  )
}

function SpaceKey() {
  return <Key name="Space" width={212} />
}

function LeftKey() {
  return (
    <Key name="Left">
      <svg viewBox="0 0 7 14">
        <path d="M5.059 10.004c0 .096-.046.169-.137.219a.277.277 0 0 1-.267 0L1.333 8.576a.23.23 0 0 1-.116-.15.391.391 0 0 1 0-.206.255.255 0 0 1 .123-.15l3.322-1.648c.082-.04.169-.036.26.014.091.045.137.116.137.212v3.356Z" />
      </svg>
    </Key>
  )
}

function UpKey() {
  return (
    <Key name="Up" className="rounded-b-[1px]">
      <svg viewBox="0 0 7 14">
        <path d="M5.045 10.168H1.682a.224.224 0 0 1-.164-.069.35.35 0 0 1-.082-.164.263.263 0 0 1 .02-.178l1.661-3.595a.211.211 0 0 1 .144-.11.316.316 0 0 1 .205 0 .211.211 0 0 1 .143.11l1.662 3.595c.045.082.043.171-.007.267-.046.096-.119.144-.22.144Z" />
      </svg>
    </Key>
  )
}

function DownKey() {
  return (
    <Key name="Down" className="rounded-t-[1px]">
      <svg viewBox="0 0 7 14">
        <path d="M1.682 6.14h3.363a.21.21 0 0 1 .164.069.35.35 0 0 1 .082.164.263.263 0 0 1-.02.178L3.608 10.14a.234.234 0 0 1-.143.116.392.392 0 0 1-.205 0 .234.234 0 0 1-.144-.116L1.456 6.55a.29.29 0 0 1 .007-.266c.05-.096.123-.143.219-.143Z" />
      </svg>
    </Key>
  )
}

function RightKey() {
  return (
    <Key name="Right" className="rounded-br-xl">
      <svg viewBox="0 0 7 14">
        <path d="M1.668 10.004V6.648c0-.096.043-.167.13-.212a.278.278 0 0 1 .266-.014L5.387 8.07a.227.227 0 0 1 .123.15.392.392 0 0 1 0 .205.23.23 0 0 1-.116.15L2.07 10.224a.277.277 0 0 1-.266 0c-.091-.05-.137-.123-.137-.219Z" />
      </svg>
    </Key>
  )
}

export function Keyboard({ highlighted = [] }: { highlighted?: string[] }) {
  return (
    <KeyboardContext.Provider value={{ highlighted }}>
      <div aria-hidden="true" className="flex flex-col gap-2">
        <Row>
          <EscapeKey />
          <F1Key />
          <F2Key />
          <F3Key />
          <F4Key />
          <F5Key />
          <F6Key />
          <F7Key />
          <F8Key />
          <F9Key />
          <F10Key />
          <F11Key />
          <F12Key />
          <LockKey />
        </Row>
        <Row>
          <BacktickKey />
          <OneKey />
          <TwoKey />
          <ThreeKey />
          <FourKey />
          <FiveKey />
          <SixKey />
          <SevenKey />
          <EightKey />
          <NineKey />
          <ZeroKey />
          <DashKey />
          <EqualsKey />
          <DeleteKey />
        </Row>
        <Row>
          <TabKey />
          <QKey />
          <WKey />
          <EKey />
          <RKey />
          <TKey />
          <YKey />
          <UKey />
          <IKey />
          <OKey />
          <PKey />
          <LeftSquareBracketKey />
          <RightSquareBracketKey />
          <BackSlashKey />
        </Row>
        <Row>
          <CapsLockKey />
          <AKey />
          <SKey />
          <DKey />
          <FKey />
          <GKey />
          <HKey />
          <JKey />
          <KKey />
          <LKey />
          <SemicolonKey />
          <SingleQuoteKey />
          <ReturnKey />
        </Row>
        <Row>
          <ShiftKey position="Left" />
          <ZKey />
          <XKey />
          <CKey />
          <VKey />
          <BKey />
          <NKey />
          <MKey />
          <CommaKey />
          <PeriodKey />
          <ForwardSlashKey />
          <ShiftKey position="Right" />
        </Row>
        <Row>
          <FunctionKey />
          <ControlKey />
          <OptionKey position="Left" />
          <CommandKey position="Left" />
          <SpaceKey />
          <CommandKey position="Right" />
          <OptionKey position="Right" />
          <LeftKey />
          <KeyGroup>
            <UpKey />
            <DownKey />
          </KeyGroup>
          <RightKey />
        </Row>
      </div>
    </KeyboardContext.Provider>
  )
}
