<?php

use App\Http\Controllers\ProfileController;
use App\Http\Controllers\PaymentController;
use App\Http\Controllers\BookingController;
use App\Http\Controllers\WebhookController;
use Illuminate\Support\Facades\Route;
use App\Livewire\Home;
use App\Livewire\ListingDetails;
use App\Livewire\ListingSearch;
use App\Livewire\Host\ListingManager;
use App\Livewire\User\Bookings;
use App\Livewire\Admin\Dashboard;
use App\Livewire\Admin\RentalTypes;
use App\Livewire\Admin\Locations;
use App\Livewire\Admin\Users;
use App\Livewire\Admin\Settings;
use App\Livewire\Admin\PickupLocationsManager as AdminPickupLocationsManager;

Route::get('/', Home::class)->name('home');
Route::get('/listings', ListingSearch::class)->name('listings.search');
Route::get('/listing/{id}', ListingDetails::class)->name('listing.details');

// Legal pages - Dynamic URL slugs
Route::get(\App\Models\Setting::get('privacy_policy_slug', '/privacy-policy'), function () {
    $content = \App\Models\Setting::get('privacy_policy_content', '');
    return view('legal.privacy-policy', compact('content'));
})->name('privacy-policy');

Route::get(\App\Models\Setting::get('terms_conditions_slug', '/terms-and-conditions'), function () {
    $content = \App\Models\Setting::get('terms_conditions_content', '');
    return view('legal.terms-conditions', compact('content'));
})->name('terms-conditions');

Route::get(\App\Models\Setting::get('advertise_slug', '/advertise'), function () {
    $content = \App\Models\Setting::get('advertise_content', '');
    return view('legal.advertise', compact('content'));
})->name('advertise');

Route::get(\App\Models\Setting::get('about_us_slug', '/about-us'), function () {
    $content = \App\Models\Setting::get('about_us_content', '');
    return view('legal.about-us', compact('content'));
})->name('about-us');

// Payment routes
Route::get('/payment/success', [PaymentController::class, 'success'])->name('payment.success');
Route::post('/payment/callback', [PaymentController::class, 'callback'])->name('payment.callback');

// Universal Webhook routes - handles multiple sources
Route::post('/webhook', [WebhookController::class, 'handle'])->name('webhook.handle');
Route::post('/webhook/{source}', [WebhookController::class, 'handle'])->name('webhook.source');

Route::middleware(['auth', 'verified'])->group(function () {
    Route::get('/dashboard', function () {
        // Redirect based on user role
        if (auth()->user()->isAdmin()) {
            return redirect()->route('admin.dashboard');
        }
        if (auth()->user()->isHost()) {
            return redirect()->route('host.listings');
        }
        // Regular users go to their bookings
        return redirect()->route('user.bookings');
    })->name('dashboard');

    Route::get('/profile', [ProfileController::class, 'edit'])->name('profile.edit');
    Route::patch('/profile', [ProfileController::class, 'update'])->name('profile.update');
    Route::patch('/profile/vacation-mode', [ProfileController::class, 'updateVacationMode'])->name('profile.vacation-mode.update');
    Route::delete('/profile', [ProfileController::class, 'destroy'])->name('profile.destroy');

    // Host routes - requires verified email and host or admin role
    Route::get('/host/listings', ListingManager::class)->middleware('host.or.admin')->name('host.listings');
    Route::get('/host/bookings', \App\Livewire\Host\BookingsManager::class)->middleware('host.or.admin')->name('host.bookings');
    Route::get('/host/pickup-locations', \App\Livewire\Host\PickupLocationsManager::class)->middleware('host.or.admin')->name('host.pickup-locations');

    // User routes - for regular users
    Route::get('/my-bookings', Bookings::class)->name('user.bookings');
    Route::get('/booking/{id}/receipt', [BookingController::class, 'receipt'])->name('booking.receipt');
    Route::get('/booking/{id}/host-receipt', [BookingController::class, 'hostReceipt'])->name('booking.host-receipt');

    // Admin payout route
    Route::post('/admin/booking/{id}/payout/complete', [BookingController::class, 'completeHostPayout'])->middleware('admin')->name('admin.payout.complete');

    // Admin routes - requires admin role
    Route::middleware('admin')->prefix('admin')->name('admin.')->group(function () {
        Route::get('/', Dashboard::class)->name('dashboard');
        Route::get('/rental-types', RentalTypes::class)->name('rental-types');
        Route::get('/locations', Locations::class)->name('locations');
        Route::get('/users', Users::class)->name('users');
        Route::get('/listings', \App\Livewire\Admin\ListingsManager::class)->name('listings');
        Route::get('/settings', Settings::class)->name('settings');
        Route::get('/seo', \App\Livewire\Admin\SeoSettings::class)->name('seo');
        Route::get('/add-ons', \App\Livewire\Admin\AddOnsManager::class)->name('add-ons');
        Route::get('/amenities', \App\Livewire\Admin\AmenitiesManager::class)->name('amenities');
        Route::get('/bookings', \App\Livewire\Admin\BookingsManager::class)->name('bookings');
        Route::get('/email-templates', \App\Livewire\Admin\EmailTemplates::class)->name('email-templates');
        Route::get('/pickup-locations', AdminPickupLocationsManager::class)->name('pickup-locations');
        Route::get('/custom-pages', \App\Livewire\Admin\CustomPagesManager::class)->name('custom-pages');
        Route::get('/badges', \App\Livewire\Admin\BadgeConfigManager::class)->name('badges');

        // Clear Cache route
        Route::post('/clear-cache', function () {
            \Illuminate\Support\Facades\Artisan::call('cache:clear');
            \Illuminate\Support\Facades\Artisan::call('config:clear');
            \Illuminate\Support\Facades\Artisan::call('route:clear');
            \Illuminate\Support\Facades\Artisan::call('view:clear');

            return redirect()->back()->with('success', 'All caches cleared successfully!');
        })->name('clear-cache');
    });
});

require __DIR__.'/auth.php';

// Custom Pages - MUST BE LAST to avoid catching other routes
Route::get('/{customPage}', function ($slug) {
    $page = \App\Models\CustomPage::where('slug', $slug)->where('is_active', true)->firstOrFail();
    return view('legal.custom-page', ['page' => $page]);
})->name('custom-page');
