<div class="space-y-6">
    {{-- Availability Info Banner --}}
    @if($listing->buffer_hours > 0)
        <div class="bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg p-3">
            <div class="flex items-start gap-2">
                <svg class="w-5 h-5 text-blue-600 dark:text-blue-400 flex-shrink-0 mt-0.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                </svg>
                <div class="text-sm text-gray-700 dark:text-gray-300">
                    <p class="font-semibold">{{ $listing->buffer_hours }} hour preparation buffer between bookings</p>
                </div>
            </div>
        </div>
    @endif

    <form wire:submit="book">
        {{-- Custom Calendar & Time Slot Picker --}}
        <div class="bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 p-4" id="custom-booking-calendar">
            {{-- This will be populated by JavaScript --}}
            <div class="text-center py-8">
                <div class="animate-pulse">Loading calendar...</div>
            </div>
        </div>

        {{-- Hidden inputs for Livewire --}}
        <input type="hidden" wire:model="check_in" id="check_in_hidden">
        <input type="hidden" wire:model="check_in_hour" id="check_in_hour_hidden">
        <input type="hidden" wire:model="check_out" id="check_out_hidden">
        <input type="hidden" wire:model="check_out_hour" id="check_out_hour_hidden">

        {{-- Error Messages --}}
        @error('check_in') <div class="text-red-500 dark:text-red-400 text-sm mt-2">{{ $message }}</div> @enderror
        @error('check_out') <div class="text-red-500 dark:text-red-400 text-sm mt-2">{{ $message }}</div> @enderror

        {{-- Rest of form only shows after dates selected --}}
        <div id="booking-details" class="hidden space-y-4">
            {{-- Guests --}}
            <div>
                <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                    Guests (Max: {{ $listing->capacity }})
                </label>
                <input type="number" wire:model.live="guests" min="1" max="{{ $listing->capacity }}" value="1" required
                       class="w-full rounded-md border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-gray-100 shadow-sm focus:border-blue-500 focus:ring-blue-500">
                @error('guests') <span class="text-red-500 dark:text-red-400 text-xs">{{ $message }}</span> @enderror
            </div>

            {{-- Pickup/Dropoff Locations --}}
            @if($listing->pickupLocations && $listing->pickupLocations->count() > 0)
                <div class="border-t border-gray-200 dark:border-gray-700 pt-4">
                    <h3 class="text-sm font-semibold text-gray-700 dark:text-gray-300 mb-3">Pickup & Dropoff Locations</h3>
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                Pickup Location <span class="text-red-500">*</span>
                            </label>
                            <select wire:model="pickup_location_id" required
                                    class="w-full rounded-md border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-gray-100 shadow-sm">
                                <option value="">Select location</option>
                                @foreach($listing->pickupLocations as $location)
                                    <option value="{{ $location->id }}">{{ $location->name }}</option>
                                @endforeach
                            </select>
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                Dropoff Location <span class="text-red-500">*</span>
                            </label>
                            <select wire:model="dropoff_location_id" required
                                    class="w-full rounded-md border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-gray-100 shadow-sm">
                                <option value="">Select location</option>
                                @foreach($listing->pickupLocations as $location)
                                    <option value="{{ $location->id }}">{{ $location->name }}</option>
                                @endforeach
                            </select>
                        </div>
                    </div>
                </div>
            @endif

            {{-- Add-ons --}}
            @if($listing->addOns && $listing->addOns->count() > 0)
                <div>
                    <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-3">Optional Add-ons</label>
                    <div class="space-y-2">
                        @foreach($listing->addOns as $addOn)
                            <label class="flex items-start p-3 bg-gray-50 dark:bg-gray-700/50 rounded border cursor-pointer hover:border-blue-500">
                                <input type="checkbox" wire:model.live="selectedAddOns" value="{{ $addOn->id }}" class="rounded mt-1">
                                <div class="ml-2 flex-1">
                                    <div class="text-sm font-medium">{{ $addOn->name }}</div>
                                    <div class="text-xs text-gray-600 dark:text-gray-400">RM {{ number_format($addOn->pivot->custom_price ?? $addOn->price, 2) }}</div>
                                </div>
                            </label>
                        @endforeach
                    </div>
                </div>
            @endif

            {{-- Payment Method --}}
            <div>
                <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Payment Method</label>
                <select wire:model="payment_channel" required class="w-full rounded-md border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-gray-100 shadow-sm">
                    <option value="fpx">FPX (Online Banking)</option>
                    <option value="tng">DuitNow QR</option>
                </select>
            </div>

            {{-- Price Summary --}}
            @if($days > 0)
                <div class="bg-gray-50 dark:bg-gray-700 rounded-lg p-4">
                    <div class="flex justify-between text-sm mb-2">
                        <span>RM {{ number_format($listing->price_per_day, 2) }} x {{ $days }} days</span>
                        <span>RM {{ number_format($subtotal, 2) }}</span>
                    </div>
                    @if($discount_amount > 0)
                        <div class="flex justify-between text-sm text-green-600 mb-2">
                            <span>Discount ({{ $discount_percentage }}%)</span>
                            <span>-RM {{ number_format($discount_amount, 2) }}</span>
                        </div>
                    @endif
                    @if($addOnsTotal > 0)
                        <div class="flex justify-between text-sm text-blue-600 mb-2">
                            <span>Add-ons</span>
                            <span>+RM {{ number_format($addOnsTotal, 2) }}</span>
                        </div>
                    @endif
                    <div class="flex justify-between font-bold text-lg pt-2 border-t-2">
                        <span>Total</span>
                        <span>RM {{ number_format($total_price, 2) }}</span>
                    </div>
                </div>
            @endif

            <button type="submit" class="w-full bg-blue-600 hover:bg-blue-700 text-white font-semibold py-3 px-6 rounded-md transition">
                Book Now & Pay
            </button>
        </div>
    </form>

    {{-- Pass data to JavaScript --}}
    <script>
        console.log('=== Booking Calendar Script Loading ===');

        window.bookingData = {
            calendarData: @json($calendarData),
            bufferHours: {{ $listing->buffer_hours }},
            listingId: {{ $listing->id }}
        };

        console.log('Booking data set:', window.bookingData);

        // Custom Slot Booking Calendar
        class SlotBookingCalendar {
            constructor() {
                // Check if data is available
                if (!window.bookingData || !window.bookingData.calendarData) {
                    console.error('Booking data not available');
                    return;
                }

                this.currentMonth = new Date();
                this.currentMonth.setDate(1);
                this.selectedCheckIn = null;
                this.selectedCheckInHour = null;
                this.selectedCheckOut = null;
                this.selectedCheckOutHour = null;
                this.step = 'check-in-date'; // check-in-date, check-in-time, check-out-date, check-out-time, complete

                this.bookedPeriods = window.bookingData.calendarData.booked_periods || {};
                this.bufferPeriods = window.bookingData.calendarData.buffer_periods || {};
                this.bufferHours = window.bookingData.bufferHours || 0;

                this.init();
            }

            init() {
                this.render();
            }

            render() {
                const container = document.getElementById('custom-booking-calendar');
                if (!container) return;

                let html = '';

                // Step indicator
                html += this.renderStepIndicator();

                if (this.step === 'check-in-date' || this.step === 'check-in-time') {
                    html += '<h3 class="text-lg font-bold mb-4 text-gray-900 dark:text-white">Select Check-in Date & Time</h3>';
                }

                if (this.step === 'check-out-date' || this.step === 'check-out-time') {
                    html += '<h3 class="text-lg font-bold mb-4 text-gray-900 dark:text-white">Select Check-out Date & Time</h3>';
                }

                if (this.step.includes('-date')) {
                    html += this.renderCalendar();
                } else if (this.step.includes('-time')) {
                    html += this.renderTimeSlots();
                }

                container.innerHTML = html;
                this.attachEventListeners();
            }

            renderStepIndicator() {
                const steps = [
                    { key: 'check-in', label: 'Check-in', complete: this.selectedCheckIn && this.selectedCheckInHour },
                    { key: 'check-out', label: 'Check-out', complete: this.selectedCheckOut && this.selectedCheckOutHour }
                ];

                let html = '<div class="flex items-center justify-center gap-4 mb-6">';
                steps.forEach((step, index) => {
                    const isActive = this.step.includes(step.key);
                    const isComplete = step.complete;

                    html += `<div class="flex items-center gap-2">`;
                    html += `<div class="flex items-center justify-center w-8 h-8 rounded-full ${isComplete ? 'bg-green-500 text-white' : isActive ? 'bg-blue-600 text-white' : 'bg-gray-300 text-gray-600'} font-bold text-sm">`;
                    html += isComplete ? '✓' : (index + 1);
                    html += `</div>`;
                    html += `<span class="${isActive ? 'font-bold' : ''} text-sm">${step.label}</span>`;
                    html += `</div>`;

                    if (index < steps.length - 1) {
                        html += '<div class="w-12 h-0.5 bg-gray-300"></div>';
                    }
                });
                html += '</div>';

                // Show selected values
                if (this.selectedCheckIn) {
                    html += `<div class="mb-4 p-3 bg-green-50 dark:bg-green-900/20 rounded-lg border border-green-200">`;
                    html += `<div class="text-sm"><strong>Check-in:</strong> ${this.formatDate(this.selectedCheckIn)} at ${this.formatHour(this.selectedCheckInHour)}</div>`;
                    html += `</div>`;
                }

                if (this.selectedCheckOut) {
                    html += `<div class="mb-4 p-3 bg-purple-50 dark:bg-purple-900/20 rounded-lg border border-purple-200">`;
                    html += `<div class="text-sm"><strong>Check-out:</strong> ${this.formatDate(this.selectedCheckOut)} at ${this.formatHour(this.selectedCheckOutHour)}</div>`;
                    html += `</div>`;
                }

                return html;
            }

            renderCalendar() {
                const year = this.currentMonth.getFullYear();
                const month = this.currentMonth.getMonth();
                const monthName = this.currentMonth.toLocaleDateString('en-US', { month: 'long', year: 'numeric' });

                let html = '<div class="calendar-view">';

                // Month navigation
                html += '<div class="flex items-center justify-between mb-4">';
                html += `<button type="button" class="prev-month p-2 rounded hover:bg-gray-100 dark:hover:bg-gray-700">`;
                html += `<svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7"/></svg>`;
                html += `</button>`;
                html += `<div class="text-lg font-bold">${monthName}</div>`;
                html += `<button type="button" class="next-month p-2 rounded hover:bg-gray-100 dark:hover:bg-gray-700">`;
                html += `<svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"/></svg>`;
                html += `</button>`;
                html += '</div>';

                // Calendar grid
                html += '<div class="grid grid-cols-7 gap-2">';

                // Day headers
                ['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun'].forEach(day => {
                    html += `<div class="text-center text-xs font-bold text-gray-600 dark:text-gray-400 py-2">${day}</div>`;
                });

                // Calendar days
                const firstDay = new Date(year, month, 1);
                const lastDay = new Date(year, month + 1, 0);
                const startDay = new Date(firstDay);
                const dayOfWeek = firstDay.getDay();
                const diff = dayOfWeek === 0 ? 6 : dayOfWeek - 1;
                startDay.setDate(firstDay.getDate() - diff);

                const today = new Date();
                today.setHours(0, 0, 0, 0);

                for (let i = 0; i < 42; i++) {
                    const currentDay = new Date(startDay);
                    currentDay.setDate(startDay.getDate() + i);

                    const dateStr = this.formatDateStr(currentDay);
                    const isCurrentMonth = currentDay.getMonth() === month;
                    const isPast = currentDay < today;
                    const dayStatus = this.getDayStatus(dateStr);

                    let className = 'aspect-square rounded-lg text-sm font-medium transition-all p-2 text-center';
                    let disabled = !isCurrentMonth || isPast || dayStatus.isFullyBooked;

                    if (disabled) {
                        className += ' opacity-50 cursor-not-allowed text-gray-400';
                    } else {
                        className += ' cursor-pointer hover:ring-2 hover:ring-blue-400';

                        if (dayStatus.availableCount > 0) {
                            className += ' bg-green-100 dark:bg-green-900/20 text-green-700';
                        } else if (dayStatus.bufferCount > 0) {
                            className += ' bg-yellow-100 dark:bg-yellow-900/20 text-yellow-700';
                        }
                    }

                    if (dayStatus.isFullyBooked) {
                        className += ' bg-red-100 dark:bg-red-900/20 text-red-500 line-through';
                    }

                    html += `<button type="button" class="day-cell ${className}" data-date="${dateStr}" ${disabled ? 'disabled' : ''}>`;
                    html += `<div>${currentDay.getDate()}</div>`;
                    if (!disabled && dayStatus.availableCount > 0) {
                        html += `<div class="text-xs">${dayStatus.availableCount}h</div>`;
                    }
                    html += `</button>`;
                }

                html += '</div>'; // End grid

                // Legend
                html += '<div class="flex flex-wrap gap-3 text-xs mt-4 pt-4 border-t">';
                html += '<div class="flex items-center gap-1"><div class="w-4 h-4 bg-green-100 rounded"></div><span>Available</span></div>';
                html += '<div class="flex items-center gap-1"><div class="w-4 h-4 bg-yellow-100 rounded"></div><span>Partial</span></div>';
                html += '<div class="flex items-center gap-1"><div class="w-4 h-4 bg-red-100 rounded"></div><span>Fully Booked</span></div>';
                html += '</div>';

                html += '</div>'; // End calendar-view

                return html;
            }

            renderTimeSlots() {
                const dateStr = this.step === 'check-in-time' ? this.selectedCheckIn : this.selectedCheckOut;
                const bookedPeriods = this.bookedPeriods[dateStr] || [];
                const bufferPeriods = this.bufferPeriods[dateStr] || [];

                let html = '<div class="time-slots-view">';
                html += `<div class="text-sm text-gray-600 dark:text-gray-400 mb-4">Selected date: <strong>${this.formatDate(dateStr)}</strong></div>`;

                html += '<div class="mb-4 flex gap-2 text-xs">';
                html += '<span class="flex items-center gap-1"><span class="w-3 h-3 bg-green-500 rounded"></span>Available</span>';
                html += '<span class="flex items-center gap-1"><span class="w-3 h-3 bg-yellow-500 rounded"></span>Buffer</span>';
                html += '<span class="flex items-center gap-1"><span class="w-3 h-3 bg-red-500 rounded"></span>Booked</span>';
                html += '</div>';

                html += '<div class="grid grid-cols-4 sm:grid-cols-6 gap-2">';

                for (let hour = 0; hour < 24; hour++) {
                    const hourDateTime = new Date(dateStr + ' ' + String(hour).padStart(2, '0') + ':00:00');
                    let isBooked = false;
                    let isBuffer = false;

                    bookedPeriods.forEach(period => {
                        const start = new Date(period.start);
                        const end = new Date(period.end);
                        if (hourDateTime >= start && hourDateTime < end) {
                            isBooked = true;
                        }
                    });

                    if (!isBooked) {
                        bufferPeriods.forEach(period => {
                            const start = new Date(period.start);
                            const end = new Date(period.end);
                            if (hourDateTime >= start && hourDateTime < end) {
                                isBuffer = true;
                            }
                        });
                    }

                    const hourText = this.formatHour(hour);
                    let className = 'p-3 rounded-lg text-sm font-medium transition-all text-center';

                    if (isBooked) {
                        className += ' bg-red-500 text-white cursor-not-allowed opacity-50';
                    } else if (isBuffer) {
                        className += ' bg-yellow-500 text-black cursor-not-allowed opacity-50';
                    } else {
                        className += ' bg-green-500 text-white cursor-pointer hover:bg-green-600 hover:scale-105';
                    }

                    html += `<button type="button" class="hour-cell ${className}" data-hour="${hour}" ${(isBooked || isBuffer) ? 'disabled' : ''}>`;
                    html += hourText;
                    html += `</button>`;
                }

                html += '</div>';

                // Back button
                html += `<button type="button" class="back-button mt-4 px-4 py-2 bg-gray-200 dark:bg-gray-700 rounded-lg hover:bg-gray-300">`;
                html += `← Back to Calendar`;
                html += `</button>`;

                html += '</div>';

                return html;
            }

            attachEventListeners() {
                // Month navigation
                const prevBtn = document.querySelector('.prev-month');
                const nextBtn = document.querySelector('.next-month');

                if (prevBtn) prevBtn.addEventListener('click', () => {
                    this.currentMonth.setMonth(this.currentMonth.getMonth() - 1);
                    this.render();
                });

                if (nextBtn) nextBtn.addEventListener('click', () => {
                    this.currentMonth.setMonth(this.currentMonth.getMonth() + 1);
                    this.render();
                });

                // Day selection
                document.querySelectorAll('.day-cell').forEach(cell => {
                    cell.addEventListener('click', (e) => {
                        const dateStr = e.currentTarget.dataset.date;
                        if (this.step === 'check-in-date') {
                            this.selectedCheckIn = dateStr;
                            this.step = 'check-in-time';
                        } else if (this.step === 'check-out-date') {
                            this.selectedCheckOut = dateStr;
                            this.step = 'check-out-time';
                        }
                        this.render();
                    });
                });

                // Hour selection
                document.querySelectorAll('.hour-cell').forEach(cell => {
                    cell.addEventListener('click', (e) => {
                        const hour = parseInt(e.currentTarget.dataset.hour);
                        if (this.step === 'check-in-time') {
                            this.selectedCheckInHour = hour;
                            document.getElementById('check_in_hidden').value = this.selectedCheckIn;
                            document.getElementById('check_in_hour_hidden').value = String(hour).padStart(2, '0');
                            document.getElementById('check_in_hidden').dispatchEvent(new Event('input'));
                            document.getElementById('check_in_hour_hidden').dispatchEvent(new Event('input'));
                            this.step = 'check-out-date';
                        } else if (this.step === 'check-out-time') {
                            this.selectedCheckOutHour = hour;
                            document.getElementById('check_out_hidden').value = this.selectedCheckOut;
                            document.getElementById('check_out_hour_hidden').value = String(hour).padStart(2, '0');
                            document.getElementById('check_out_hidden').dispatchEvent(new Event('input'));
                            document.getElementById('check_out_hour_hidden').dispatchEvent(new Event('input'));
                            document.getElementById('booking-details').classList.remove('hidden');
                            this.step = 'complete';
                        }
                        this.render();
                    });
                });

                // Back button
                const backBtn = document.querySelector('.back-button');
                if (backBtn) {
                    backBtn.addEventListener('click', () => {
                        if (this.step === 'check-in-time') {
                            this.step = 'check-in-date';
                        } else if (this.step === 'check-out-time') {
                            this.step = 'check-out-date';
                        }
                        this.render();
                    });
                }
            }

            getDayStatus(dateStr) {
                const bookedPeriods = this.bookedPeriods[dateStr] || [];
                const bufferPeriods = this.bufferPeriods[dateStr] || [];

                let bookedHours = 0;
                let bufferHours = 0;
                let availableHours = 0;

                for (let hour = 0; hour < 24; hour++) {
                    const hourDateTime = new Date(dateStr + ' ' + String(hour).padStart(2, '0') + ':00:00');
                    let isBooked = false;
                    let isBuffer = false;

                    bookedPeriods.forEach(period => {
                        const start = new Date(period.start);
                        const end = new Date(period.end);
                        if (hourDateTime >= start && hourDateTime < end) {
                            isBooked = true;
                        }
                    });

                    if (!isBooked) {
                        bufferPeriods.forEach(period => {
                            const start = new Date(period.start);
                            const end = new Date(period.end);
                            if (hourDateTime >= start && hourDateTime < end) {
                                isBuffer = true;
                            }
                        });
                    }

                    if (isBooked) bookedHours++;
                    else if (isBuffer) bufferHours++;
                    else availableHours++;
                }

                return {
                    isFullyBooked: availableHours === 0 && bufferHours === 0,
                    availableCount: availableHours,
                    bufferCount: bufferHours,
                    bookedCount: bookedHours
                };
            }

            formatDateStr(date) {
                const year = date.getFullYear();
                const month = String(date.getMonth() + 1).padStart(2, '0');
                const day = String(date.getDate()).padStart(2, '0');
                return `${year}-${month}-${day}`;
            }

            formatDate(dateStr) {
                const date = new Date(dateStr);
                return date.toLocaleDateString('en-US', { month: 'long', day: 'numeric', year: 'numeric' });
            }

            formatHour(hour) {
                const h = hour === 0 ? 12 : (hour > 12 ? hour - 12 : hour);
                const period = hour < 12 ? 'AM' : 'PM';
                return `${h}:00 ${period}`;
            }
        }

        // Initialize calendar
        let calendarInstance = null;

        function initCalendar() {
            console.log('Attempting to initialize calendar...');
            console.log('Booking data:', window.bookingData);

            const container = document.getElementById('custom-booking-calendar');
            if (container) {
                console.log('Container found, creating calendar instance');
                calendarInstance = new SlotBookingCalendar();
            } else {
                console.error('Calendar container not found');
            }
        }

        // Initialize when DOM is ready
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', initCalendar);
        } else {
            // DOM already loaded
            initCalendar();
        }

        // Reinitialize on Livewire updates
        document.addEventListener('livewire:navigated', initCalendar);

        // Also try immediate initialization
        setTimeout(initCalendar, 100);
    </script>
</div>