@props(['calendarData', 'listing'])

<div x-data="slotCalendar()" x-init="init()" class="slot-calendar">
    {{-- Month Navigation --}}
    <div class="flex items-center justify-between mb-4">
        <button @click="previousMonth()" type="button"
                class="p-2 rounded-lg hover:bg-gray-100 dark:hover:bg-gray-700 transition">
            <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7"/>
            </svg>
        </button>

        <h3 class="text-lg font-semibold text-gray-900 dark:text-white" x-text="currentMonthYear"></h3>

        <button @click="nextMonth()" type="button"
                class="p-2 rounded-lg hover:bg-gray-100 dark:hover:bg-gray-700 transition">
            <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"/>
            </svg>
        </button>
    </div>

    {{-- Calendar Grid --}}
    <div class="grid grid-cols-7 gap-1 mb-4">
        {{-- Day Headers --}}
        <template x-for="day in ['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun']">
            <div class="text-center text-xs font-semibold text-gray-600 dark:text-gray-400 py-2" x-text="day"></div>
        </template>

        {{-- Calendar Days --}}
        <template x-for="day in calendarDays" :key="day.dateStr">
            <div>
                <button
                    type="button"
                    @click="selectDate(day)"
                    :disabled="!day.isCurrentMonth || day.isPast || day.isFullyBooked"
                    :class="{
                        'bg-blue-600 text-white': selectedDate === day.dateStr,
                        'bg-gray-100 dark:bg-gray-700': day.isCurrentMonth && !day.isPast && !day.isFullyBooked && selectedDate !== day.dateStr,
                        'bg-green-100 dark:bg-green-900/20 text-green-700 dark:text-green-400': day.hasAvailableSlots && selectedDate !== day.dateStr,
                        'bg-yellow-100 dark:bg-yellow-900/20 text-yellow-700 dark:text-yellow-400': day.hasBufferOnly && selectedDate !== day.dateStr,
                        'bg-red-100 dark:bg-red-900/20 text-red-400 line-through': day.isFullyBooked,
                        'text-gray-400 dark:text-gray-600': !day.isCurrentMonth || day.isPast,
                        'cursor-pointer hover:ring-2 hover:ring-blue-400': day.isCurrentMonth && !day.isPast && !day.isFullyBooked,
                        'cursor-not-allowed opacity-50': !day.isCurrentMonth || day.isPast || day.isFullyBooked
                    }"
                    class="w-full aspect-square rounded-lg text-sm font-medium transition-all p-1 relative"
                    x-text="day.day">
                </button>
                {{-- Availability indicator --}}
                <div x-show="day.availableSlotCount > 0 && day.isCurrentMonth && !day.isPast"
                     class="text-center text-xs text-green-600 dark:text-green-400 mt-0.5">
                    <span x-text="day.availableSlotCount + ' slot' + (day.availableSlotCount > 1 ? 's' : '')"></span>
                </div>
            </div>
        </template>
    </div>

    {{-- Selected Date Info --}}
    <div x-show="selectedDate" class="mb-4">
        <h4 class="text-sm font-semibold text-gray-700 dark:text-gray-300 mb-2">
            Pick a time slot for <span class="text-blue-600 dark:text-blue-400" x-text="selectedDateFormatted"></span>
        </h4>

        {{-- Time Slots --}}
        <div x-show="availableSlots.length > 0" class="grid grid-cols-2 sm:grid-cols-4 gap-2">
            <template x-for="slot in availableSlots" :key="slot.hour">
                <button
                    type="button"
                    @click="selectSlot(slot)"
                    :class="{
                        'bg-blue-600 text-white border-blue-600': selectedSlot && selectedSlot.hour === slot.hour,
                        'bg-green-50 dark:bg-green-900/20 text-green-700 dark:text-green-400 border-green-300 dark:border-green-700 hover:bg-green-100 dark:hover:bg-green-900/30': !selectedSlot || selectedSlot.hour !== slot.hour
                    }"
                    class="px-4 py-2 rounded-lg border-2 font-medium text-sm transition-all"
                    x-text="slot.label">
                </button>
            </template>
        </div>

        <div x-show="availableSlots.length === 0" class="text-center py-4 text-gray-500 dark:text-gray-400 text-sm">
            No available time slots for this date
        </div>
    </div>

    {{-- Legend --}}
    <div class="flex flex-wrap gap-3 text-xs mt-4 pt-4 border-t border-gray-200 dark:border-gray-700">
        <div class="flex items-center gap-1">
            <div class="w-4 h-4 bg-green-100 dark:bg-green-900/20 rounded border border-green-300"></div>
            <span class="text-gray-600 dark:text-gray-400">Available</span>
        </div>
        <div class="flex items-center gap-1">
            <div class="w-4 h-4 bg-yellow-100 dark:bg-yellow-900/20 rounded border border-yellow-300"></div>
            <span class="text-gray-600 dark:text-gray-400">Partial</span>
        </div>
        <div class="flex items-center gap-1">
            <div class="w-4 h-4 bg-red-100 dark:bg-red-900/20 rounded border border-red-300"></div>
            <span class="text-gray-600 dark:text-gray-400">Fully Booked</span>
        </div>
        <div class="flex items-center gap-1">
            <div class="w-4 h-4 bg-blue-600 rounded"></div>
            <span class="text-gray-600 dark:text-gray-400">Selected</span>
        </div>
    </div>
</div>

<script>
function slotCalendar() {
    return {
        currentDate: new Date(),
        selectedDate: null,
        selectedSlot: null,
        calendarDays: [],
        availableSlots: [],
        currentMonthYear: '',
        bookingData: @json($calendarData ?? []),
        bufferHours: {{ $listing->buffer_hours ?? 0 }},

        init() {
            this.renderCalendar();
        },

        previousMonth() {
            this.currentDate.setMonth(this.currentDate.getMonth() - 1);
            this.renderCalendar();
        },

        nextMonth() {
            this.currentDate.setMonth(this.currentDate.getMonth() + 1);
            this.renderCalendar();
        },

        renderCalendar() {
            const year = this.currentDate.getFullYear();
            const month = this.currentDate.getMonth();

            this.currentMonthYear = new Date(year, month, 1).toLocaleDateString('en-US', {
                month: 'long',
                year: 'numeric'
            });

            const firstDay = new Date(year, month, 1);
            const lastDay = new Date(year, month + 1, 0);

            // Adjust to start on Monday
            const startDay = new Date(firstDay);
            const dayOfWeek = firstDay.getDay();
            const diff = dayOfWeek === 0 ? 6 : dayOfWeek - 1;
            startDay.setDate(firstDay.getDate() - diff);

            // Generate 42 days (6 weeks)
            this.calendarDays = [];
            const today = new Date();
            today.setHours(0, 0, 0, 0);

            for (let i = 0; i < 42; i++) {
                const currentDay = new Date(startDay);
                currentDay.setDate(startDay.getDate() + i);

                const dateStr = this.formatDate(currentDay);
                const isCurrentMonth = currentDay.getMonth() === month;
                const isPast = currentDay < today;

                // Check booking status
                const dayStatus = this.getDayStatus(dateStr);

                this.calendarDays.push({
                    day: currentDay.getDate(),
                    dateStr: dateStr,
                    date: currentDay,
                    isCurrentMonth: isCurrentMonth,
                    isPast: isPast,
                    isFullyBooked: dayStatus.isFullyBooked,
                    hasAvailableSlots: dayStatus.availableCount > 0,
                    hasBufferOnly: dayStatus.bufferCount > 0 && dayStatus.availableCount === 0,
                    availableSlotCount: dayStatus.availableCount
                });
            }
        },

        getDayStatus(dateStr) {
            const bookedPeriods = this.bookingData.booked_periods?.[dateStr] || [];
            const bufferPeriods = this.bookingData.buffer_periods?.[dateStr] || [];

            let bookedHours = 0;
            let bufferHours = 0;
            let availableHours = 0;

            for (let hour = 0; hour < 24; hour++) {
                const hourDateTime = new Date(dateStr + ' ' + String(hour).padStart(2, '0') + ':00:00');
                let isBooked = false;
                let isBuffer = false;

                // Check booked
                bookedPeriods.forEach(period => {
                    const start = new Date(period.start);
                    const end = new Date(period.end);
                    if (hourDateTime >= start && hourDateTime < end) {
                        isBooked = true;
                    }
                });

                // Check buffer
                if (!isBooked) {
                    bufferPeriods.forEach(period => {
                        const start = new Date(period.start);
                        const end = new Date(period.end);
                        if (hourDateTime >= start && hourDateTime < end) {
                            isBuffer = true;
                        }
                    });
                }

                if (isBooked) bookedHours++;
                else if (isBuffer) bufferHours++;
                else availableHours++;
            }

            return {
                isFullyBooked: availableHours === 0 && bufferHours === 0,
                availableCount: availableHours,
                bufferCount: bufferHours,
                bookedCount: bookedHours
            };
        },

        selectDate(day) {
            if (!day.isCurrentMonth || day.isPast || day.isFullyBooked) return;

            this.selectedDate = day.dateStr;
            this.selectedDateFormatted = day.date.toLocaleDateString('en-US', {
                month: 'long',
                day: 'numeric',
                year: 'numeric'
            });

            this.loadAvailableSlots(day.dateStr);
        },

        loadAvailableSlots(dateStr) {
            const bookedPeriods = this.bookingData.booked_periods?.[dateStr] || [];
            const bufferPeriods = this.bookingData.buffer_periods?.[dateStr] || [];

            this.availableSlots = [];

            for (let hour = 0; hour < 24; hour++) {
                const hourDateTime = new Date(dateStr + ' ' + String(hour).padStart(2, '0') + ':00:00');
                let isBooked = false;
                let isBuffer = false;

                bookedPeriods.forEach(period => {
                    const start = new Date(period.start);
                    const end = new Date(period.end);
                    if (hourDateTime >= start && hourDateTime < end) {
                        isBooked = true;
                    }
                });

                if (!isBooked) {
                    bufferPeriods.forEach(period => {
                        const start = new Date(period.start);
                        const end = new Date(period.end);
                        if (hourDateTime >= start && hourDateTime < end) {
                            isBuffer = true;
                        }
                    });
                }

                if (!isBooked && !isBuffer) {
                    const hourLabel = (hour === 0 ? 12 : (hour > 12 ? hour - 12 : hour)) + ':00 ' + (hour < 12 ? 'AM' : 'PM');
                    this.availableSlots.push({
                        hour: hour,
                        label: hourLabel,
                        value: String(hour).padStart(2, '0')
                    });
                }
            }
        },

        selectSlot(slot) {
            this.selectedSlot = slot;

            // Dispatch event to parent component
            this.$dispatch('slot-selected', {
                date: this.selectedDate,
                hour: slot.value,
                label: this.selectedDateFormatted + ' at ' + slot.label
            });
        },

        formatDate(date) {
            const year = date.getFullYear();
            const month = String(date.getMonth() + 1).padStart(2, '0');
            const day = String(date.getDate()).padStart(2, '0');
            return `${year}-${month}-${day}`;
        }
    };
}
</script>
