/**
 * Custom Booking Calendar - Built from scratch for gengsewa
 * No external dependencies - pure vanilla JavaScript
 */

class CustomBookingCalendar {
    constructor() {
        this.currentStep = 1;
        this.currentMonth = new Date();
        this.checkoutMonth = null;  // Will be set when checkin is selected
        this.selectedCheckinDate = null;
        this.selectedCheckinTime = null;
        this.selectedCheckoutDate = null;
        this.selectedCheckoutTime = null;

        this.bookedPeriods = window.bookingData?.booked_periods || {};
        this.bufferPeriods = window.bookingData?.buffer_periods || {};
        this.bufferHours = window.bufferHours || 0;

        this.stepObserver = null; // Will hold MutationObserver

        this.init();
    }

    init() {
        console.log('Custom Booking Calendar initialized');
        console.log('Booking data:', window.bookingData);

        this.renderCheckinCalendar();
        this.attachEventListeners();
    }

    attachEventListeners() {
        // Check-in month navigation
        document.getElementById('prev-month-checkin')?.addEventListener('click', () => {
            this.currentMonth.setMonth(this.currentMonth.getMonth() - 1);
            this.renderCheckinCalendar();
        });

        document.getElementById('next-month-checkin')?.addEventListener('click', () => {
            this.currentMonth.setMonth(this.currentMonth.getMonth() + 1);
            this.renderCheckinCalendar();
        });

        // Checkout month navigation
        document.getElementById('prev-month-checkout')?.addEventListener('click', () => {
            if (this.checkoutMonth) {
                this.checkoutMonth.setMonth(this.checkoutMonth.getMonth() - 1);
                this.renderCheckoutCalendar();
            }
        });

        document.getElementById('next-month-checkout')?.addEventListener('click', () => {
            if (this.checkoutMonth) {
                this.checkoutMonth.setMonth(this.checkoutMonth.getMonth() + 1);
                this.renderCheckoutCalendar();
            }
        });
    }

    renderCheckinCalendar() {
        console.log('renderCheckinCalendar called, current step:', this.currentStep);

        // DEFENSIVE: Only render if we're actually on step 1
        if (this.currentStep !== 1) {
            console.log('Not on step 1, skipping check-in calendar render');
            return;
        }

        const calendar = document.getElementById('custom-checkin-calendar');
        if (!calendar) {
            console.error('Check-in calendar element not found!');
            return;
        }
        console.log('Calendar element found:', calendar);

        const year = this.currentMonth.getFullYear();
        const month = this.currentMonth.getMonth();
        console.log('Rendering calendar for:', year, month);

        // Update month/year display
        const monthDisplay = document.getElementById('month-year-checkin');
        if (monthDisplay) {
            monthDisplay.textContent = new Date(year, month).toLocaleDateString('en-US', {
                month: 'long',
                year: 'numeric'
            });
            console.log('Month display updated');
        }

        // Get first day of month and number of days
        const firstDay = new Date(year, month, 1).getDay();
        const daysInMonth = new Date(year, month + 1, 0).getDate();
        const today = new Date();
        today.setHours(0, 0, 0, 0);

        let html = '<div class="calendar-grid" style="display: grid !important; grid-template-columns: repeat(7, 1fr); gap: 4px; visibility: visible !important; opacity: 1 !important;">';

        // Day headers
        const dayNames = ['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat'];
        dayNames.forEach(day => {
            html += `<div class="calendar-day-header" style="text-align: center; font-size: 10px; color: #9ca3af; padding: 4px 0; display: block !important;">${day}</div>`;
        });

        // Empty cells before first day
        for (let i = 0; i < firstDay; i++) {
            html += '<div class="calendar-day empty" style="background: transparent; height: 50px;"></div>';
        }

        // Days of month
        for (let day = 1; day <= daysInMonth; day++) {
            const date = new Date(year, month, day);
            const dateStr = this.formatDate(date);
            const status = this.getDayStatus(dateStr);

            let classes = 'calendar-day';
            let disabled = false;
            let statusBar = '';

            // Check if past date
            if (date < today) {
                classes += ' past';
                disabled = true;
            } else if (status.isFullyBooked) {
                classes += ' fully-booked';
                disabled = true;
                statusBar = '<div class="status-bar" style="background: #ef4444;"></div>';
            } else if (status.availableCount > 0) {
                classes += ' available';
                statusBar = '<div class="status-bar" style="background: #10b981;"></div>';
            } else if (status.bufferCount > 0) {
                classes += ' buffer';
                disabled = true;
                statusBar = '<div class="status-bar" style="background: #f59e0b;"></div>';
            }

            const clickHandler = disabled ? '' : `onclick="bookingCalendar.selectCheckinDate('${dateStr}')"`;

            // Inline styles for visibility and new layout
            const baseStyle = 'display: flex !important; flex-direction: column; align-items: center; justify-content: space-between; height: 50px; padding: 6px 4px 2px 4px; border-radius: 6px; cursor: pointer; visibility: visible !important; opacity: 1 !important;';
            let cellStyle = baseStyle;

            const isDark = document.documentElement.classList.contains('dark');

            if (classes.includes('past')) {
                cellStyle += isDark
                    ? ' background: #374151; color: #6b7280; border: 1px solid #4b5563;'
                    : ' background: #f3f4f6; color: #9ca3af; border: 1px solid #e5e7eb;';
            } else {
                cellStyle += isDark
                    ? ' background: #1f2937; color: #e5e7eb; border: 1px solid #374151;'
                    : ' background: #ffffff; color: #1f2937; border: 1px solid #d1d5db;';
            }

            html += `
                <div class="${classes}" ${clickHandler} data-date="${dateStr}" style="${cellStyle}">
                    <div class="day-number" style="font-size: 16px; font-weight: 700; line-height: 1;">${day}</div>
                    ${statusBar}
                </div>
            `;
        }

        html += '</div>';
        console.log('Generated HTML length:', html.length);
        console.log('Setting calendar innerHTML...');
        calendar.innerHTML = html;
        console.log('Calendar innerHTML set successfully. Grid should now be visible.');
    }

    renderCheckoutCalendar() {
        console.log('renderCheckoutCalendar called, current step:', this.currentStep);

        // DEFENSIVE: Only render if we're on step 3
        if (this.currentStep !== 3) {
            console.log('Not on step 3, skipping checkout calendar render');
            return;
        }

        const calendar = document.getElementById('custom-checkout-calendar');
        if (!calendar) {
            console.warn('Checkout calendar element not found');
            return;
        }

        // Initialize checkout month to checkin month if not set
        if (this.selectedCheckinDate && !this.checkoutMonth) {
            this.checkoutMonth = new Date(this.selectedCheckinDate);
        }

        // Safety check - if still no checkout month, can't render
        if (!this.checkoutMonth) {
            console.warn('Checkout month not initialized');
            return;
        }

        const year = this.checkoutMonth.getFullYear();
        const month = this.checkoutMonth.getMonth();

        console.log('Rendering checkout calendar for:', year, month);

        // Update month/year display
        const monthDisplay = document.getElementById('month-year-checkout');
        if (monthDisplay) {
            monthDisplay.textContent = new Date(year, month).toLocaleDateString('en-US', {
                month: 'long',
                year: 'numeric'
            });
        }

        const firstDay = new Date(year, month, 1).getDay();
        const daysInMonth = new Date(year, month + 1, 0).getDate();
        const checkinDate = new Date(this.selectedCheckinDate);
        checkinDate.setHours(0, 0, 0, 0);

        let html = '<div class="calendar-grid" style="display: grid !important; grid-template-columns: repeat(7, 1fr); gap: 4px; visibility: visible !important; opacity: 1 !important;">';

        // Day headers
        const dayNames = ['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat'];
        dayNames.forEach(day => {
            html += `<div class="calendar-day-header" style="text-align: center; font-size: 10px; color: #9ca3af; padding: 4px 0; display: block !important;">${day}</div>`;
        });

        // Empty cells
        for (let i = 0; i < firstDay; i++) {
            html += '<div class="calendar-day empty" style="background: transparent; height: 50px;"></div>';
        }

        // Days
        for (let day = 1; day <= daysInMonth; day++) {
            const date = new Date(year, month, day);
            const dateStr = this.formatDate(date);
            const status = this.getDayStatus(dateStr);

            let classes = 'calendar-day';
            let disabled = false;
            let statusBar = '';

            // Must be after check-in date
            if (date <= checkinDate) {
                classes += ' past';
                disabled = true;
            } else if (status.isFullyBooked) {
                classes += ' fully-booked';
                disabled = true;
                statusBar = '<div class="status-bar" style="background: #ef4444;"></div>';
            } else if (status.availableCount > 0) {
                classes += ' available';
                statusBar = '<div class="status-bar" style="background: #10b981;"></div>';
            } else if (status.bufferCount > 0) {
                classes += ' buffer';
                disabled = true;
                statusBar = '<div class="status-bar" style="background: #f59e0b;"></div>';
            }

            const clickHandler = disabled ? '' : `onclick="bookingCalendar.selectCheckoutDate('${dateStr}')"`;

            // Inline styles for visibility and new layout
            const baseStyle = 'display: flex !important; flex-direction: column; align-items: center; justify-content: space-between; height: 50px; padding: 6px 4px 2px 4px; border-radius: 6px; cursor: pointer; visibility: visible !important; opacity: 1 !important;';
            let cellStyle = baseStyle;

            const isDark = document.documentElement.classList.contains('dark');

            if (classes.includes('past')) {
                cellStyle += isDark
                    ? ' background: #374151; color: #6b7280; border: 1px solid #4b5563;'
                    : ' background: #f3f4f6; color: #9ca3af; border: 1px solid #e5e7eb;';
            } else {
                cellStyle += isDark
                    ? ' background: #1f2937; color: #e5e7eb; border: 1px solid #374151;'
                    : ' background: #ffffff; color: #1f2937; border: 1px solid #d1d5db;';
            }

            html += `
                <div class="${classes}" ${clickHandler} data-date="${dateStr}" style="${cellStyle}">
                    <div class="day-number" style="font-size: 16px; font-weight: 700; line-height: 1;">${day}</div>
                    ${statusBar}
                </div>
            `;
        }

        html += '</div>';
        console.log('Checkout calendar HTML generated, length:', html.length);
        calendar.innerHTML = html;
        console.log('Checkout calendar rendered successfully');
    }

    selectCheckinDate(dateStr) {
        this.selectedCheckinDate = dateStr;

        // Update UI
        const displayDiv = document.getElementById('selected-checkin');
        const displayText = document.getElementById('checkin-date-display');

        if (displayDiv && displayText) {
            displayDiv.classList.remove('hidden');
            displayText.textContent = this.formatDateDisplay(dateStr);
        }

        // Show time slots
        this.showCheckinTimeSlots(dateStr);
        this.goToStep(2);
    }

    showCheckinTimeSlots(dateStr) {
        const container = document.getElementById('checkin-timeslots');
        const dateDisplay = document.getElementById('step2-date-display');

        if (dateDisplay) {
            dateDisplay.textContent = this.formatDateDisplay(dateStr);
        }

        const slots = this.getTimeSlots(dateStr);
        container.innerHTML = '';

        slots.forEach(slot => {
            const btn = document.createElement('button');
            btn.type = 'button';
            btn.className = 'timeslot-btn timeslot-' + slot.status;
            btn.textContent = slot.label;
            btn.disabled = slot.status !== 'available';

            if (slot.status === 'available') {
                btn.onclick = () => this.selectCheckinTime(slot.hour);
            }

            container.appendChild(btn);
        });
    }

    showCheckoutTimeSlots(dateStr) {
        const container = document.getElementById('checkout-timeslots');
        const slots = this.getTimeSlots(dateStr);

        container.innerHTML = '';

        slots.forEach(slot => {
            const btn = document.createElement('button');
            btn.type = 'button';
            btn.className = 'timeslot-btn timeslot-' + slot.status;
            btn.textContent = slot.label;
            btn.disabled = slot.status !== 'available';

            if (slot.status === 'available') {
                btn.onclick = () => this.selectCheckoutTime(slot.hour);
            }

            container.appendChild(btn);
        });
    }

    selectCheckinTime(hour) {
        this.selectedCheckinTime = hour;
        console.log('Selected check-in time:', hour, 'Date:', this.selectedCheckinDate);

        // Update Livewire properties directly
        const checkinDateStr = this.selectedCheckinDate;
        const checkinHour = String(hour).padStart(2, '0');

        // Update hidden inputs
        document.getElementById('hidden_check_in').value = checkinDateStr;
        document.getElementById('hidden_check_in_hour').value = checkinHour;

        // Trigger Livewire update using wire:model.live
        document.getElementById('hidden_check_in').dispatchEvent(new Event('input', { bubbles: true }));
        document.getElementById('hidden_check_in_hour').dispatchEvent(new Event('input', { bubbles: true }));

        // Initialize checkout month intelligently
        const checkinDate = new Date(this.selectedCheckinDate);
        const daysLeftInMonth = new Date(checkinDate.getFullYear(), checkinDate.getMonth() + 1, 0).getDate() - checkinDate.getDate();

        if (daysLeftInMonth < 7) {
            // Less than 7 days left in month, show next month
            this.checkoutMonth = new Date(checkinDate.getFullYear(), checkinDate.getMonth() + 1, 1);
        } else {
            // Enough days left, show same month
            this.checkoutMonth = new Date(this.selectedCheckinDate);
        }
        console.log('Checkout month initialized to:', this.checkoutMonth);

        // Go to step 3 first, then render the calendar
        this.goToStep(3);

        // Render checkout calendar after step transition
        setTimeout(() => {
            this.renderCheckoutCalendar();
        }, 50);
    }

    selectCheckoutDate(dateStr) {
        this.selectedCheckoutDate = dateStr;
        console.log('Selected checkout date:', dateStr);

        // Update date display
        const dateDisplay = document.getElementById('checkout-date-display');
        if (dateDisplay) {
            dateDisplay.textContent = this.formatDateDisplay(dateStr);
        }

        // Show checkout time slots
        this.showCheckoutTimeSlots(dateStr);

        // IMPORTANT: Go to step 4 BEFORE any calendar can re-render
        this.goToStep(4);
    }

    selectCheckoutTime(hour) {
        this.selectedCheckoutTime = hour;

        // Update Livewire properties
        const checkoutDate = this.selectedCheckoutDate;
        const checkoutHour = String(hour).padStart(2, '0');

        // Update hidden inputs
        document.getElementById('hidden_check_out').value = checkoutDate;
        document.getElementById('hidden_check_out_hour').value = checkoutHour;

        // Trigger Livewire update
        document.getElementById('hidden_check_out').dispatchEvent(new Event('input', { bubbles: true }));
        document.getElementById('hidden_check_out_hour').dispatchEvent(new Event('input', { bubbles: true }));

        // Wait for Livewire to calculate price, then go to step 5 (booking details)
        setTimeout(() => {
            this.goToStep(5);
            // Update summary after step transition
            setTimeout(() => {
                this.updateSummary();
            }, 100);
        }, 500);
    }

    updateSummary() {
        const checkinStr = `${this.formatDateDisplay(this.selectedCheckinDate)} at ${this.formatHour(this.selectedCheckinTime)}`;
        const checkoutStr = `${this.formatDateDisplay(this.selectedCheckoutDate)} at ${this.formatHour(this.selectedCheckoutTime)}`;

        document.getElementById('summary-checkin').textContent = checkinStr;
        document.getElementById('summary-checkout').textContent = checkoutStr;

        // Calculate duration
        const checkin = new Date(this.selectedCheckinDate + ' ' + String(this.selectedCheckinTime).padStart(2, '0') + ':00');
        const checkout = new Date(this.selectedCheckoutDate + ' ' + String(this.selectedCheckoutTime).padStart(2, '0') + ':00');
        const hours = Math.round((checkout - checkin) / (1000 * 60 * 60));
        const days = Math.ceil(hours / 24);

        document.getElementById('summary-duration').textContent = `${days} day${days > 1 ? 's' : ''} (${hours} hours)`;
    }

    getTimeSlots(dateStr) {
        const slots = [];
        const bookedPeriods = this.bookedPeriods[dateStr] || [];
        const bufferPeriods = this.bufferPeriods[dateStr] || [];

        for (let hour = 0; hour < 24; hour++) {
            const hourDateTime = new Date(dateStr + ' ' + String(hour).padStart(2, '0') + ':00:00');
            let status = 'available';

            // Check if booked
            for (let period of bookedPeriods) {
                const start = new Date(period.start);
                const end = new Date(period.end);
                if (hourDateTime >= start && hourDateTime < end) {
                    status = 'booked';
                    break;
                }
            }

            // Check if buffer (only if not booked)
            if (status === 'available') {
                for (let period of bufferPeriods) {
                    const start = new Date(period.start);
                    const end = new Date(period.end);
                    if (hourDateTime >= start && hourDateTime < end) {
                        status = 'buffer';
                        break;
                    }
                }
            }

            slots.push({
                hour: hour,
                label: this.formatHour(hour),
                status: status
            });
        }

        return slots;
    }

    getDayStatus(dateStr) {
        const bookedPeriods = this.bookedPeriods[dateStr] || [];
        const bufferPeriods = this.bufferPeriods[dateStr] || [];

        let bookedHours = 0;
        let bufferHours = 0;
        let availableHours = 0;

        for (let hour = 0; hour < 24; hour++) {
            const hourDateTime = new Date(dateStr + ' ' + String(hour).padStart(2, '0') + ':00:00');
            let isBooked = false;
            let isBuffer = false;

            for (let period of bookedPeriods) {
                const start = new Date(period.start);
                const end = new Date(period.end);
                if (hourDateTime >= start && hourDateTime < end) {
                    isBooked = true;
                    break;
                }
            }

            if (!isBooked) {
                for (let period of bufferPeriods) {
                    const start = new Date(period.start);
                    const end = new Date(period.end);
                    if (hourDateTime >= start && hourDateTime < end) {
                        isBuffer = true;
                        break;
                    }
                }
            }

            if (isBooked) bookedHours++;
            else if (isBuffer) bufferHours++;
            else availableHours++;
        }

        return {
            isFullyBooked: availableHours === 0 && bufferHours === 0,
            availableCount: availableHours,
            bufferCount: bufferHours,
            bookedCount: bookedHours
        };
    }

    goToStep(stepNumber) {
        console.log(`========================`);
        console.log(`TRANSITIONING TO STEP ${stepNumber}`);
        console.log(`========================`);

        // Update current step FIRST
        this.currentStep = stepNumber;

        // NUCLEAR OPTION: Use requestAnimationFrame to ensure DOM updates
        requestAnimationFrame(() => {
            // Hide ALL steps - SIMPLE approach
            for (let i = 1; i <= 5; i++) {
                const step = document.getElementById(`step-${i}`);
                if (step) {
                    if (i === stepNumber) {
                        step.style.cssText = 'display: block !important;';
                        step.classList.add('active');
                    } else {
                        step.style.cssText = 'display: none !important;';
                        step.classList.remove('active');
                    }
                }
            }

            console.log(`✓ Step ${stepNumber} is now active`);

            // Install MutationObserver to prevent Livewire from changing visibility
            this.enforceStepVisibility(stepNumber);
        });
    }

    enforceStepVisibility(activeStep) {
        // Disconnect previous observer if exists
        if (this.stepObserver) {
            this.stepObserver.disconnect();
        }

        // Create new observer that enforces visibility
        this.stepObserver = new MutationObserver((mutations) => {
            mutations.forEach((mutation) => {
                if (mutation.type === 'attributes' && mutation.attributeName === 'style') {
                    const stepId = mutation.target.id;
                    const stepNum = parseInt(stepId.replace('step-', ''));

                    if (stepNum === this.currentStep) {
                        // Current step MUST be visible
                        if (mutation.target.style.display !== 'block') {
                            console.warn(`⚠️ Step ${stepNum} visibility was changed! Forcing it back to visible.`);
                            mutation.target.style.cssText = 'display: block !important;';
                        }
                    } else {
                        // Other steps MUST be hidden
                        if (mutation.target.style.display !== 'none') {
                            console.warn(`⚠️ Step ${stepNum} was made visible! Forcing it back to hidden.`);
                            mutation.target.style.cssText = 'display: none !important;';
                        }
                    }
                }
            });
        });

        // Observe all step divs
        for (let i = 1; i <= 5; i++) {
            const step = document.getElementById(`step-${i}`);
            if (step) {
                this.stepObserver.observe(step, {
                    attributes: true,
                    attributeFilter: ['style', 'class']
                });
            }
        }
    }

    formatDate(date) {
        const year = date.getFullYear();
        const month = String(date.getMonth() + 1).padStart(2, '0');
        const day = String(date.getDate()).padStart(2, '0');
        return `${year}-${month}-${day}`;
    }

    formatDateDisplay(dateStr) {
        const date = new Date(dateStr + 'T00:00:00');
        return date.toLocaleDateString('en-US', {
            weekday: 'short',
            year: 'numeric',
            month: 'long',
            day: 'numeric'
        });
    }

    formatHour(hour) {
        const h = hour === 0 ? 12 : (hour > 12 ? hour - 12 : hour);
        const period = hour < 12 ? 'AM' : 'PM';
        return `${h}:00 ${period}`;
    }
}

// Initialize when DOM is ready
function initBookingCalendar() {
    console.log('Initializing booking calendar...');
    console.log('Document ready state:', document.readyState);
    window.bookingCalendar = new CustomBookingCalendar();
}

if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initBookingCalendar);
} else {
    // DOM is already loaded
    initBookingCalendar();
}

// REMOVED: This was causing the check-in calendar to re-render during checkout
// Livewire event listener removed to prevent calendar interference
