<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\User;
use App\Models\RentalType;
use App\Models\Location;
use App\Models\Listing;
use App\Models\Booking;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class SampleDataSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        echo "Starting Sample Data Seeding...\n";

        // Create Users
        echo "\n1. Creating Users...\n";
        $users = $this->createUsers();

        // Create or get Rental Types
        echo "\n2. Setting up Rental Types...\n";
        $rentalTypes = $this->createRentalTypes();

        // Create or get Locations
        echo "\n3. Setting up Locations...\n";
        $locations = $this->createLocations();

        // Create Listings with Images
        echo "\n4. Creating Listings with Images...\n";
        $listings = $this->createListings($users['host'], $rentalTypes, $locations);

        // Create Bookings
        echo "\n5. Creating Bookings...\n";
        $this->createBookings($users, $listings);

        echo "\n✓ Sample data seeding completed successfully!\n";
        echo "  - 3 Users created (admin, user, host)\n";
        echo "  - 15 Listings created with 8-12 images each\n";
        echo "  - 40 Bookings created\n";
    }

    private function createUsers(): array
    {
        $admin = User::create([
            'name' => 'Admin User',
            'email' => 'admin@gengsewa.com',
            'phone' => '+9779841234567',
            'role' => 'admin',
            'password' => Hash::make('password'),
            'email_verified_at' => now(),
        ]);
        echo "  ✓ Admin created: admin@gengsewa.com\n";

        $user = User::create([
            'name' => 'John Customer',
            'email' => 'user@gengsewa.com',
            'phone' => '+9779841234568',
            'role' => 'user',
            'password' => Hash::make('password'),
            'email_verified_at' => now(),
        ]);
        echo "  ✓ User created: user@gengsewa.com\n";

        $host = User::create([
            'name' => 'Sarah Host',
            'email' => 'host@gengsewa.com',
            'phone' => '+9779841234569',
            'role' => 'host',
            'password' => Hash::make('password'),
            'email_verified_at' => now(),
            'can_create_custom_locations' => true,
        ]);
        echo "  ✓ Host created: host@gengsewa.com\n";

        return compact('admin', 'user', 'host');
    }

    private function createRentalTypes(): array
    {
        $types = [
            ['name' => 'Car', 'slug' => 'car', 'description' => 'Rent comfortable cars for your journey'],
            ['name' => 'Houseboat', 'slug' => 'houseboat', 'description' => 'Experience luxury on water'],
            ['name' => 'Guesthouse', 'slug' => 'guesthouse', 'description' => 'Cozy accommodations for your stay'],
            ['name' => 'Boat', 'slug' => 'boat', 'description' => 'Explore waterways with our boats'],
            ['name' => 'Fishing Trip', 'slug' => 'fishing-trip', 'description' => 'Guided fishing experiences'],
        ];

        $rentalTypes = [];
        foreach ($types as $type) {
            $rentalTypes[$type['slug']] = RentalType::firstOrCreate(
                ['slug' => $type['slug']],
                $type
            );
            echo "  ✓ {$type['name']}\n";
        }

        return $rentalTypes;
    }

    private function createLocations(): array
    {
        $locationData = [
            ['name' => 'Kathmandu', 'slug' => 'kathmandu'],
            ['name' => 'Pokhara', 'slug' => 'pokhara'],
            ['name' => 'Chitwan', 'slug' => 'chitwan'],
            ['name' => 'Lumbini', 'slug' => 'lumbini'],
            ['name' => 'Phewa Lake', 'slug' => 'phewa-lake'],
        ];

        $locations = [];
        foreach ($locationData as $loc) {
            $locations[$loc['slug']] = Location::firstOrCreate(
                ['slug' => $loc['slug']],
                $loc
            );
            echo "  ✓ {$loc['name']}\n";
        }

        return $locations;
    }

    private function createListings($host, $rentalTypes, $locations): array
    {
        $listingsData = [
            // Cars (3 listings)
            [
                'type' => 'car',
                'title' => 'Toyota Hiace - Comfortable Van',
                'description' => 'Perfect for group travel with comfortable seating for up to 12 passengers. Well-maintained vehicle with air conditioning and modern amenities.',
                'price' => 8500,
                'capacity' => 12,
                'units' => 2,
                'location' => 'kathmandu',
                'amenities' => ['AC', 'GPS', 'Music System', 'First Aid Kit'],
                'images' => $this->getUnsplashImages('car suv van', 10),
            ],
            [
                'type' => 'car',
                'title' => 'Mahindra Scorpio 4WD',
                'description' => 'Rugged SUV perfect for mountain roads and off-road adventures. Equipped with 4-wheel drive and all safety features.',
                'price' => 7000,
                'capacity' => 7,
                'units' => 3,
                'location' => 'pokhara',
                'amenities' => ['4WD', 'AC', 'Roof Rack', 'Bluetooth'],
                'images' => $this->getUnsplashImages('suv 4x4 offroad', 12),
            ],
            [
                'type' => 'car',
                'title' => 'Honda City Sedan',
                'description' => 'Fuel-efficient sedan ideal for city tours and comfortable highway drives. Modern features with excellent mileage.',
                'price' => 4500,
                'capacity' => 4,
                'units' => 5,
                'location' => 'kathmandu',
                'amenities' => ['AC', 'Bluetooth', 'USB Charging', 'Backup Camera'],
                'images' => $this->getUnsplashImages('sedan car honda', 9),
            ],

            // Houseboats (3 listings)
            [
                'type' => 'houseboat',
                'title' => 'Luxury Houseboat - Phewa Lake',
                'description' => 'Experience ultimate luxury on Phewa Lake with our premium houseboat. Features 3 bedrooms, full kitchen, and spacious deck.',
                'price' => 25000,
                'capacity' => 8,
                'units' => 1,
                'location' => 'phewa-lake',
                'amenities' => ['Kitchen', '3 Bedrooms', 'WiFi', 'BBQ Deck', 'Bathroom'],
                'images' => $this->getUnsplashImages('houseboat luxury yacht', 11),
            ],
            [
                'type' => 'houseboat',
                'title' => 'Romantic Houseboat for Couples',
                'description' => 'Intimate houseboat perfect for romantic getaways. Private bedroom, small kitchen, and cozy living area with stunning lake views.',
                'price' => 12000,
                'capacity' => 2,
                'units' => 2,
                'location' => 'phewa-lake',
                'amenities' => ['King Bed', 'Kitchenette', 'Private Deck', 'Bathroom'],
                'images' => $this->getUnsplashImages('romantic boat yacht', 10),
            ],
            [
                'type' => 'houseboat',
                'title' => 'Family Houseboat Adventure',
                'description' => 'Spacious houseboat ideal for families. 2 bedrooms, full amenities, and plenty of space for kids to play safely.',
                'price' => 18000,
                'capacity' => 6,
                'units' => 2,
                'location' => 'phewa-lake',
                'amenities' => ['2 Bedrooms', 'Kitchen', 'Life Jackets', 'Play Area', 'Safety Rails'],
                'images' => $this->getUnsplashImages('family boat lake', 12),
            ],

            // Guesthouses (3 listings)
            [
                'type' => 'guesthouse',
                'title' => 'Mountain View Guesthouse',
                'description' => 'Cozy guesthouse with breathtaking mountain views. Traditional Nepali hospitality with modern comforts.',
                'price' => 3500,
                'capacity' => 4,
                'units' => 8,
                'location' => 'pokhara',
                'amenities' => ['WiFi', 'Mountain View', 'Hot Water', 'Restaurant', 'Garden'],
                'images' => $this->getUnsplashImages('guesthouse mountain resort', 10),
            ],
            [
                'type' => 'guesthouse',
                'title' => 'Heritage Guesthouse Kathmandu',
                'description' => 'Stay in a beautifully restored traditional Newari house in the heart of Kathmandu. Rich cultural experience.',
                'price' => 4000,
                'capacity' => 3,
                'units' => 5,
                'location' => 'kathmandu',
                'amenities' => ['WiFi', 'Traditional Architecture', 'Rooftop', 'Cultural Tours'],
                'images' => $this->getUnsplashImages('traditional house heritage hotel', 11),
            ],
            [
                'type' => 'guesthouse',
                'title' => 'Jungle Safari Guesthouse',
                'description' => 'Located near Chitwan National Park, perfect base for jungle safaris. Comfortable rooms with nature all around.',
                'price' => 3000,
                'capacity' => 5,
                'units' => 10,
                'location' => 'chitwan',
                'amenities' => ['WiFi', 'Safari Tours', 'Restaurant', 'Campfire', 'Bird Watching'],
                'images' => $this->getUnsplashImages('jungle lodge safari resort', 9),
            ],

            // Boats (3 listings)
            [
                'type' => 'boat',
                'title' => 'Speed Boat - Lake Adventure',
                'description' => 'Thrilling speed boat for exciting lake adventures. Licensed captain included for safe and fun rides.',
                'price' => 6000,
                'capacity' => 6,
                'units' => 3,
                'location' => 'phewa-lake',
                'amenities' => ['Life Jackets', 'Captain Included', 'Safety Equipment', 'Sound System'],
                'images' => $this->getUnsplashImages('speedboat motorboat lake', 10),
            ],
            [
                'type' => 'boat',
                'title' => 'Traditional Wooden Boat',
                'description' => 'Peaceful traditional wooden boat for serene lake experiences. Perfect for photography and relaxation.',
                'price' => 2000,
                'capacity' => 4,
                'units' => 8,
                'location' => 'phewa-lake',
                'amenities' => ['Paddles', 'Life Jackets', 'Cushioned Seats', 'Umbrella'],
                'images' => $this->getUnsplashImages('wooden boat traditional rowboat', 8),
            ],
            [
                'type' => 'boat',
                'title' => 'Party Pontoon Boat',
                'description' => 'Large pontoon boat perfect for groups and celebrations. Spacious deck with music system and shade.',
                'price' => 9000,
                'capacity' => 12,
                'units' => 2,
                'location' => 'phewa-lake',
                'amenities' => ['Sound System', 'Shade Cover', 'Cooler', 'Safety Equipment', 'Captain'],
                'images' => $this->getUnsplashImages('pontoon boat party deck', 11),
            ],

            // Fishing Trips (3 listings)
            [
                'type' => 'fishing-trip',
                'title' => 'Guided Fishing Experience - Half Day',
                'description' => 'Professional guided fishing trip with all equipment provided. Perfect for beginners and experienced anglers.',
                'price' => 5000,
                'capacity' => 4,
                'units' => 5,
                'location' => 'phewa-lake',
                'amenities' => ['Fishing Rods', 'Bait', 'Guide', 'Life Jackets', 'Refreshments'],
                'images' => $this->getUnsplashImages('fishing trip lake angler', 10),
            ],
            [
                'type' => 'fishing-trip',
                'title' => 'Full Day Fishing Adventure',
                'description' => 'Complete day fishing experience with lunch included. Explore the best fishing spots with expert guides.',
                'price' => 8500,
                'capacity' => 6,
                'units' => 3,
                'location' => 'phewa-lake',
                'amenities' => ['All Equipment', 'Lunch', 'Expert Guide', 'Boat', 'Photos'],
                'images' => $this->getUnsplashImages('fishing adventure river boat', 12),
            ],
            [
                'type' => 'fishing-trip',
                'title' => 'Sunrise Fishing Special',
                'description' => 'Early morning fishing trip to catch the best fish during sunrise. Includes breakfast and equipment.',
                'price' => 4000,
                'capacity' => 3,
                'units' => 4,
                'location' => 'phewa-lake',
                'amenities' => ['Equipment', 'Breakfast', 'Guide', 'Sunrise View', 'Coffee/Tea'],
                'images' => $this->getUnsplashImages('sunrise fishing morning lake', 9),
            ],
        ];

        $listings = [];
        foreach ($listingsData as $index => $data) {
            $listing = Listing::create([
                'rental_type_id' => $rentalTypes[$data['type']]->id,
                'location_id' => $locations[$data['location']]->id,
                'user_id' => $host->id,
                'title' => $data['title'],
                'description' => $data['description'],
                'price_per_day' => $data['price'],
                'capacity' => $data['capacity'],
                'available_units' => $data['units'],
                'amenities' => $data['amenities'],
                'images' => $data['images'],
                'is_available' => true,
                'is_featured' => $index < 5, // First 5 are featured
                'buffer_hours' => 4,
            ]);

            $listings[] = $listing;
            echo "  ✓ {$data['title']} ({$data['type']}) - " . count($data['images']) . " images\n";
        }

        return $listings;
    }

    private function getUnsplashImages(string $query, int $count): array
    {
        $images = [];
        $queries = explode(' ', $query);

        for ($i = 0; $i < $count; $i++) {
            // Use Unsplash Source API with different queries for variety
            $queryTerm = $queries[array_rand($queries)];
            $randomSeed = rand(1000, 9999);

            // Mix of different dimensions for variety
            $dimensions = [
                '1920x1080',
                '1600x900',
                '1280x720',
                '1440x900',
            ];
            $dimension = $dimensions[array_rand($dimensions)];

            $images[] = "https://source.unsplash.com/{$dimension}/?{$queryTerm}&sig={$randomSeed}";
        }

        return $images;
    }

    private function createBookings($users, $listings): void
    {
        $statuses = ['pending', 'confirmed', 'completed', 'cancelled'];
        $statusWeights = [0.1, 0.5, 0.3, 0.1]; // More confirmed bookings

        for ($i = 0; $i < 40; $i++) {
            $listing = $listings[array_rand($listings)];

            // Random date in the past 3 months or future 2 months
            $daysOffset = rand(-90, 60);
            $checkIn = now()->addDays($daysOffset);

            // Random duration 1-7 days
            $duration = rand(1, 7);
            $checkOut = $checkIn->copy()->addDays($duration);

            // Select status based on weights
            $rand = mt_rand() / mt_getrandmax();
            $cumulative = 0;
            $status = 'confirmed';
            foreach ($statusWeights as $idx => $weight) {
                $cumulative += $weight;
                if ($rand <= $cumulative) {
                    $status = $statuses[$idx];
                    break;
                }
            }

            // Calculate pricing
            $subtotal = $listing->price_per_day * $duration;
            $discount = rand(0, 10) > 7 ? rand(5, 15) : 0; // 30% chance of discount
            $discountAmount = $subtotal * ($discount / 100);
            $platformFee = ($subtotal - $discountAmount) * 0.05; // 5% platform fee
            $totalPrice = $subtotal - $discountAmount + $platformFee;

            $booking = Booking::create([
                'listing_id' => $listing->id,
                'user_id' => $users['user']->id,
                'check_in' => $checkIn,
                'check_out' => $checkOut,
                'guests' => rand(1, min(4, $listing->capacity)),
                'total_price' => $totalPrice,
                'discount_amount' => $discountAmount,
                'discount_percentage' => $discount,
                'platform_fee' => $platformFee,
                'status' => $status,
            ]);

            echo "  ✓ Booking #{$booking->invoice_number} - {$listing->title} ({$status})\n";
        }
    }
}
