<?php

namespace Database\Seeders;

use Illuminate\Database\Console\Seeds\WithoutModelEvents;
use Illuminate\Database\Seeder;
use App\Models\Booking;
use App\Models\Listing;
use App\Models\User;
use Carbon\Carbon;

class HourlyBookingSeeder extends Seeder
{
    /**
     * Run the database seeds.
     *
     * Creates sample hourly bookings to demonstrate:
     * - Same-day double bookings separated by hours
     * - Buffer time between bookings
     * - Color-coded time dropdowns (red=booked, yellow=buffer, green=available)
     */
    public function run(): void
    {
        // Get first available listing (preferably a car rental)
        $listing = Listing::where('is_available', true)->first();

        if (!$listing) {
            $this->command->error('No available listings found. Please create a listing first.');
            return;
        }

        // Get first available user
        $user = User::first();

        if (!$user) {
            $this->command->error('No users found. Please create a user first.');
            return;
        }

        $this->command->info("Creating hourly bookings for listing: {$listing->title} (ID: {$listing->id})");
        $this->command->info("Buffer hours for this listing: {$listing->buffer_hours} hours");

        // Clear existing bookings for this listing to start fresh
        Booking::where('listing_id', $listing->id)
            ->whereIn('status', ['pending', 'confirmed'])
            ->delete();

        $this->command->info('Cleared existing bookings for clean demonstration');

        // Scenario 1: Same-day double booking (Oct 29)
        // Booking 1: 8:00 AM - 12:00 PM (4 hours)
        $checkIn1 = Carbon::parse('2025-10-29 08:00:00');
        $checkOut1 = Carbon::parse('2025-10-29 12:00:00');

        Booking::create([
            'user_id' => $user->id,
            'listing_id' => $listing->id,
            'check_in' => $checkIn1,
            'check_out' => $checkOut1,
            'guests' => 2,
            'total_price' => 100.00,
            'status' => 'confirmed',
            'discount_amount' => 0,
            'discount_percentage' => 0,
            'platform_fee' => 5.00,
        ]);

        $this->command->info("✓ Created Booking 1: Oct 29, 8:00 AM - 12:00 PM");
        $this->command->info("  Buffer period: 12:00 PM - " . $checkOut1->copy()->addHours($listing->buffer_hours)->format('g:i A'));

        // Booking 2: 3:00 PM - 8:00 PM (5 hours) - Same day after buffer
        $checkIn2 = Carbon::parse('2025-10-29 15:00:00');
        $checkOut2 = Carbon::parse('2025-10-29 20:00:00');

        Booking::create([
            'user_id' => $user->id,
            'listing_id' => $listing->id,
            'check_in' => $checkIn2,
            'check_out' => $checkOut2,
            'guests' => 3,
            'total_price' => 125.00,
            'status' => 'confirmed',
            'discount_amount' => 0,
            'discount_percentage' => 0,
            'platform_fee' => 6.25,
        ]);

        $this->command->info("✓ Created Booking 2: Oct 29, 3:00 PM - 8:00 PM (Same day!)");
        $this->command->info("  Buffer period: 8:00 PM - " . $checkOut2->copy()->addHours($listing->buffer_hours)->format('g:i A'));

        // Scenario 2: Multi-day booking (Oct 30-31)
        $checkIn3 = Carbon::parse('2025-10-30 10:00:00');
        $checkOut3 = Carbon::parse('2025-10-31 18:00:00');

        Booking::create([
            'user_id' => $user->id,
            'listing_id' => $listing->id,
            'check_in' => $checkIn3,
            'check_out' => $checkOut3,
            'guests' => 4,
            'total_price' => 350.00,
            'status' => 'confirmed',
            'discount_amount' => 25.00,
            'discount_percentage' => 10,
            'platform_fee' => 17.50,
        ]);

        $this->command->info("✓ Created Booking 3: Oct 30, 10:00 AM - Oct 31, 6:00 PM (Multi-day)");
        $this->command->info("  Buffer period: Oct 31, 6:00 PM - " . $checkOut3->copy()->addHours($listing->buffer_hours)->format('M d, g:i A'));

        // Scenario 3: Same day as multi-day ends (Nov 1) - Can book after buffer
        $checkIn4 = Carbon::parse('2025-11-01 22:00:00');
        $checkOut4 = Carbon::parse('2025-11-02 08:00:00');

        Booking::create([
            'user_id' => $user->id,
            'listing_id' => $listing->id,
            'check_in' => $checkIn4,
            'check_out' => $checkOut4,
            'guests' => 2,
            'total_price' => 150.00,
            'status' => 'pending',
            'discount_amount' => 0,
            'discount_percentage' => 0,
            'platform_fee' => 7.50,
        ]);

        $this->command->info("✓ Created Booking 4: Nov 1, 10:00 PM - Nov 2, 8:00 AM (After buffer from Booking 3)");

        // Scenario 4: Short hourly bookings on Nov 3
        // Morning booking: 6:00 AM - 9:00 AM
        $checkIn5 = Carbon::parse('2025-11-03 06:00:00');
        $checkOut5 = Carbon::parse('2025-11-03 09:00:00');

        Booking::create([
            'user_id' => $user->id,
            'listing_id' => $listing->id,
            'check_in' => $checkIn5,
            'check_out' => $checkOut5,
            'guests' => 1,
            'total_price' => 75.00,
            'status' => 'confirmed',
            'discount_amount' => 0,
            'discount_percentage' => 0,
            'platform_fee' => 3.75,
        ]);

        $this->command->info("✓ Created Booking 5: Nov 3, 6:00 AM - 9:00 AM (Morning slot)");

        // Afternoon booking: 1:00 PM - 5:00 PM
        $checkIn6 = Carbon::parse('2025-11-03 13:00:00');
        $checkOut6 = Carbon::parse('2025-11-03 17:00:00');

        Booking::create([
            'user_id' => $user->id,
            'listing_id' => $listing->id,
            'check_in' => $checkIn6,
            'check_out' => $checkOut6,
            'guests' => 2,
            'total_price' => 100.00,
            'status' => 'confirmed',
            'discount_amount' => 0,
            'discount_percentage' => 0,
            'platform_fee' => 5.00,
        ]);

        $this->command->info("✓ Created Booking 6: Nov 3, 1:00 PM - 5:00 PM (Afternoon slot)");

        // Evening booking: 9:00 PM - 11:00 PM
        $checkIn7 = Carbon::parse('2025-11-03 21:00:00');
        $checkOut7 = Carbon::parse('2025-11-03 23:00:00');

        Booking::create([
            'user_id' => $user->id,
            'listing_id' => $listing->id,
            'check_in' => $checkIn7,
            'check_out' => $checkOut7,
            'guests' => 3,
            'total_price' => 80.00,
            'status' => 'confirmed',
            'discount_amount' => 0,
            'discount_percentage' => 0,
            'platform_fee' => 4.00,
        ]);

        $this->command->info("✓ Created Booking 7: Nov 3, 9:00 PM - 11:00 PM (Evening slot)");

        $this->command->info('');
        $this->command->info('========================================');
        $this->command->info('Sample Hourly Bookings Created Successfully!');
        $this->command->info('========================================');
        $this->command->info('');
        $this->command->info('What to expect in the UI:');
        $this->command->info('');
        $this->command->info('📅 Oct 29 (Same-day double booking):');
        $this->command->info('  🔴 8:00 AM - 12:00 PM: Booked (Booking 1)');
        $this->command->info('  🟡 12:00 PM - ' . $checkOut1->copy()->addHours($listing->buffer_hours)->format('g:i A') . ': Buffer');
        $this->command->info('  🟢 After buffer: Available');
        $this->command->info('  🔴 3:00 PM - 8:00 PM: Booked (Booking 2)');
        $this->command->info('  🟡 8:00 PM - ' . $checkOut2->copy()->addHours($listing->buffer_hours)->format('g:i A') . ': Buffer');
        $this->command->info('  🟢 After buffer: Available');
        $this->command->info('');
        $this->command->info('📅 Oct 30-31 (Multi-day booking):');
        $this->command->info('  🔴 All hours: Booked (Booking 3)');
        $this->command->info('');
        $this->command->info('📅 Nov 1:');
        $this->command->info('  🟡 12:00 AM - ' . $checkOut3->copy()->addHours($listing->buffer_hours)->format('g:i A') . ': Buffer');
        $this->command->info('  🟢 After buffer: Available');
        $this->command->info('  🔴 10:00 PM - 11:59 PM: Booked (Booking 4)');
        $this->command->info('');
        $this->command->info('📅 Nov 3 (Triple booking - multiple short slots):');
        $this->command->info('  🔴 6:00 AM - 9:00 AM: Booked (Booking 5)');
        $this->command->info('  🟡 9:00 AM - ' . $checkOut5->copy()->addHours($listing->buffer_hours)->format('g:i A') . ': Buffer');
        $this->command->info('  🟢 Available slots between bookings');
        $this->command->info('  🔴 1:00 PM - 5:00 PM: Booked (Booking 6)');
        $this->command->info('  🟡 5:00 PM - ' . $checkOut6->copy()->addHours($listing->buffer_hours)->format('g:i A') . ': Buffer');
        $this->command->info('  🟢 Available slots between bookings');
        $this->command->info('  🔴 9:00 PM - 11:00 PM: Booked (Booking 7)');
        $this->command->info('  🟡 11:00 PM - ' . $checkOut7->copy()->addHours($listing->buffer_hours)->format('g:i A') . ': Buffer');
        $this->command->info('');
        $this->command->info("Test by visiting: /listings/{$listing->id}");
        $this->command->info('Then select dates: Oct 29, Oct 30-31, Nov 1, or Nov 3 to see the color-coded hour dropdowns!');
    }
}
