<?php

namespace Database\Seeders;

use App\Models\User;
use App\Models\Listing;
use App\Models\Booking;
use Illuminate\Database\Seeder;
use Carbon\Carbon;

class FutureBookingsSeeder extends Seeder
{
    /**
     * Run the database seeds.
     * Creates future bookings to demonstrate buffer time and availability calendar
     */
    public function run(): void
    {
        $this->command->info('Creating future bookings to demonstrate buffer periods...');

        // Get some listings to create bookings for
        $listings = Listing::where('is_available', true)->limit(10)->get();

        if ($listings->isEmpty()) {
            $this->command->error('No available listings found.');
            return;
        }

        // Get users with 'user' role for bookings
        $users = User::where('role', 'user')->get();

        if ($users->isEmpty()) {
            $this->command->error('No users found.');
            return;
        }

        $bookingsCreated = 0;
        $today = Carbon::today();

        // Create bookings for different scenarios
        $bookingScenarios = [
            // Scenario 1: Booking starting tomorrow
            [
                'start_offset_days' => 1,
                'duration_days' => 3,
                'status' => 'confirmed',
                'description' => 'Booking starting tomorrow (3 days)'
            ],
            // Scenario 2: Booking next week
            [
                'start_offset_days' => 7,
                'duration_days' => 2,
                'status' => 'confirmed',
                'description' => 'Booking next week (2 days)'
            ],
            // Scenario 3: Booking in 2 weeks
            [
                'start_offset_days' => 14,
                'duration_days' => 4,
                'status' => 'confirmed',
                'description' => 'Booking in 2 weeks (4 days)'
            ],
            // Scenario 4: Pending booking (also blocks dates)
            [
                'start_offset_days' => 21,
                'duration_days' => 2,
                'status' => 'pending',
                'description' => 'Pending booking in 3 weeks (2 days)'
            ],
            // Scenario 5: Another booking next month
            [
                'start_offset_days' => 30,
                'duration_days' => 5,
                'status' => 'confirmed',
                'description' => 'Booking next month (5 days)'
            ],
        ];

        foreach ($listings as $listing) {
            $this->command->info("Creating bookings for listing: {$listing->title}");

            // Create 2-3 bookings per listing
            $numBookings = rand(2, 3);
            $selectedScenarios = array_rand($bookingScenarios, min($numBookings, count($bookingScenarios)));

            if (!is_array($selectedScenarios)) {
                $selectedScenarios = [$selectedScenarios];
            }

            foreach ($selectedScenarios as $scenarioIndex) {
                $scenario = $bookingScenarios[$scenarioIndex];
                $user = $users->random();

                // Calculate dates
                $checkIn = $today->copy()->addDays($scenario['start_offset_days'])->setTime(10, 0, 0);
                $checkOut = $checkIn->copy()->addDays($scenario['duration_days'])->setTime(10, 0, 0);

                // Calculate price
                $days = $scenario['duration_days'];
                $totalPrice = $listing->price_per_day * $days;

                // Create booking
                Booking::create([
                    'listing_id' => $listing->id,
                    'user_id' => $user->id,
                    'check_in' => $checkIn,
                    'check_out' => $checkOut,
                    'guests' => rand(1, min(4, $listing->capacity)),
                    'total_price' => $totalPrice,
                    'status' => $scenario['status'],
                    'discount_amount' => 0,
                    'discount_percentage' => 0,
                    'platform_fee' => 0,
                ]);

                $bookingsCreated++;
                $this->command->info("  ✓ {$scenario['description']} - Check-in: {$checkIn->format('Y-m-d')}");
            }
        }

        $this->command->info("Successfully created {$bookingsCreated} future bookings!");

        // Show summary with buffer info
        $this->command->info("\n=== Buffer Period Summary ===");
        foreach ($listings->take(3) as $listing) {
            $bufferHours = $listing->buffer_hours ?? 0;
            $bookingCount = $listing->bookings()->whereIn('status', ['confirmed', 'pending'])->count();

            $this->command->info("{$listing->title}:");
            $this->command->info("  - Buffer hours: {$bufferHours}h");
            $this->command->info("  - Active bookings: {$bookingCount}");

            if ($bufferHours > 0) {
                $this->command->info("  - Note: {$bufferHours} hours will be blocked after each checkout");
            }
        }
    }
}
