<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // Note: Some indexes might already exist, so we wrap them in try-catch to avoid errors

        // Bookings table indexes - most critical for performance
        $this->tryAddIndexes('bookings', [
            'status',
            'user_id',
            'listing_id',
            ['check_in', 'check_out'],
            'created_at',
            ['listing_id', 'status'],
            ['user_id', 'status'],
        ]);

        // Listings table indexes
        $this->tryAddIndexes('listings', [
            'is_available',
            'is_featured',
            'rental_type_id',
            'location_id',
            'user_id',
            ['is_available', 'is_featured'],
            ['location_id', 'is_available'],
            ['rental_type_id', 'is_available'],
        ]);

        // Users table indexes
        $this->tryAddIndexes('users', [
            'role',
            'email_verified_at',
            'vacation_mode',
            ['role', 'vacation_mode'],
        ]);

        // Payments table indexes
        $this->tryAddIndexes('payments', [
            'booking_id',
            'status',
            'transaction_id',
            'created_at',
        ]);

        // Pickup locations indexes
        $this->tryAddIndexes('pickup_locations', [
            'is_active',
            'type',
            'created_by',
            ['is_active', 'type'],
        ]);

        // Discount tiers indexes
        $this->tryAddIndexes('discount_tiers', [
            ['listing_id', 'min_days'],
        ]);

        // Add-ons indexes
        $this->tryAddIndexes('add_ons', [
            'is_active',
        ]);

        // Reviews indexes (if you have them)
        if (Schema::hasTable('reviews')) {
            $this->tryAddIndexes('reviews', [
                'listing_id',
                'user_id',
                'created_at',
            ]);
        }
    }

    /**
     * Helper function to add indexes with error handling
     */
    private function tryAddIndexes(string $tableName, array $indexes): void
    {
        foreach ($indexes as $index) {
            try {
                Schema::table($tableName, function (Blueprint $table) use ($index) {
                    $table->index($index);
                });
            } catch (\Exception $e) {
                // Index likely already exists, continue
                continue;
            }
        }
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::table('bookings', function (Blueprint $table) {
            $table->dropIndex(['status']);
            $table->dropIndex(['user_id']);
            $table->dropIndex(['listing_id']);
            $table->dropIndex(['check_in', 'check_out']);
            $table->dropIndex(['created_at']);
            $table->dropIndex(['listing_id', 'status']);
            $table->dropIndex(['user_id', 'status']);
        });

        Schema::table('listings', function (Blueprint $table) {
            $table->dropIndex(['is_available']);
            $table->dropIndex(['is_featured']);
            $table->dropIndex(['rental_type_id']);
            $table->dropIndex(['location_id']);
            $table->dropIndex(['user_id']);
            $table->dropIndex(['is_available', 'is_featured']);
            $table->dropIndex(['location_id', 'is_available']);
            $table->dropIndex(['rental_type_id', 'is_available']);
        });

        Schema::table('users', function (Blueprint $table) {
            $table->dropIndex(['role']);
            $table->dropIndex(['email_verified_at']);
            $table->dropIndex(['vacation_mode']);
            $table->dropIndex(['role', 'vacation_mode']);
        });

        Schema::table('payments', function (Blueprint $table) {
            $table->dropIndex(['booking_id']);
            $table->dropIndex(['status']);
            $table->dropIndex(['transaction_id']);
            $table->dropIndex(['created_at']);
        });

        Schema::table('pickup_locations', function (Blueprint $table) {
            $table->dropIndex(['is_active']);
            $table->dropIndex(['type']);
            $table->dropIndex(['created_by']);
            $table->dropIndex(['is_active', 'type']);
        });

        Schema::table('discount_tiers', function (Blueprint $table) {
            $table->dropIndex(['listing_id']);
            $table->dropIndex(['listing_id', 'min_days']);
        });

        Schema::table('add_ons', function (Blueprint $table) {
            $table->dropIndex(['is_active']);
        });

        if (Schema::hasTable('reviews')) {
            Schema::table('reviews', function (Blueprint $table) {
                $table->dropIndex(['listing_id']);
                $table->dropIndex(['user_id']);
                $table->dropIndex(['created_at']);
            });
        }
    }
};
