<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // Create pickup_locations table for managing pickup/dropoff points
        Schema::create('pickup_locations', function (Blueprint $table) {
            $table->id();
            $table->string('name'); // e.g., "Kuala Lumpur International Airport"
            $table->string('address')->nullable(); // Full address
            $table->string('type')->default('general'); // airport, hotel, public_spot, office
            $table->decimal('latitude', 10, 7)->nullable(); // For map display
            $table->decimal('longitude', 10, 7)->nullable();
            $table->boolean('is_active')->default(true);
            $table->foreignId('created_by')->nullable()->constrained('users')->nullOnDelete(); // Admin/Host who created
            $table->timestamps();
        });

        // Pivot table: which listings support which pickup locations
        Schema::create('listing_pickup_locations', function (Blueprint $table) {
            $table->id();
            $table->foreignId('listing_id')->constrained()->cascadeOnDelete();
            $table->foreignId('pickup_location_id')->constrained()->cascadeOnDelete();
            $table->timestamps();

            $table->unique(['listing_id', 'pickup_location_id']);
        });

        // Add buffer_hours to listings (turnover/preparation time)
        Schema::table('listings', function (Blueprint $table) {
            $table->integer('buffer_hours')->default(0)->after('price_per_day')
                ->comment('Hours needed between bookings for cleaning/preparation');
        });

        // Add pickup/dropoff to bookings
        Schema::table('bookings', function (Blueprint $table) {
            $table->foreignId('pickup_location_id')->nullable()->after('listing_id')
                ->constrained('pickup_locations')->nullOnDelete();
            $table->foreignId('dropoff_location_id')->nullable()->after('pickup_location_id')
                ->constrained('pickup_locations')->nullOnDelete();
            $table->time('pickup_time')->nullable()->after('dropoff_location_id');
            $table->time('dropoff_time')->nullable()->after('pickup_time');
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::table('bookings', function (Blueprint $table) {
            $table->dropForeign(['pickup_location_id']);
            $table->dropForeign(['dropoff_location_id']);
            $table->dropColumn(['pickup_location_id', 'dropoff_location_id', 'pickup_time', 'dropoff_time']);
        });

        Schema::table('listings', function (Blueprint $table) {
            $table->dropColumn('buffer_hours');
        });

        Schema::dropIfExists('listing_pickup_locations');
        Schema::dropIfExists('pickup_locations');
    }
};
