<?php

return [
    /*
    |--------------------------------------------------------------------------
    | Performance Caching Configuration
    |--------------------------------------------------------------------------
    |
    | This file contains caching strategies for GengSewa application
    | to handle 3000+ bookings per month efficiently.
    |
    */

    'cache' => [
        /*
        |--------------------------------------------------------------------------
        | Cache TTL (Time To Live) in seconds
        |--------------------------------------------------------------------------
        */
        'ttl' => [
            // Static data - cache for 1 hour
            'locations' => env('CACHE_TTL_LOCATIONS', 3600),
            'rental_types' => env('CACHE_TTL_RENTAL_TYPES', 3600),
            'amenities' => env('CACHE_TTL_AMENITIES', 3600),
            'settings' => env('CACHE_TTL_SETTINGS', 3600),

            // Semi-static data - cache for 10 minutes
            'featured_listings' => env('CACHE_TTL_FEATURED', 600),
            'pickup_locations' => env('CACHE_TTL_PICKUP_LOCATIONS', 600),

            // Dynamic data - cache for 5 minutes
            'listings_search' => env('CACHE_TTL_SEARCH', 300),
            'available_listings' => env('CACHE_TTL_AVAILABLE', 300),

            // User-specific data - cache for 2 minutes
            'user_bookings' => env('CACHE_TTL_USER_BOOKINGS', 120),
            'host_listings' => env('CACHE_TTL_HOST_LISTINGS', 120),
        ],

        /*
        |--------------------------------------------------------------------------
        | Cache Tags (Redis only)
        |--------------------------------------------------------------------------
        */
        'tags' => [
            'locations' => ['locations', 'static'],
            'rental_types' => ['rental_types', 'static'],
            'listings' => ['listings', 'dynamic'],
            'bookings' => ['bookings', 'transactional'],
            'users' => ['users', 'authentication'],
        ],

        /*
        |--------------------------------------------------------------------------
        | Cache Keys Prefix
        |--------------------------------------------------------------------------
        */
        'prefix' => env('CACHE_PREFIX', 'gengsewa'),
    ],

    /*
    |--------------------------------------------------------------------------
    | Query Optimization
    |--------------------------------------------------------------------------
    */
    'query' => [
        // Enable query result caching
        'cache_results' => env('QUERY_CACHE_RESULTS', true),

        // Default eager loading relationships
        'eager_load' => [
            'bookings' => ['listing', 'user', 'payment', 'pickupLocation', 'dropoffLocation'],
            'listings' => ['location', 'rentalType', 'host', 'amenities'],
            'users' => ['bookings', 'listings'],
        ],

        // Pagination limits
        'pagination' => [
            'default' => 25,
            'max' => 100,
        ],
    ],

    /*
    |--------------------------------------------------------------------------
    | Session Optimization
    |--------------------------------------------------------------------------
    */
    'session' => [
        // Use database for session storage in production
        'driver' => env('SESSION_DRIVER', 'database'),

        // Session lifetime in minutes
        'lifetime' => env('SESSION_LIFETIME', 120),

        // Expire session on browser close
        'expire_on_close' => false,
    ],

    /*
    |--------------------------------------------------------------------------
    | Image Optimization
    |--------------------------------------------------------------------------
    */
    'images' => [
        // Enable lazy loading
        'lazy_load' => env('IMAGE_LAZY_LOAD', true),

        // Thumbnail sizes
        'thumbnails' => [
            'small' => [150, 150],
            'medium' => [300, 300],
            'large' => [600, 600],
        ],

        // Image quality (1-100)
        'quality' => env('IMAGE_QUALITY', 85),

        // Enable WebP conversion
        'webp' => env('IMAGE_WEBP', true),
    ],

    /*
    |--------------------------------------------------------------------------
    | Database Optimization
    |--------------------------------------------------------------------------
    */
    'database' => [
        // Enable query logging (disable in production)
        'log_queries' => env('DB_LOG_QUERIES', false),

        // Slow query threshold in milliseconds
        'slow_query_threshold' => env('DB_SLOW_QUERY_MS', 1000),

        // Connection pool size
        'pool_size' => env('DB_POOL_SIZE', 10),
    ],

    /*
    |--------------------------------------------------------------------------
    | API Rate Limiting
    |--------------------------------------------------------------------------
    */
    'rate_limit' => [
        // Requests per minute for authenticated users
        'authenticated' => env('RATE_LIMIT_AUTHENTICATED', 60),

        // Requests per minute for guests
        'guest' => env('RATE_LIMIT_GUEST', 30),

        // Webhook requests per minute
        'webhook' => env('RATE_LIMIT_WEBHOOK', 100),
    ],

    /*
    |--------------------------------------------------------------------------
    | Asset Optimization
    |--------------------------------------------------------------------------
    */
    'assets' => [
        // Enable CDN for assets
        'cdn_enabled' => env('CDN_ENABLED', false),
        'cdn_url' => env('CDN_URL', ''),

        // Asset versioning
        'version' => env('ASSET_VERSION', '1.0.0'),

        // Minification
        'minify_html' => env('MINIFY_HTML', true),
    ],
];
