#!/bin/bash

# GengSewa Production Cleanup Script
# Removes unnecessary files and optimizes for production deployment

echo "======================================"
echo "GengSewa Production Cleanup Script"
echo "======================================"
echo ""

# Function to confirm action
confirm() {
    read -p "$1 (y/n): " response
    case "$response" in
        [yY][eE][sS]|[yY])
            return 0
            ;;
        *)
            return 1
            ;;
    esac
}

# 1. Remove development dependencies
echo "[1/8] Removing development dependencies..."
if confirm "Remove composer development dependencies?"; then
    composer install --no-dev --optimize-autoloader
    echo "✓ Development dependencies removed"
fi

# 2. Remove documentation files (keep essential ones)
echo ""
echo "[2/8] Cleaning up documentation files..."
if confirm "Remove old/redundant documentation files?"; then
    rm -f CLEANUP_SUMMARY.md
    rm -f OPTIMIZATION_SUMMARY.md
    rm -f PICKUP_DROPOFF_FEATURE.md
    rm -f PRODUCTION_DEPLOYMENT.md
    rm -f QUICK_PRODUCTION_GUIDE.md
    rm -f HTML_DESCRIPTION_GUIDE.md
    echo "✓ Old documentation removed (kept: README.md, SETUP.md, PRODUCTION_OPTIMIZATION_COMPLETE.md)"
fi

# 3. Remove testing files
echo ""
echo "[3/8] Removing test files..."
if confirm "Remove PHPUnit and test files?"; then
    rm -rf tests/
    rm -f phpunit.xml
    echo "✓ Test files removed"
fi

# 4. Clear old logs
echo ""
echo "[4/8] Clearing old log files..."
if confirm "Remove log files older than 30 days?"; then
    find storage/logs -name "*.log" -type f -mtime +30 -delete 2>/dev/null || echo "No old logs found"
    echo "✓ Old logs cleared"
fi

# 5. Remove node_modules (after npm run build)
echo ""
echo "[5/8] Handling node_modules..."
if confirm "Remove node_modules? (Make sure npm run build was executed first!)"; then
    rm -rf node_modules
    rm -f package-lock.json
    echo "✓ node_modules removed"
fi

# 6. Remove unused TinyMCE plugins (if applicable)
echo ""
echo "[6/8] Cleaning up TinyMCE..."
if [ -d "public/vendor/tinymce" ]; then
    if confirm "Remove unused TinyMCE plugins to save space?"; then
        # Keep only essential plugins
        cd public/vendor/tinymce/plugins
        ls | grep -v -E "(lists|link|image|table|code|preview)" | xargs rm -rf
        cd ../../../../
        echo "✓ Unused TinyMCE plugins removed"
    fi
fi

# 7. Optimize autoloader and clear caches
echo ""
echo "[7/8] Optimizing Laravel..."
echo "Clearing all caches..."
php artisan optimize:clear

echo "Building optimized caches..."
php artisan config:cache
php artisan route:cache
php artisan view:cache
php artisan event:cache

echo "Optimizing composer autoloader..."
composer dump-autoload --optimize --classmap-authoritative

echo "✓ Laravel optimized for production"

# 8. Set proper permissions
echo ""
echo "[8/8] Setting permissions..."
if confirm "Set proper file permissions?"; then
    chmod -R 755 storage bootstrap/cache
    chown -R www-data:www-data storage bootstrap/cache 2>/dev/null || echo "Note: Run as root to change ownership"
    echo "✓ Permissions set"
fi

# Final summary
echo ""
echo "======================================"
echo "Cleanup Complete!"
echo "======================================"
echo ""
echo "Production-ready optimizations applied:"
echo "  ✓ Development dependencies removed"
echo "  ✓ Documentation cleaned up"
echo "  ✓ Test files removed"
echo "  ✓ Old logs cleared"
echo "  ✓ Laravel caches optimized"
echo "  ✓ Permissions configured"
echo ""
echo "Next steps:"
echo "1. Run: npm run build (if not done yet)"
echo "2. Test the application thoroughly"
echo "3. Deploy to production server"
echo "4. Monitor logs: tail -f storage/logs/laravel.log"
echo ""
echo "For complete optimization guide, see:"
echo "  PRODUCTION_OPTIMIZATION_COMPLETE.md"
echo ""
