<?php

namespace App\Services;

use App\Models\Setting;

class PlatformFeeService
{
    /**
     * Calculate platform fee for a booking amount
     */
    public static function calculateFee(float $bookingAmount): float
    {
        $feeType = Setting::get('platform_fee_type', 'percentage');
        $fixedFee = floatval(Setting::get('platform_fee_fixed', 0));
        $percentageFee = floatval(Setting::get('platform_fee_percentage', 0));

        $fee = 0;

        if ($feeType === 'fixed') {
            $fee = $fixedFee;
        } elseif ($feeType === 'percentage') {
            $fee = ($bookingAmount * $percentageFee) / 100;
        } else { // both
            $fee = $fixedFee + (($bookingAmount * $percentageFee) / 100);
        }

        return round($fee, 2);
    }

    /**
     * Get who the fee is charged to (guest or host)
     */
    public static function getChargedTo(): string
    {
        return Setting::get('platform_fee_charged_to', 'guest');
    }

    /**
     * Calculate total amount guest needs to pay
     */
    public static function calculateGuestTotal(float $bookingAmount): float
    {
        $fee = self::calculateFee($bookingAmount);

        if (self::getChargedTo() === 'guest') {
            return round($bookingAmount + $fee, 2);
        }

        return round($bookingAmount, 2);
    }

    /**
     * Calculate total amount host receives
     */
    public static function calculateHostEarnings(float $bookingAmount): float
    {
        $fee = self::calculateFee($bookingAmount);

        if (self::getChargedTo() === 'host') {
            return round($bookingAmount - $fee, 2);
        }

        return round($bookingAmount, 2);
    }

    /**
     * Get fee breakdown for display
     */
    public static function getFeeBreakdown(float $bookingAmount): array
    {
        $fee = self::calculateFee($bookingAmount);
        $chargedTo = self::getChargedTo();

        return [
            'booking_amount' => round($bookingAmount, 2),
            'platform_fee' => $fee,
            'guest_total' => self::calculateGuestTotal($bookingAmount),
            'host_earnings' => self::calculateHostEarnings($bookingAmount),
            'charged_to' => $chargedTo,
        ];
    }
}
