<?php

namespace App\Services;

use App\Models\OutgoingWebhookConfig;
use App\Models\Booking;
use App\Models\Payment;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class OutgoingWebhookService
{
    /**
     * Send payment success webhook to external URL
     */
    public function sendPaymentSuccess(Payment $payment): bool
    {
        $config = OutgoingWebhookConfig::getConfig();

        // Check if webhook is enabled and payment_success event is enabled
        if (!$config->enabled || !$config->payment_success || !$config->webhook_url) {
            return false;
        }

        // Load all necessary relationships
        $payment->load([
            'booking.user',
            'booking.listing.host',
            'booking.listing.location',
            'booking.listing.rentalType',
            'booking.addOns'
        ]);

        $booking = $payment->booking;
        $user = $booking->user;
        $listing = $booking->listing;
        $host = $listing->host;

        // Prepare webhook payload with all relevant data
        $payload = [
            'event' => 'payment.success',
            'timestamp' => now()->toIso8601String(),

            // Payment details
            'payment' => [
                'id' => $payment->id,
                'transaction_id' => $payment->transaction_id,
                'payment_intent_id' => $payment->payment_intent_id,
                'amount' => (float) $payment->amount,
                'currency' => $payment->currency,
                'payment_channel' => $payment->payment_channel,
                'status' => $payment->status,
            ],

            // Booking/Order details
            'booking' => [
                'id' => $booking->id,
                'invoice_number' => $booking->invoice_number,
                'order_number' => 'BOOK-' . $booking->id,
                'check_in' => $booking->check_in->format('Y-m-d'),
                'check_out' => $booking->check_out->format('Y-m-d'),
                'guests' => $booking->guests,
                'total_price' => (float) $booking->total_price,
                'status' => $booking->status,
                'discount_amount' => (float) $booking->discount_amount ?? 0,
                'platform_fee' => (float) $booking->platform_fee ?? 0,
                'days' => $booking->check_in->diffInDays($booking->check_out),
            ],

            // Customer/User details
            'customer' => [
                'id' => $user->id,
                'name' => $user->name,
                'email' => $user->email,
                'phone' => $user->phone,
            ],

            // Host listing details
            'listing' => [
                'id' => $listing->id,
                'title' => $listing->title,
                'price_per_day' => (float) $listing->price_per_day,
                'location' => $listing->location ? $listing->location->name : null,
                'rental_type' => $listing->rentalType ? $listing->rentalType->name : null,
            ],

            // Host details
            'host' => [
                'id' => $host->id,
                'name' => $host->name,
                'email' => $host->email,
                'phone' => $host->phone,
            ],

            // Add-ons if any
            'add_ons' => $booking->addOns->map(function ($addOn) {
                return [
                    'id' => $addOn->id,
                    'name' => $addOn->name,
                    'quantity' => $addOn->pivot->quantity,
                    'price_paid' => (float) $addOn->pivot->price_paid,
                ];
            })->toArray(),
        ];

        return $this->sendWebhook($config->webhook_url, $payload, $config->secret_key);
    }

    /**
     * Send payment failed webhook to external URL
     */
    public function sendPaymentFailed(Payment $payment): bool
    {
        $config = OutgoingWebhookConfig::getConfig();

        if (!$config->enabled || !$config->payment_failed || !$config->webhook_url) {
            return false;
        }

        $payment->load(['booking.user', 'booking.listing']);

        $payload = [
            'event' => 'payment.failed',
            'timestamp' => now()->toIso8601String(),
            'payment' => [
                'id' => $payment->id,
                'transaction_id' => $payment->transaction_id,
                'amount' => (float) $payment->amount,
                'status' => $payment->status,
            ],
            'booking' => [
                'id' => $payment->booking->id,
                'invoice_number' => $payment->booking->invoice_number,
                'order_number' => 'BOOK-' . $payment->booking->id,
            ],
            'customer' => [
                'id' => $payment->booking->user->id,
                'name' => $payment->booking->user->name,
                'email' => $payment->booking->user->email,
            ],
        ];

        return $this->sendWebhook($config->webhook_url, $payload, $config->secret_key);
    }

    /**
     * Send actual HTTP request to webhook URL
     */
    protected function sendWebhook(string $url, array $payload, ?string $secretKey = null): bool
    {
        try {
            Log::info('Sending Outgoing Webhook', [
                'url' => $url,
                'event' => $payload['event'] ?? 'unknown',
            ]);

            $headers = [
                'Content-Type' => 'application/json',
                'User-Agent' => 'GengSewa-Webhook/1.0',
            ];

            // Add signature if secret key is configured
            if ($secretKey) {
                $signature = hash_hmac('sha256', json_encode($payload), $secretKey);
                $headers['X-Webhook-Signature'] = $signature;
            }

            // Configure HTTP client
            $http = Http::timeout(30)->withHeaders($headers);

            // Disable SSL verification in local environment
            // For production, ensure proper SSL certificates are configured
            if (config('app.env') === 'local') {
                $http = $http->withOptions(['verify' => false]);
            }

            $response = $http->post($url, $payload);

            if ($response->successful()) {
                Log::info('Outgoing Webhook Sent Successfully', [
                    'url' => $url,
                    'status' => $response->status(),
                ]);
                return true;
            } else {
                Log::warning('Outgoing Webhook Failed', [
                    'url' => $url,
                    'status' => $response->status(),
                    'response' => $response->body(),
                ]);
                return false;
            }
        } catch (\Exception $e) {
            Log::error('Outgoing Webhook Exception', [
                'url' => $url,
                'error' => $e->getMessage(),
            ]);
            return false;
        }
    }
}
