<?php

namespace App\Services;

use App\Models\EmailTemplate;
use App\Mail\TemplateMail;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;

class EmailService
{
    /**
     * Send welcome email to new user
     */
    public function sendWelcomeEmail($user)
    {
        $template = EmailTemplate::getTemplate('welcome');

        if (!$template) {
            Log::warning('Welcome email template not found or disabled');
            return false;
        }

        $data = [
            'user_name' => $user->name,
        ];

        return $this->sendEmail($user->email, $template, $data);
    }

    /**
     * Send email verification
     */
    public function sendVerificationEmail($user, $verificationUrl)
    {
        $template = EmailTemplate::getTemplate('verify_email');

        if (!$template) {
            Log::warning('Email verification template not found or disabled');
            return false;
        }

        $data = [
            'user_name' => $user->name,
            'verification_url' => $verificationUrl,
        ];

        return $this->sendEmail($user->email, $template, $data);
    }

    /**
     * Send password reset email
     */
    public function sendPasswordResetEmail($user, $resetUrl)
    {
        $template = EmailTemplate::getTemplate('reset_password');

        if (!$template) {
            Log::warning('Password reset email template not found or disabled');
            return false;
        }

        $data = [
            'user_name' => $user->name,
            'reset_url' => $resetUrl,
        ];

        return $this->sendEmail($user->email, $template, $data);
    }

    /**
     * Send booking success email
     */
    public function sendBookingSuccessEmail($booking)
    {
        $template = EmailTemplate::getTemplate('booking_success');

        if (!$template) {
            Log::warning('Booking success email template not found or disabled');
            return false;
        }

        // Load relationships
        $booking->load(['user', 'listing.host', 'listing.location']);

        $data = [
            'customer_name' => $booking->user->name,
            'invoice_number' => $booking->invoice_number,
            'listing_title' => $booking->listing->title,
            'check_in' => $booking->check_in->format('M d, Y'),
            'check_out' => $booking->check_out->format('M d, Y'),
            'guests' => $booking->guests,
            'total_price' => number_format($booking->total_price, 2),
            'host_name' => $booking->listing->host->name,
            'host_email' => $booking->listing->host->email,
            'host_phone' => $booking->listing->host->phone ?? 'N/A',
            'receipt_url' => route('booking.receipt', $booking->id),
        ];

        return $this->sendEmail($booking->user->email, $template, $data);
    }

    /**
     * Send booking success email to host
     */
    public function sendBookingNotificationToHost($booking)
    {
        // Load relationships
        $booking->load(['user', 'listing.host']);

        $subject = 'New Booking Received - ' . $booking->invoice_number;

        $body = $this->getHostNotificationTemplate([
            'host_name' => $booking->listing->host->name,
            'customer_name' => $booking->user->name,
            'listing_title' => $booking->listing->title,
            'check_in' => $booking->check_in->format('M d, Y'),
            'check_out' => $booking->check_out->format('M d, Y'),
            'guests' => $booking->guests,
            'total_price' => number_format($booking->total_price, 2),
            'customer_email' => $booking->user->email,
            'customer_phone' => $booking->user->phone ?? 'N/A',
        ]);

        try {
            Mail::send([], [], function ($message) use ($booking, $subject, $body) {
                $message->to($booking->listing->host->email)
                    ->subject($subject)
                    ->html($body);
            });

            Log::info('Booking notification email sent to host', [
                'booking_id' => $booking->id,
                'host_email' => $booking->listing->host->email,
            ]);

            return true;
        } catch (\Exception $e) {
            Log::error('Failed to send booking notification to host', [
                'booking_id' => $booking->id,
                'error' => $e->getMessage(),
            ]);
            return false;
        }
    }

    /**
     * Send email using template
     */
    protected function sendEmail($to, EmailTemplate $template, array $data)
    {
        try {
            $rendered = $template->render($data);

            Mail::send([], [], function ($message) use ($to, $rendered) {
                $message->to($to)
                    ->subject($rendered['subject'])
                    ->html($rendered['body']);
            });

            Log::info('Email sent successfully', [
                'template' => $template->name,
                'to' => $to,
            ]);

            return true;
        } catch (\Exception $e) {
            Log::error('Failed to send email', [
                'template' => $template->name,
                'to' => $to,
                'error' => $e->getMessage(),
            ]);
            return false;
        }
    }

    /**
     * Get host notification template
     */
    protected function getHostNotificationTemplate($data)
    {
        $defaults = [
            'site_name' => \App\Models\Setting::get('site_name', config('app.name')),
            'site_url' => config('app.url'),
            'logo_url' => \App\Helpers\SeoHelper::siteLogo() ? url(\App\Helpers\SeoHelper::siteLogo()) : url('/images/logo.png'),
            'year' => date('Y'),
        ];

        $data = array_merge($defaults, $data);

        $body = '<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
</head>
<body style="margin: 0; padding: 0; font-family: Arial, sans-serif; background-color: #f4f4f4;">
    <table width="100%" cellpadding="0" cellspacing="0" style="background-color: #f4f4f4; padding: 20px;">
        <tr>
            <td align="center">
                <table width="600" cellpadding="0" cellspacing="0" style="background-color: #ffffff; border-radius: 8px; overflow: hidden;">
                    <tr>
                        <td style="background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%); padding: 40px 20px; text-align: center;">
                            <img src="' . $data['logo_url'] . '" alt="' . $data['site_name'] . '" style="max-width: 150px; height: auto;">
                            <h1 style="color: #ffffff; margin: 20px 0 0 0; font-size: 28px;">New Booking Received!</h1>
                        </td>
                    </tr>
                    <tr>
                        <td style="padding: 40px 30px;">
                            <h2 style="color: #333333; margin: 0 0 20px 0;">Hi ' . $data['host_name'] . '!</h2>
                            <p style="color: #666666; line-height: 1.6; margin: 0 0 20px 0;">
                                Great news! You have received a new booking for your listing.
                            </p>

                            <table width="100%" cellpadding="10" cellspacing="0" style="border: 1px solid #eeeeee; border-radius: 5px; margin: 20px 0;">
                                <tr>
                                    <td style="background-color: #f9f9f9; padding: 15px; border-bottom: 1px solid #eeeeee;">
                                        <strong style="color: #333333;">Booking Details</strong>
                                    </td>
                                </tr>
                                <tr>
                                    <td style="padding: 15px;">
                                        <table width="100%" cellpadding="5" cellspacing="0">
                                            <tr>
                                                <td style="color: #666666; width: 40%;">Listing:</td>
                                                <td style="color: #333333; font-weight: bold;">' . $data['listing_title'] . '</td>
                                            </tr>
                                            <tr>
                                                <td style="color: #666666;">Guest Name:</td>
                                                <td style="color: #333333; font-weight: bold;">' . $data['customer_name'] . '</td>
                                            </tr>
                                            <tr>
                                                <td style="color: #666666;">Check-in:</td>
                                                <td style="color: #333333;">' . $data['check_in'] . '</td>
                                            </tr>
                                            <tr>
                                                <td style="color: #666666;">Check-out:</td>
                                                <td style="color: #333333;">' . $data['check_out'] . '</td>
                                            </tr>
                                            <tr>
                                                <td style="color: #666666;">Guests:</td>
                                                <td style="color: #333333;">' . $data['guests'] . '</td>
                                            </tr>
                                            <tr>
                                                <td style="color: #666666;">Total Amount:</td>
                                                <td style="color: #3b82f6; font-weight: bold; font-size: 18px;">RM ' . $data['total_price'] . '</td>
                                            </tr>
                                        </table>
                                    </td>
                                </tr>
                            </table>

                            <p style="color: #666666; line-height: 1.6; margin: 20px 0;">
                                <strong>Guest Contact Information:</strong><br>
                                Email: ' . $data['customer_email'] . '<br>
                                Phone: ' . $data['customer_phone'] . '
                            </p>

                            <p style="color: #666666; line-height: 1.6; margin: 20px 0;">
                                Please prepare your listing and contact the guest if needed.
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <td style="background-color: #f9f9f9; padding: 30px; text-align: center; border-top: 1px solid #eeeeee;">
                            <p style="color: #999999; font-size: 14px; margin: 0 0 10px 0;">
                                Thank you for hosting with ' . $data['site_name'] . '!
                            </p>
                            <p style="color: #999999; font-size: 12px; margin: 0;">
                                © ' . $data['year'] . ' ' . $data['site_name'] . '. All rights reserved.
                            </p>
                        </td>
                    </tr>
                </table>
            </td>
        </tr>
    </table>
</body>
</html>';

        return $body;
    }
}
