<?php

namespace App\Services;

use Webimpian\BayarcashSdk\Bayarcash;
use App\Models\Payment;
use App\Models\Booking;

class BayarcashService
{
    protected $bayarcash;

    public function __construct()
    {
        // Initialize Bayarcash with Personal Access Token (PAT)
        $this->bayarcash = new Bayarcash(
            config('bayarcash.pat')
        );

        // Use production mode (no sandbox)
        // Production is the default, so we don't need to call useSandbox()
    }

    public function createPaymentIntent(Booking $booking, string $paymentChannel = 'fpx')
    {
        // Load user relationship if not already loaded
        $booking->load('user');

        // Set API version to v3 for webhook support
        $this->bayarcash->setApiVersion('v3');

        // Map payment channel string to Bayarcash constants
        $channelMap = [
            'fpx' => Bayarcash::FPX,
            // 'card' => Bayarcash::CREDIT_CARD, // Commented out for future use
            'tng' => Bayarcash::DUITNOW_QR,
            // 'boost' => Bayarcash::BOOST_PAYFLEX, // Commented out for future use
        ];

        $data = [
            'portal_key' => config('bayarcash.portal_key'),
            'amount' => $booking->total_price,
            'order_number' => 'BOOK-' . $booking->id,
            'payer_email' => $booking->user->email,
            'payer_name' => $booking->user->name,
            'payer_telephone_number' => $booking->user->phone ?? '+60123456789', // Default if not set
            'payment_channel' => $channelMap[$paymentChannel] ?? Bayarcash::FPX,
            'return_url' => route('payment.success'),
            'callback_url' => route('payment.callback'), // v3: POST webhook for payment notifications
        ];

        // Generate checksum if secret key is configured
        if (config('bayarcash.secret_key')) {
            $data['checksum'] = $this->bayarcash->createPaymentIntenChecksumValue(
                config('bayarcash.secret_key'),
                $data
            );
        }

        $paymentIntent = $this->bayarcash->createPaymentIntent($data);

        // Convert object to array to safely access properties
        $paymentArray = json_decode(json_encode($paymentIntent), true);

        // Get payment intent ID from the response
        $paymentIntentId = $paymentArray['id']
            ?? $paymentArray['order_number']
            ?? $paymentArray['orderNumber']
            ?? 'BOOK-' . $booking->id;

        // Create payment record
        $payment = Payment::create([
            'booking_id' => $booking->id,
            'payment_intent_id' => $paymentIntentId,
            'amount' => $booking->total_price,
            'currency' => 'MYR',
            'payment_channel' => $paymentChannel,
            'status' => 'pending',
            'bayarcash_response' => json_encode($paymentArray),
        ]);

        return [
            'payment' => $payment,
            'payment_url' => $paymentIntent->url ?? $paymentArray['url'],
        ];
    }

    public function verifyPayment($paymentIntentId)
    {
        $this->bayarcash->setApiVersion('v3');
        return $this->bayarcash->getPaymentIntent($paymentIntentId);
    }

    public function handleCallback($payload)
    {
        // Log the callback payload for debugging
        \Log::info('BayarCash Callback Received', $payload);

        // Verify callback signature if secret key is configured and checksum is provided
        if (config('bayarcash.secret_key') && isset($payload['checksum'])) {
            $isValid = $this->bayarcash->verifyTransactionCallbackData(
                $payload,
                config('bayarcash.secret_key')
            );

            if (!$isValid) {
                \Log::error('BayarCash Callback Verification Failed', $payload);
                return false;
            }
        }

        // v3 callback uses order_number to match the payment
        $orderNumber = $payload['order_number'] ?? null;

        if (!$orderNumber) {
            \Log::error('BayarCash Callback Missing Order Number', $payload);
            return false;
        }

        // Find payment by order number pattern: BOOK-{booking_id}
        $payment = Payment::whereHas('booking', function($query) use ($orderNumber) {
            $bookingId = str_replace('BOOK-', '', $orderNumber);
            $query->where('id', $bookingId);
        })->first();

        if (!$payment) {
            \Log::error('BayarCash Callback Payment Not Found', ['order_number' => $orderNumber]);
            return false;
        }

        // Map v3 status codes to payment status
        // 0: New, 1: Pending, 2: Failed, 3: Success, 4: Cancelled
        $statusMap = [
            0 => 'pending',    // New
            1 => 'pending',    // Pending
            2 => 'failed',     // Failed
            3 => 'paid',       // Success
            4 => 'cancelled',  // Cancelled
        ];

        $status = $statusMap[$payload['status'] ?? 1] ?? 'pending';

        // Update payment with full callback data
        $payment->update([
            'status' => $status,
            'transaction_id' => $payload['transaction_id'] ?? null,
            'bayarcash_response' => json_encode($payload),
        ]);

        // Update booking status based on payment status
        if ($status === 'paid') {
            $payment->booking->update(['status' => 'confirmed']);

            \Log::info('Booking Confirmed', [
                'booking_id' => $payment->booking->id,
                'order_number' => $orderNumber
            ]);

            // Send outgoing webhook for payment success
            try {
                $webhookService = app(\App\Services\OutgoingWebhookService::class);
                $webhookService->sendPaymentSuccess($payment);
            } catch (\Exception $e) {
                \Log::error('Failed to send outgoing webhook', [
                    'payment_id' => $payment->id,
                    'error' => $e->getMessage()
                ]);
            }

            // Send confirmation email
            try {
                \Mail::to($payment->booking->user->email)
                    ->send(new \App\Mail\BookingConfirmation($payment->booking));
            } catch (\Exception $e) {
                \Log::error('Failed to send booking confirmation email', [
                    'booking_id' => $payment->booking->id,
                    'error' => $e->getMessage()
                ]);
            }
        } elseif (in_array($status, ['failed', 'cancelled'])) {
            $payment->booking->update(['status' => 'cancelled']);

            \Log::info('Booking Cancelled', [
                'booking_id' => $payment->booking->id,
                'order_number' => $orderNumber,
                'reason' => $status
            ]);

            // Send outgoing webhook for payment failed
            try {
                $webhookService = app(\App\Services\OutgoingWebhookService::class);
                $webhookService->sendPaymentFailed($payment);
            } catch (\Exception $e) {
                \Log::error('Failed to send outgoing webhook for payment failed', [
                    'payment_id' => $payment->id,
                    'error' => $e->getMessage()
                ]);
            }
        }

        return true;
    }
}
