<?php

namespace App\Services;

use App\Models\User;
use App\Models\HostBadge;
use App\Models\BadgeConfig;
use App\Models\Review;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class BadgeService
{
    /**
     * Update user's badge stats and check for new badges
     */
    public function updateUserBadges(User $user): void
    {
        // Update cached stats
        $this->updateUserStats($user);

        // Check and award badges
        $this->checkAndAwardBadges($user);
    }

    /**
     * Update user's cached review stats
     */
    protected function updateUserStats(User $user): void
    {
        // Get reviews for all host's listings
        $listingIds = $user->listings()->pluck('id');

        $stats = Review::whereIn('listing_id', $listingIds)
            ->select(DB::raw('COUNT(*) as total, AVG(rating) as average'))
            ->first();

        $user->update([
            'total_reviews_count' => $stats->total ?? 0,
            'average_rating' => round($stats->average ?? 0, 2),
            'last_badge_check' => now(),
        ]);
    }

    /**
     * Check if user qualifies for badges and award them
     */
    protected function checkAndAwardBadges(User $user): void
    {
        $badgeConfigs = BadgeConfig::getActiveBadges();

        foreach ($badgeConfigs as $badgeConfig) {
            $criteria = [
                'min_reviews' => $badgeConfig->min_reviews,
                'min_rating' => $badgeConfig->min_rating,
                'time_period_months' => $badgeConfig->time_period_months,
            ];

            $qualifies = $this->checkBadgeQualification($user, $criteria);

            if ($qualifies) {
                $this->awardBadge($user, $badgeConfig->badge_type);
            } else {
                $this->revokeBadge($user, $badgeConfig->badge_type);
            }
        }
    }

    /**
     * Check if user qualifies for a specific badge
     */
    protected function checkBadgeQualification(User $user, array $criteria): bool
    {
        $listingIds = $user->listings()->pluck('id');

        $cutoffDate = Carbon::now()->subMonths($criteria['time_period_months']);

        $recentReviews = Review::whereIn('listing_id', $listingIds)
            ->where('created_at', '>=', $cutoffDate)
            ->get();

        $reviewCount = $recentReviews->count();
        $avgRating = $recentReviews->avg('rating');

        return $reviewCount >= $criteria['min_reviews']
            && $avgRating >= $criteria['min_rating'];
    }

    /**
     * Award badge to user
     */
    protected function awardBadge(User $user, string $badgeType): void
    {
        HostBadge::updateOrCreate(
            [
                'user_id' => $user->id,
                'badge_type' => $badgeType,
            ],
            [
                'earned_at' => now(),
                'expires_at' => null,
            ]
        );
    }

    /**
     * Revoke badge from user
     */
    protected function revokeBadge(User $user, string $badgeType): void
    {
        HostBadge::where('user_id', $user->id)
            ->where('badge_type', $badgeType)
            ->delete();
    }

    /**
     * Get user's highest badge
     */
    public function getHighestBadge(User $user): ?HostBadge
    {
        $badgeOrder = [
            HostBadge::TYPE_SUPER_HOST,
            HostBadge::TYPE_LOCAL_LEGEND,
            HostBadge::TYPE_TOP_RATED,
            HostBadge::TYPE_RISING_STAR,
        ];

        foreach ($badgeOrder as $badgeType) {
            $badge = HostBadge::where('user_id', $user->id)
                ->where('badge_type', $badgeType)
                ->first();

            if ($badge && $badge->isActive()) {
                return $badge;
            }
        }

        return null;
    }
}
