<?php

namespace App\Providers;

use Illuminate\Support\ServiceProvider;
use App\Models\Setting;

class MailConfigServiceProvider extends ServiceProvider
{
    /**
     * Register services.
     */
    public function register(): void
    {
        //
    }

    /**
     * Bootstrap services.
     */
    public function boot(): void
    {
        try {
            // Load mail settings from database
            $mailMailer = Setting::get('mail_mailer', config('mail.default'));

            if ($mailMailer === 'log') {
                // Use log driver - emails will be written to storage/logs/laravel.log
                config(['mail.default' => 'log']);
            } elseif ($mailMailer === 'ses') {
                $awsKey = Setting::get('aws_access_key_id');
                $awsSecret = Setting::get('aws_secret_access_key');

                // Only use SES if credentials are properly configured
                if (!empty($awsKey) && !empty($awsSecret)) {
                    config(['mail.default' => 'ses']);
                    config(['services.ses.key' => $awsKey]);
                    config(['services.ses.secret' => $awsSecret]);
                    config(['services.ses.region' => Setting::get('aws_default_region', 'us-east-1')]);
                } else {
                    // Fall back to log if SES credentials are missing
                    config(['mail.default' => 'log']);
                    \Log::warning('AWS SES credentials not configured, falling back to log mailer');
                }
            } elseif ($mailMailer === 'smtp') {
                $mailHost = Setting::get('mail_host', config('mail.mailers.smtp.host'));

                // Only update SMTP config if host is configured
                if (!empty($mailHost)) {
                    config(['mail.default' => 'smtp']);
                    config(['mail.mailers.smtp.host' => $mailHost]);
                    config(['mail.mailers.smtp.port' => Setting::get('mail_port', config('mail.mailers.smtp.port'))]);
                    config(['mail.mailers.smtp.username' => Setting::get('mail_username')]);
                    config(['mail.mailers.smtp.password' => Setting::get('mail_password')]);
                    config(['mail.mailers.smtp.encryption' => Setting::get('mail_encryption', 'tls')]);
                } else {
                    // Fall back to log if SMTP host is not configured
                    config(['mail.default' => 'log']);
                    \Log::warning('SMTP host not configured, falling back to log mailer');
                }
            }

            // Update from address and name if configured
            $fromAddress = Setting::get('mail_from_address', config('mail.from.address'));
            $fromName = Setting::get('mail_from_name', config('mail.from.name'));

            if (!empty($fromAddress)) {
                config(['mail.from.address' => $fromAddress]);
            }
            if (!empty($fromName)) {
                config(['mail.from.name' => $fromName]);
            }
        } catch (\Exception $e) {
            // If database is not ready or settings table doesn't exist, use default config
            \Log::info('Mail config from database not loaded: ' . $e->getMessage());
        }
    }
}
