<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;

class Listing extends Model
{
    protected $fillable = [
        'rental_type_id',
        'location_id',
        'user_id',
        'title',
        'description',
        'price_per_day',
        'buffer_hours',
        'capacity',
        'available_units',
        'images',
        'amenities',
        'is_available',
        'is_featured',
    ];

    protected $casts = [
        'images' => 'array',
        'amenities' => 'array',
        'is_available' => 'boolean',
        'is_featured' => 'boolean',
        'price_per_day' => 'decimal:2',
    ];

    public function rentalType(): BelongsTo
    {
        return $this->belongsTo(RentalType::class);
    }

    public function location(): BelongsTo
    {
        return $this->belongsTo(Location::class);
    }

    public function host(): BelongsTo
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    public function bookings(): HasMany
    {
        return $this->hasMany(Booking::class);
    }

    public function reviews(): HasMany
    {
        return $this->hasMany(Review::class);
    }

    public function discountTiers(): HasMany
    {
        return $this->hasMany(DiscountTier::class)->orderBy('min_days');
    }

    public function addOns(): BelongsToMany
    {
        return $this->belongsToMany(AddOn::class, 'listing_add_on')
            ->withPivot('custom_price')
            ->withTimestamps();
    }

    public function pickupLocations(): BelongsToMany
    {
        return $this->belongsToMany(PickupLocation::class, 'listing_pickup_locations')
            ->withTimestamps();
    }

    public function scopeAvailable($query)
    {
        return $query->where('is_available', true);
    }

    public function scopeFeatured($query)
    {
        return $query->where('is_featured', true);
    }

    // Get average rating
    public function getAverageRatingAttribute()
    {
        return $this->reviews()->avg('rating') ?? 0;
    }

    // Get total reviews count
    public function getReviewsCountAttribute()
    {
        return $this->reviews()->count();
    }

    /**
     * Get applicable discount for given number of days
     */
    public function getApplicableDiscount(int $days): ?DiscountTier
    {
        return $this->discountTiers()
            ->get()
            ->first(function ($tier) use ($days) {
                return $tier->appliesTo($days);
            });
    }

    /**
     * Calculate price with discount for given days
     */
    public function calculatePriceWithDiscount(int $days): array
    {
        $subtotal = $this->price_per_day * $days;
        $discount = $this->getApplicableDiscount($days);
        $discountAmount = 0;

        if ($discount) {
            $discountAmount = $discount->calculateDiscount($subtotal, $days);
        }

        return [
            'subtotal' => $subtotal,
            'discount' => $discountAmount,
            'total' => $subtotal - $discountAmount,
            'discount_tier' => $discount,
        ];
    }

    /**
     * Check if listing is available for given dates
     * Takes into account buffer_hours for turnover/cleaning time
     */
    public function isAvailableForDates($checkIn, $checkOut): bool
    {
        if (!$this->is_available) {
            return false;
        }

        // Get confirmed bookings that might conflict
        $conflictingBookings = $this->bookings()
            ->whereIn('status', ['confirmed', 'pending'])
            ->get()
            ->filter(function ($booking) use ($checkIn, $checkOut) {
                return $booking->overlapsWithDates($checkIn, $checkOut, $this->buffer_hours);
            });

        return $conflictingBookings->isEmpty();
    }

    /**
     * Get unavailable dates for calendar display
     */
    public function getUnavailableDates(): array
    {
        $unavailableDates = [];

        $this->bookings()
            ->whereIn('status', ['confirmed', 'pending'])
            ->get()
            ->each(function ($booking) use (&$unavailableDates) {
                $current = $booking->check_in->copy();
                $end = $booking->check_out->copy()->addHours($this->buffer_hours ?? 0);

                while ($current <= $end) {
                    $unavailableDates[] = $current->format('Y-m-d');
                    $current->addDay();
                }
            });

        return array_unique($unavailableDates);
    }

    /**
     * Get detailed booking calendar data with booked and buffer dates
     * Returns array with 'booked', 'buffer', and 'unavailable' date arrays
     */
    public function getBookingCalendarData(): array
    {
        $bookedDates = [];
        $bufferDates = [];
        $unavailableDates = [];
        $bufferPeriods = []; // Store buffer start/end times
        $bookedPeriods = []; // Store actual booked start/end times

        $this->bookings()
            ->whereIn('status', ['confirmed', 'pending'])
            ->get()
            ->each(function ($booking) use (&$bookedDates, &$bufferDates, &$unavailableDates, &$bufferPeriods, &$bookedPeriods) {
                // Add booked dates (check-in to check-out)
                $current = $booking->check_in->copy();
                $checkOutEnd = $booking->check_out->copy();

                while ($current < $checkOutEnd) {
                    $dateStr = $current->format('Y-m-d');
                    $bookedDates[] = $dateStr;
                    $unavailableDates[] = $dateStr;

                    // Store booked period info for hour filtering
                    if (!isset($bookedPeriods[$dateStr])) {
                        $bookedPeriods[$dateStr] = [];
                    }
                    $bookedPeriods[$dateStr][] = [
                        'start' => $booking->check_in->format('Y-m-d H:i:s'),
                        'end' => $booking->check_out->format('Y-m-d H:i:s'),
                    ];

                    $current->addDay();
                }

                // Add buffer period dates (after check-out)
                if ($this->buffer_hours > 0) {
                    $bufferStart = $booking->check_out->copy();
                    $bufferEnd = $booking->check_out->copy()->addHours($this->buffer_hours);

                    $current = $bufferStart->copy();
                    while ($current < $bufferEnd) {
                        $dateStr = $current->format('Y-m-d');
                        // Only add to buffer if not already in booked dates
                        if (!in_array($dateStr, $bookedDates)) {
                            $bufferDates[] = $dateStr;
                            // Don't add to unavailable - allow selection but restrict hours

                            // Store buffer period info for hour filtering
                            if (!isset($bufferPeriods[$dateStr])) {
                                $bufferPeriods[$dateStr] = [];
                            }
                            $bufferPeriods[$dateStr][] = [
                                'start' => $bufferStart->format('Y-m-d H:i:s'),
                                'end' => $bufferEnd->format('Y-m-d H:i:s'),
                            ];
                        }
                        $current->addDay();
                    }
                }
            });

        return [
            'booked' => array_unique($bookedDates),
            'buffer' => array_unique($bufferDates),
            'unavailable' => array_unique($unavailableDates),
            'buffer_periods' => $bufferPeriods, // Detailed buffer time info
            'booked_periods' => $bookedPeriods, // Detailed booked time info
        ];
    }
}
