<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class HostBadge extends Model
{
    protected $fillable = [
        'user_id',
        'badge_type',
        'earned_at',
        'expires_at',
    ];

    protected $casts = [
        'earned_at' => 'datetime',
        'expires_at' => 'datetime',
    ];

    // Badge types
    const TYPE_RISING_STAR = 'rising_star';
    const TYPE_TOP_RATED = 'top_rated';
    const TYPE_LOCAL_LEGEND = 'local_legend';
    const TYPE_SUPER_HOST = 'super_host';

    // Badge configuration
    public static function getBadgeConfig(): array
    {
        return [
            self::TYPE_RISING_STAR => [
                'name' => 'Rising Star',
                'icon' => '⭐',
                'color' => 'bronze',
                'min_reviews' => 10,
                'min_rating' => 4.0,
                'time_period_months' => 3,
            ],
            self::TYPE_TOP_RATED => [
                'name' => 'Top Rated',
                'icon' => '🌟',
                'color' => 'silver',
                'min_reviews' => 20,
                'min_rating' => 4.3,
                'time_period_months' => 4,
            ],
            self::TYPE_LOCAL_LEGEND => [
                'name' => 'Local Legend',
                'icon' => '🏆',
                'color' => 'gold',
                'min_reviews' => 30,
                'min_rating' => 4.5,
                'time_period_months' => 6,
            ],
            self::TYPE_SUPER_HOST => [
                'name' => 'Super Host',
                'icon' => '👑',
                'color' => 'platinum',
                'min_reviews' => 50,
                'min_rating' => 4.8,
                'time_period_months' => 12,
            ],
        ];
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function getDisplayNameAttribute(): string
    {
        $config = self::getBadgeConfig();
        return $config[$this->badge_type]['name'] ?? $this->badge_type;
    }

    public function getIconAttribute(): string
    {
        $config = self::getBadgeConfig();
        return $config[$this->badge_type]['icon'] ?? '🏅';
    }

    public function getColorAttribute(): string
    {
        $config = self::getBadgeConfig();
        return $config[$this->badge_type]['color'] ?? 'gray';
    }

    public function isActive(): bool
    {
        return $this->expires_at === null || $this->expires_at->isFuture();
    }
}
