<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;

class AddOn extends Model
{
    protected $fillable = [
        'name',
        'description',
        'price',
        'pricing_type',
        'is_active',
    ];

    protected $casts = [
        'price' => 'decimal:2',
        'is_active' => 'boolean',
    ];

    public function listings(): BelongsToMany
    {
        return $this->belongsToMany(Listing::class, 'listing_add_on')
            ->withPivot('custom_price')
            ->withTimestamps();
    }

    public function bookings(): BelongsToMany
    {
        return $this->belongsToMany(Booking::class, 'booking_add_on')
            ->withPivot('quantity', 'price_paid')
            ->withTimestamps();
    }

    /**
     * Calculate total price for add-on based on pricing type
     */
    public function calculatePrice(int $days, int $guests, ?float $customPrice = null): float
    {
        $price = $customPrice ?? $this->price;

        return match($this->pricing_type) {
            'per_booking' => $price,
            'per_day' => $price * $days,
            'per_person' => $price * $guests,
            default => $price,
        };
    }

    /**
     * Get pricing type label
     */
    public function getPricingTypeLabel(): string
    {
        return match($this->pricing_type) {
            'per_booking' => 'One-time fee',
            'per_day' => 'Per day',
            'per_person' => 'Per person',
            default => $this->pricing_type,
        };
    }
}
