<?php

namespace App\Livewire\User;

use Livewire\Component;
use App\Models\Booking;
use App\Models\Review;
use Livewire\WithPagination;

class Bookings extends Component
{
    use WithPagination;

    public $statusFilter = '';
    public $showReviewModal = false;
    public $reviewBookingId;
    public $rating = 5;
    public $comment = '';

    public function openReviewModal($bookingId)
    {
        $this->reviewBookingId = $bookingId;
        $this->rating = 5;
        $this->comment = '';
        $this->showReviewModal = true;
    }

    public function closeReviewModal()
    {
        $this->showReviewModal = false;
        $this->reviewBookingId = null;
        $this->rating = 5;
        $this->comment = '';
    }

    public function submitReview()
    {
        $this->validate([
            'rating' => 'required|integer|min:1|max:5',
            'comment' => 'required|string|min:10|max:1000',
        ]);

        $booking = Booking::findOrFail($this->reviewBookingId);

        // Check if user owns this booking
        if ($booking->user_id !== auth()->id()) {
            session()->flash('error', 'Unauthorized action.');
            return;
        }

        // Check if booking is completed
        if ($booking->status !== 'completed') {
            session()->flash('error', 'You can only review completed bookings.');
            return;
        }

        // Check if review already exists
        if ($booking->review) {
            session()->flash('error', 'You have already reviewed this booking.');
            return;
        }

        // Create review
        Review::create([
            'listing_id' => $booking->listing_id,
            'user_id' => auth()->id(),
            'booking_id' => $booking->id,
            'rating' => $this->rating,
            'comment' => $this->comment,
        ]);

        session()->flash('success', 'Thank you for your review!');
        $this->closeReviewModal();
    }

    public function render()
    {
        $query = Booking::with(['listing.location', 'listing.rentalType', 'review', 'pickupLocation', 'dropoffLocation'])
            ->where('user_id', auth()->id())
            ->orderBy('created_at', 'desc');

        if ($this->statusFilter) {
            $query->where('status', $this->statusFilter);
        }

        return view('livewire.user.bookings', [
            'bookings' => $query->paginate(10),
        ])->layout('layouts.app');
    }
}
